/* DCTC - a Direct Connect text clone for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * sema_master.h: Copyright (C) Eric Prevoteau <www@a2pb.gotdns.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: sema_master.h,v 1.4 2004/01/09 17:27:41 ericprev Exp $
*/
#ifndef __SEMA_MASTER_H__
#define __SEMA_MASTER_H__

#ifndef SEMVMX
/* this value comes from linux/sem.h */
/* but I don't know how to get it from the include without compilation errors */
#define SEMVMX 32767
#endif

#include "sema_common.h"

/******************************************/
/* number of semaphores in the SEMA array */
/***********************************************************/
/* sema 0: to choose which client is the clock master      */
/* sema 1: remaining part of sema 2 for the current second */
/* sema 2: number of 512bytes blocs which can be sent by   */
/*         clients per second.                             */
/* sema 3: remaining part of sema 4 for the current second */
/* sema 4: number of 512bytes blocs which can be downloaded*/
/*         clients per second.                             */
/* sema 5: remaining part of sema 6 for the current second */
/* sema 6: number of 8KBytes blocs which can be read per   */
/*         second when clients gather GDL parts.           */
/* sema 7: is a semaphore used to locked the 2 following   */
/*         ones. It controls global DL slots allocation.   */
/* sema 8: First, it is not used as semaphore but as a     */
/*         1 short int shared value. This value is the     */
/*         number of existing upload slots.                */
/* sema 9: First, it is not used as semaphore but as a     */
/*         1 short int shared value. This value is the     */
/*         number of busy upload slots.                    */
/***********************************************************/
#define SEMA_ARRAY_LEN (1+2+2+2+3)

typedef enum
{
	MASTER_SEMA=0,
	UL_SEMA=1,
	UL_SPD_SEMA=2,
	DL_SEMA=3,
	DL_SPD_SEMA=4,
	GATHER_SEMA=5,
	GATHER_SPD_SEMA=6,
	UL_SLOT_SEMA=7,
	UL_SLOT_TTL_SEMA=8,
	UL_SLOT_BUSY_SEMA=9,
} SPD_SEMA;

/******************************/
/* initialize semaphore array */
/*************************************************************************************/
/* input: keyfile : if not exists, it is created and the semaphore key is put inside */
/*                  if exists but the semaphore key inside is invalid, same as above */
/*                  if exists and contains a valid key, nothing is done              */
/*        spd_limit is the default speed limit (in number of 512bytes slice)         */
/*************************************************************************************/
/* output: 0=ok, !=0=error                                              */
/*         on success, *cur_semid is the semaphore id to use with semop */
/************************************************************************/
int do_sema_init(char *keyfile, int *cur_semid, int spd_limit, int dl_spd_limit, int gath_spd_limit, int ttl_up_slot);

/******************************************************************************************/
/* to avoid forever hanging of download, we must regularly check if a clock master exists */
/******************************************************************************************/
/* output: 1= master created, 0= master not created */
/****************************************************/
int check_sema_master(int semid);

/************************/
/* get 1 512Bytes slice */
/*******************************************/
/* the function ends when it has the slice */
/*******************************************/
void get_slice(int semid, SPD_SEMA semnum);

/************************/
/* get nb 8KBytes slice */
/*******************************************/
/* the function ends when it has the slice */
/*******************************************/
void get_gather_slices(int semid,int nb);

/* ------------------------ Upload slot control ------------------------ */
/******************************/
/* lock the UL slot controler */
/******************************/
void lock_ul_slot_controler(int semid);

/*********************************/
/* release the UL slot controler */
/*********************************/
void release_ul_slot_controler(int semid);

/*********************************************************/
/* get upload slot values (total ul slot and #busy slots */
/*********************************************************/
/* WARNING: this function locks the UL controler itself */
/********************************************************/
void get_ul_slot_values(int semid,int *ttl_slot, int *busy_slot);

/*******************************************************************/
/* count the number of free slots. This function performs NO LOCK  */
/* and is only design to return the number of free slots at a time */
/* If the number of free slot is negative, it is rounded to 0.     */
/*******************************************************************/
int number_of_free_slot(int semid);

/*********************************/
/* set the number of upload slot */
/*********************************/
void set_number_of_ul_slot(int semid, unsigned int number_of_slot);

/*********************************/
/* set the number of upload slot */
/*********************************/
int get_number_of_ul_slot(int semid);

#endif
