/* DCTC - a Direct Connect text clone for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * db.c: Copyright (C) Eric Prevoteau <www@a2pb.gotdns.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: mydb.c,v 1.14 2004/01/03 16:22:16 ericprev Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <unistd.h>
#include <string.h>
#include <dirent.h>
#include <netdb.h>
#include <netinet/in.h>
#include <errno.h>
#include <ctype.h>
#include <glib.h>

#include "mydb.h"
#include "display.h"
#include "var.h"
#include "he3.h"
#include "md.h"
#include "macro.h"
#include "sema.h"
#include "md_crc.h"
#include "md_db.h"

/* directory containing virtual shares */
char *vshare_directory=NULL;

/******************************/
/* list of shared directories */
/******************************/
G_LOCK_DEFINE_STATIC(shared_dirs);
GPtrArray *shared_dirs=NULL;

/********************************************************************/
/* database containing the list of all shared files with their type */
/********************************************************************/
G_LOCK_DEFINE(shared_info);
GArray *shared_info=NULL;					/* array of DB_ENTRY */

/********************************************************************/
/* it is the same info as shared_info except                        */
/* it is in "/LS" format and already compressed using HE3 algorithm */
/* This data are sent when a remote user does a $GetDirList to      */
/* retrieve the list of files shared by this client.                */
/********************************************************************/
GByteArray *he3_ls_info=NULL;		/* protected by shared_info lock */

/************************************************************/
/* using file extension, return a value to define file type */
/*********************************************************************/
/* output: -1= error                                                 */
/*          2=audio,3=compressed,4=document,5=exe,6=picture,7=videos */
/*********************************************************************/
static int get_file_type_from_ext(char *fullname)
{
	static struct
	{
		const char *ext;
		int ftype;
	} db_ext[]=	{ 
						/* audio type */
						{"mp2",2},
						{"mp3",2},
						{"mp+",2},
						{"ogg",2},
						{"au",2},
						{"wav",2},
						{"mid",2},		/* midi files */

						{"8svx",2},	/* the following file extension comes from SOX manual */
						{"aiff",2},
						{"avr",2},
						{"cdr",2},
						{"gsm",2},
						{"hcom",2},
						{"mod",2},
						{"maud",2},
						{"ossdsp",2},
						{"sf",2},
						{"sph",2},
						{"smp",2},
						{"snd",2},
						{"txw",2},
						{"voc",2},
						{"wve",2},
						{"raw",2},

						/* compressed type */
						{"zip",3},
						{"rar",3},
						{"gz",3},
						{"bz2",3},
						{"arj",3},
						{"tar",3},		/* not compressed but it is an archive format */
						{"tgz",3},
						{"lzh",3},
						{"lha",3},
						{"ace",3},
						{"cab",3},

						/* document type */
						{"doc",4},
						{"txt",4},
						{"sub",4},
						{"nfo",4},
						{"c",4},
						{"C",4},
						{"cc",4},
						{"asm",4},
						{"s",4},
						{"pdf",4},
						{"xls",4},
						{"ps",4},
						{"eps",4},
						{"html",4},
						{"htm",4},

						/* exe type */
						{"exe",5},
						{"com",5},
						{"iso",5},
						{"bin",5},
						{"cue",5},
						{"img",5},

						/* picture type */
						{"gif",6},
						{"jpg",6},
						{"jpeg",6},
						{"bmp",6},
						{"xpm",6},
						{"pnm",6},
						{"ppm",6},
						{"pbm",6},
						{"pgm",6},
						{"pcx",6},
						{"pix",6},
						{"png",6},
						{"sgi",6},
						{"tga",6},
						{"tif",6},
						{"tiff",6},
						{"xcf",6},
						{"xwd",6},

						/* video type */
						{"mpg",7},
						{"mpe",7},
						{"m1v",7},
						{"mpeg",7},
						{"avi",7},
						{"ogm",7},
						{"divx",7},
						{"mov",7},
						{"rm",7},
						{"mkv",7},
						{"pva",7},	/* DVB card video format */

						{NULL,-1}		/* last line */
					};
	char *fname;
	char *ext;
	int i;

	/* find filename */
	fname=strrchr(fullname,'/');
	if(fname==NULL)
		fname=fullname;

	/* find extension */
	ext=strrchr(fname,'.');
	if(ext==NULL)
		return -1;						/* no extension */

	ext++;								/* skip the dot */
	if(strlen(ext)<1)
		return -1;						/* no extension */

	i=0;
	while(db_ext[i].ext!=NULL)
	{
		if(!strcasecmp(db_ext[i].ext,ext))
			return db_ext[i].ftype;
		i++;
	}

	return 4;				/* let's assume it is a document */
}

/**********************************************************************/
/* create a DB_ENTRY for the given object and insert it into database */
/**********************************************************************/
/* input: db= database where entries must be added       */
/*        sod=sizeof already shared data                 */
/*        fullname= object full path/name                */
/* output: db after adding entries.                      */
/*        *is_dir=1 if the object is a directory, else 0 */
/*         *sod have been updated                        */
/*********************************************************/
static GArray *add_db_entry(GArray *db, char *fullname, double *sod,int *is_dir, off_t *fsize)
{
	struct stat st;
	DB_ENTRY de;

	*is_dir=0;		/* default: not a directory */
	*fsize=0;

	if(stat(fullname,&st))
	{
		disp_msg(ERR_MSG,"add_db_entry","unable to view file (ignored)",fullname,NULL);
		return db;
	}

	if(S_ISDIR(st.st_mode))
	{
		/* it is a directory, we notify this and we leave */
		/* we don't add directory. I think it is not very useful */
		*is_dir=1;
		return db;
	}

	de.filetype=get_file_type_from_ext(fullname);
	if(de.filetype==-1)
	{
		disp_msg(ERR_MSG,"add_db_entry","invalid filename (ignored)",fullname,NULL);
		return db;
	}

	de.filesize=st.st_size;
	de.virtual=0;

	/* if file is big enough, we compute its MD5SUM */
	if(de.filesize>BLOCKSIZE)
	{
		/* no md5sum support -> clear the checksum */
		memset(de.md5sum,0,MD5SUMLEN);
	}

	if(fullname[0]=='/')
	{
		de.filename=strdup(fullname+1);
	}
	else
		de.filename=strdup(fullname);
	if(de.filename==NULL)
	{
		disp_msg(ERR_MSG,"add_db_entry","out of memory",NULL);
		return db;
	}

	(*sod)+=st.st_size;
	*fsize=st.st_size;
	return g_array_append_val(db,de);
}

static void add_entry_to_ls(GString **ls_format,int level,const char *name)
{
	int i=0;
	while(i<level)
	{
		(*ls_format)=g_string_append_c(*ls_format,'\t');
		i++;
	}
	(*ls_format)=g_string_append(*ls_format,name);
}

/**********************************************************************************************************************/
/* take the given path and put it into /LS format. To simplify, the string is splitted along / and level is increased */
/* each time a / is encountered                                                                                       */
/**********************************************************************************************************************/
static void add_initial_dir_to_ls(GString **ls_format,int *level, const char *path)
{
	char *cpy;
	char *t;
	char *ptr;

	cpy=strdup(path);

	if(*cpy=='/')
	{
		*cpy=255;
		t=cpy+1;
	}
	else
		t=cpy;

	t=strtok_r(t,"/",&ptr);
	while(t!=NULL)
	{
		add_entry_to_ls(ls_format,*level,t);
		(*ls_format)=g_string_append((*ls_format),"\r\n");
		
#if 0
		/* the two following piece of code produce the same result */
		/* even if the first one seems to be erroneous. In fact, *level value */
		/* is false when t becomes NULL but it is not very important because */
		/* *level is no more used in such case. */
		(*level)++;				/* new level */
		t=strtok_r(NULL,"/",&ptr);
#else
		/* patch from David Toth */
		t=strtok_r(NULL,"/",&ptr);
		if(t)
			(*level)++;				/* new level */
#endif
	}

	free(cpy);
}

/***********************************************/
/* add one virtual share entry to the database */
/***********************************************/
static GArray *add_one_vshare_entry_to_ls(GArray *db,double *sod,GString **ls_format,char *fullname, char *fsize, GString **cur_vdir_name, int *level)
{
	DB_ENTRY de;

	/* 1) create the entry in the database */
	de.filetype=get_file_type_from_ext(fullname);
	if(de.filetype==-1)
	{
		disp_msg(ERR_MSG,"add_db_entry","invalid filename (ignored)",fullname,NULL);
		return db;
	}

	de.filesize=strtoul(fsize,NULL,10);
	de.virtual=1;

	/* no md5sum support -> clear the checksum */
	memset(de.md5sum,0,MD5SUMLEN);

	if(fullname[0]=='/')
		de.filename=strdup(fullname+1);
	else
		de.filename=strdup(fullname);

	if(de.filename==NULL)
	{
		disp_msg(ERR_MSG,"add_one_vshare_entry_to_ls","out of memory",NULL);
		return db;
	}

	(*sod)+=de.filesize;
	db=g_array_append_val(db,de);

	/* 2) we must also put the filename in the share list */
	{
		GString *to_add=g_string_new(de.filename);
		char *t;

		/* adjust cur_vdir_name to keep only the part existing in to_add string */
		while((*cur_vdir_name)->len>0)
		{
			if(!strncmp((*cur_vdir_name)->str,to_add->str,(*cur_vdir_name)->len))
			{
				/* to_add is in cur_vdir_name directory */
				to_add=g_string_erase(to_add,0,(*cur_vdir_name)->len);
				/* now to_add is a cur_vdir_name relative path */
				break;
			}

			/* go one directory up */
			(*level)=(*level)-1;
			if((*level)==0)
			{
				(*cur_vdir_name)=g_string_truncate((*cur_vdir_name),0);	/* we have reached the root directory */
				break;
			}

			/* remove the trailing '/' */
			(*cur_vdir_name)=g_string_truncate((*cur_vdir_name),(*cur_vdir_name)->len-1);
			t=strrchr((*cur_vdir_name)->str,'/');
			if(t==NULL)
			{	
				/* we should never come here because this means we have reached root directory but with a level != 0 */
				break;
			}
			/* and discard the last directory name */
			(*cur_vdir_name)=g_string_truncate((*cur_vdir_name),(t+1)-((*cur_vdir_name)->str));
		}

		/* and now, we must "enter" in directories still in to_add */
		while((t=strchr(to_add->str,'/'))!=NULL)
		{
			*t='\0';

			/* "enter" in the directory */
			add_entry_to_ls(ls_format,(*level),to_add->str);
			(*ls_format)=g_string_append((*ls_format),"\r\n");
			(*level)=(*level)+1;
			(*cur_vdir_name)=g_string_append((*cur_vdir_name),to_add->str);
			(*cur_vdir_name)=g_string_append_c((*cur_vdir_name),'/');

			*t='/';
			to_add=g_string_erase(to_add,0,(t+1)-(to_add->str));		/* discard this directory level from the string */
		}

		/* we are reaching the end, we are "virtually' in the good directory */
		/* we just have to put the file inside */
		add_entry_to_ls(ls_format,(*level),to_add->str);
		g_string_sprintfa((*ls_format),"|%lu\r\n",de.filesize);

		g_string_free(to_add,TRUE);
	}

	return db;
}

/****************************************************************************/
/* read the given virtual share file and load its content into the database */
/****************************************************************************/
static GArray *add_one_vshare_directory_file_to_ls(GArray *db, char *filename, double *sod, GString **ls_format)
{
	FILE *f;
	char buf[MAXPATHLEN+8192];
	GString *cur_vdir_name=NULL;
	int level=0;

	f=fopen(filename,"rb");
	if(f==NULL)
	{
		disp_msg(ERR_MSG,"add_one_vshare_directory_file_to_ls","unable to open file (ignored)",filename,NULL);
		return db;
	}

	cur_vdir_name=g_string_new("");

	while(fgets(buf,sizeof(buf)-1,f)!=NULL)
	{	
		char *fname;
		char *fsize;
		char *t;
		/* line format is something like this */
		/* -rwxr-xr-x    1 root     root        17508 Dec 18 04:19 ./gconv/ANSI_X3.110.so */

		/* remove trailing line feed */
		t=strchr(buf,'\n');
		if(t!=NULL)
			*t='\0';

		if(strlen(buf)<17)	/* there is 9 fields separated by space (9 fields + 8 spaces = 17 characters mini) */
			continue;

		if(strncmp(buf,"-r",2))	/* line must begins with "-r" */
			continue;

		t=buf;

#define NEXT_FIELD(t) {SKIP_UNTIL_SPACE(t); if(*t=='\0') continue; SKIP_SPACE(t); if(*t=='\0') continue;}

		/* we are on the access rights, go to file size */
		NEXT_FIELD(t)
		NEXT_FIELD(t)
		NEXT_FIELD(t)
		NEXT_FIELD(t)
		fsize=t;
		NEXT_FIELD(t)
		NEXT_FIELD(t)
		NEXT_FIELD(t)
		NEXT_FIELD(t)
		fname=t;

		if(!strncmp(fname,"./",2))
			fname+=2;
		else if(fname[0]=='/')
			fname++;

		if(strlen(fname))
		{
			db=add_one_vshare_entry_to_ls(db,sod,ls_format,fname,fsize,&cur_vdir_name,&level);
		}
	}

	fclose(f);

	g_string_free(cur_vdir_name,TRUE);
	return db;
}

/***********************************************************************/
/* scan the vshare directory and create all its DB_ENTRY into database */
/***********************************************************************/
static GArray *add_vshare_directory_content_to_ls(GArray *db, double *sod, GString **ls_format,char *vshare_directory)
{
	DIR *dir;
	GString *np;
	struct dirent *obj;

	if(vshare_directory==NULL)
		return db;

	/* open directory */
	dir=opendir(vshare_directory);
	if(dir==NULL)
	{
		disp_msg(ERR_MSG,"add_vshare_directory_content_to_ls","unable to open directory (ignored)",vshare_directory,NULL);
		return db;
	}

	np=g_string_new(NULL);
	/* read the whole directory */
	while((obj=readdir(dir))!=NULL)
	{
		if(obj->d_name[0]=='.')		/* ignore everything starting with '.' */
			continue;

		/* create the full path */
		np=g_string_assign(np,vshare_directory);
		np=g_string_append_c(np,'/');
		np=g_string_append(np,obj->d_name);

		db=add_one_vshare_directory_file_to_ls(db,np->str,sod,ls_format);
	}

	g_string_free(np,TRUE);

	closedir(dir);

	return db;
}

/**********************************************************************************/
/* recursively scan the given directory and create all its DB_ENTRY into database */
/**********************************************************************************/
/* input: db= database where entries must be added */
/*        sod=sizeof already shared data           */
/*        path= directory to scan                  */
/*        ls_format= string of all filename in db  */
/*        level=number of \t to prepend before each*/
/*              filename                           */
/* output: db after adding entries.                */
/*         *sod have been updated                  */
/***************************************************/
static GArray *rebuild_dir_database(GArray *db,double *sod,char *path, GString **ls_format, int level)
{
	DIR *dir;
	GString *np;
	struct dirent *obj;

	if(path==NULL)
		return db;

	/* open directory */
	dir=opendir(path);
	if(dir==NULL)
	{
		disp_msg(ERR_MSG,"rebuild_dir_database","unable to open directory (ignored)",path,NULL);
		return db;
	}

	np=g_string_new(NULL);

	level++;			/* we are now inside the directory */

	/* read the whole directory */
	while((obj=readdir(dir))!=NULL)
	{
		int is_dir;
		off_t fsize;

		if(obj->d_name[0]=='.')		/* ignore everything starting with '.' */
			continue;

		/* create the full path */
		np=g_string_assign(np,path);
		np=g_string_append_c(np,'/');
		np=g_string_append(np,obj->d_name);

		/* build an entry for this object and add it to the database */
		db=add_db_entry(db,np->str,sod,&is_dir,&fsize);

		/* if the entry is a directory, scan it */
		if(is_dir)
		{
			add_entry_to_ls(ls_format,level,obj->d_name);
			(*ls_format)=g_string_append((*ls_format),"\r\n");
			db=rebuild_dir_database(db,sod,np->str,ls_format,level);
		}
		else if(fsize!=0)
		{
			add_entry_to_ls(ls_format,level,obj->d_name);
#ifndef __USE_FILE_OFFSET64
			g_string_sprintfa((*ls_format),"|%lu\r\n",fsize);
#else
			g_string_sprintfa((*ls_format),"|%llu\r\n",fsize);
#endif
		}
	}
	g_string_free(np,TRUE);

	closedir(dir);

	return db;
}

/***************************************************************/
/* fully rebuild shared file database                          */
/* this rebuilding can be done even if a search is in progress */
/***************************************************************/
void rebuild_database(void)
{
	GArray *old;
	GArray *new_shared=NULL;
	int i;
	double sod=0;
	GByteArray *nw_he3;
	GByteArray *old_he3;
	GString *ls_format=NULL;

	/* rebuild a new database */
	ls_format=g_string_sized_new(65536);
	ls_format=g_string_assign(ls_format,"");
	
	G_LOCK(shared_dirs);
	new_shared=g_array_new(FALSE,FALSE,sizeof(DB_ENTRY));
	if(shared_dirs!=NULL)
	{
		for(i=0;i<shared_dirs->len;i++)
		{
			int lvl=0;
			add_initial_dir_to_ls(&ls_format,&lvl,g_ptr_array_index(shared_dirs,i));
			
			new_shared=rebuild_dir_database(new_shared,&sod,g_ptr_array_index(shared_dirs,i),&ls_format,lvl);
		}
	}

	if(vshare_directory!=NULL)
	{
		new_shared=add_vshare_directory_content_to_ls(new_shared,&sod,&ls_format,vshare_directory);
	}
	G_UNLOCK(shared_dirs);

	/* compress the file list using he3 encoding */
	nw_he3=encode_he3_data(ls_format);
	g_string_free(ls_format,TRUE);

	/* replace old database by new one */
	G_LOCK(shared_info);
	/* update db values */
	old=shared_info;
	shared_info=new_shared;
	sizeof_data=sod;

	/* update HE3 ls */
	old_he3=he3_ls_info;
	he3_ls_info=nw_he3;

	G_UNLOCK(shared_info);
	new_shared=NULL;							/* to avoid erroneous free */

	/* if an old base exists, free it */
	if(old!=NULL)
	{
		for(i=0;i<old->len;i++)
		{
			char *t;

			/* free DB_ENTRY filename if exists */
			t=g_array_index(old,DB_ENTRY,i).filename;
			if(t!=NULL)
				free(t);
		}

		g_array_free(old,TRUE);
	}

	if(old_he3!=NULL)
		g_byte_array_free(old_he3,TRUE);
}

/*****************************************************/
/* add a directory to the list of shared directories */
/**********************************************************/
/* due to the fact this function runs in the main thread, */
/* it is not possible to be locked forever, even if there */
/* is lot of $Search request because they also are run    */
/* from this thread (at least at the beginning)           */
/**********************************************************/
void add_shared_directory(char *dir)
{
	struct stat st;
	char *str;
	int ln;


	if(shared_dirs==NULL)
		shared_dirs=g_ptr_array_new();

	if(stat(dir,&st))
	{
		disp_msg(ERR_MSG,"add_shared_directory","invalid dir",dir,NULL);
		return;
	}

	if(!S_ISDIR(st.st_mode))
	{
		disp_msg(ERR_MSG,"add_shared_directory","not a dir",dir,NULL);
		return;
	}

	str=strdup(dir);
	if(str==NULL)
	{
		disp_msg(ERR_MSG,"add_shared_directory","out of memory for",dir,NULL);
		return;
	}

	/* remove the trailing / of the shared directory else, the database will contain a string with 2 consecutives / */
	/* and nobody will by able to download files in ths directory */
	ln=strlen(str);
	if(ln>1)
	{
		if(str[ln-1]=='/')
			str[ln-1]='\0';
	}

	disp_msg(DEBUG_MSG,NULL,"sharing",dir,NULL);

	/* add this directory to the list and rebuild database */
	G_LOCK(shared_dirs);
	g_ptr_array_add(shared_dirs,str);
	G_UNLOCK(shared_dirs);

	rebuild_database();
}

/********************************************************/
/* remove a directory to the list of shared directories */
/**********************************************************/
/* due to the fact this function runs in the main thread, */
/* it is not possible to be locked forever, even if there */
/* is lot of $Search request because they also are run    */
/* from this thread (at least at the beginning)           */
/**********************************************************/
void remove_shared_directory(char *dir)
{
	unsigned int i;

	if(shared_dirs==NULL)
		return;

	/* remove this directory to the list and rebuild database */
	G_LOCK(shared_dirs);
	for(i=0;i<shared_dirs->len;i++)
	{
		if(!strcmp(dir, g_ptr_array_index(shared_dirs,i)) )
		{
			g_ptr_array_remove_index_fast(shared_dirs,i);
			disp_msg(DEBUG_MSG,NULL,"unsharing",dir,NULL);
			break;
		}
	}
	G_UNLOCK(shared_dirs);

	rebuild_database();
}

/***********************************/
/* set the virtual share directory */
/***********************************/
void set_vshare_directory(char *dir)
{
	if(vshare_directory!=NULL)
	{
		free(vshare_directory);
		vshare_directory=NULL;
	}

	if(dir!=NULL)
	{
		vshare_directory=strdup(dir);
	}
	rebuild_database();
}

/******************************************************************************/
/* get the list of all shared directories. Each directory is separated by a | */
/******************************************************************************/
/* output: GString containing the list of directories. Must be freed */
/*********************************************************************/
GString *get_shared_directory_list(void)
{
	GString *lst;
	int i;

	lst=g_string_new("");
	G_LOCK(shared_dirs);
	if(shared_dirs!=NULL)
	{
		for(i=0;i<shared_dirs->len;i++)
		{
			if(i!=0)
				lst=g_string_append_c(lst,'|');
			g_string_sprintfa(lst,"%s",(char*)g_ptr_array_index(shared_dirs,i));
		}
	}
	G_UNLOCK(shared_dirs);

	return lst;
}

/**************************************************************/
/* check if the given file size matching the size requirement */
/**************************************************************/
/* output: 1=ok, 0=no */
/**********************/
static int size_match(SIZE_ATTRIB size_matter, unsigned long size, unsigned long filesize)
{
	switch(size_matter)
	{
		case SA_NO_MATTER:	return 1;		/* it is ok */

		case SA_AT_LEAST:
									if(filesize>=size)
										return 1;
									return 0;

		case SA_AT_MOST:
									if(filesize<=size)
										return 1;
									return 0;
	}
	return 0;
}

/************************************************/
/* check if the given file type can be accepted */
/************************************************/
/* output: 1=ok, 0=no */
/**********************/
static int type_match(int wanted_type, int filetype)
{
	if(wanted_type==1)				/* anytype */
		return 1;						/* ok */

	if(wanted_type==filetype)
		return 1;						/* ok */
	return 0;
}

/*********************************************************************************/
/* explode the given pattern (using space as separator) into one or more strings */
/* and put them into the given decpat array (array of GString)                   */
/*********************************************************************************/
static GPtrArray *decode_pattern(GPtrArray *decpat, char *pattern)
{
	GString *npat;
	GString *nw;
	char *t;

	npat=g_string_new(pattern);
	t=strrchr(npat->str,'$');
	while(t!=NULL)
	{
		if(strlen(t)>1)		/* at least 2 bytes per search string */
		{
			nw=g_string_new(t+1);
			g_ptr_array_add(decpat,nw);
		}
		npat=g_string_truncate(npat,t-npat->str);
		t=strrchr(npat->str,'$');
	}

	/* add the remaining string to the array (if long enough) */
	if(strlen(npat->str)>1)
	{
		g_ptr_array_add(decpat,npat);
	}
	else
		g_string_free(npat,TRUE);

	return decpat;
}

/************************************************************************/
/* check if the given pattern exists inside filename (case insensitive) */
/************************************************************************/
/* output: address of the pattern or NULL */
/******************************************/
char *my_strcasestr(GString *pattern, char *filename)
{
	int max_pos;
	int i;

	max_pos=strlen(filename)-pattern->len;
	if(max_pos<0)					/* filename length < pattern length ? */
		return NULL;				/* no match */

	for(i=0;i<=max_pos;i++)
	{
		if(!strncasecmp(pattern->str,filename+i,pattern->len))
			return filename+i;
	}

	return NULL;					/* no match */
}

/***********************************************************************/
/* check if the given filename contains all pattern stored into decpat */
/***********************************************************************/
/* output: 1=ok, 0=no */
/**********************/
static int pattern_match(GPtrArray *decpat, char *filename)
{
	int i;

	for(i=0;i<decpat->len;i++)
	{
		if(my_strcasestr(g_ptr_array_index(decpat,i),filename)==NULL)
			return 0;	/* fail */
	}
	return 1;		/* ok */
}

/* convert path from unix to dos format */
static GString *convert_path(GString *adapted)
{
	int i;
	char *t;
	t=adapted->str;

	for(i=0;i<adapted->len;i++)
	{
		if(t[i]=='/')
			t[i]='\\';
	}

	return adapted;
}

/*****************************************************************************/
/* this function build a search result line and write it to the given socket */
/******************************************************************************************/
/* input: output_sck= socket to write the result line                                     */
/*        dest_nick= destination nickname (can be NULL when connected to dest user client */
/*        de= db_entry to send                                                            */
/******************************************************************************************/
static void send_a_db_result(int output_sck, char *dest_nick, DB_ENTRY *de, struct sockaddr_in *dest_addr, char *md5sum)
{
	GString *str;
	GString *adapted;
	int out;
	int ttl_dl_slot;
	int busy_slot;
	int free_dl_slot;

	str=g_string_sized_new(1024);

	adapted=g_string_new(de->filename);
	adapted=convert_path(adapted);

	get_ul_slot_values(bl_semid,&ttl_dl_slot,&busy_slot);
	free_dl_slot=ttl_dl_slot-busy_slot;

	LOCK_READ(user_info);
	if(md5sum==NULL)
	{
		g_string_sprintfa(str,"$SR %s %s\005%lu %d/%d\005%s (%s)",
											nickname,adapted->str,de->filesize,
											(dl_on? ((free_dl_slot>0)?free_dl_slot:0) :0), ttl_dl_slot,
											hubname->str,org_hubip->str);
	}
	else
	{
		char strmd5[512];
		md5tostr(md5sum,strmd5);
		g_string_sprintfa(str,"$SR %s %s\005%lu.%s %d/%d\005%s (%s)",
											nickname,adapted->str,de->filesize,strmd5,
											(dl_on? ((free_dl_slot>0)?free_dl_slot:0) :0), ttl_dl_slot,
											hubname->str,org_hubip->str);
	}

	UNLOCK_READ(user_info);
	g_string_free(adapted,TRUE);

	if(dest_nick!=NULL)
	{
		str=g_string_append_c(str,'\005');
		str=g_string_append(str,dest_nick);
		str=g_string_append_c(str,'|');
	}
	str=g_string_append_c(str,'|');

	if(dest_addr==NULL)
		out=send(output_sck,str->str,str->len,MSG_NOSIGNAL);
	else
		out=sendto(srch_sck,str->str,str->len,MSG_NOSIGNAL,(void*)dest_addr, sizeof(struct sockaddr_in));
	if(out==-1)
	{
		out=errno;
		disp_msg(ERR_MSG,"send_a_db_result","|d",(int)dest_addr,"|d",(int)srch_sck,"|d",(int)out,NULL);
	}

	disp_msg(DEBUG_MSG,"send_search_result_line",str->str,NULL);

	g_string_free(str,TRUE);
}

/***************************************************************************/
/* search a pattern inside database and send results into the given socket */
/*******************************************************************************/
/* input: sck= socket where results should be written                          */
/*        dest_nick= nickname of the destination user (can be NULL)            */
/*        size_matter= 0:no size requirement, 1:size at least, 2: size at most */
/*        size: if size_matter, it is the required size.                       */
/*        pattern: one or more words separated by space.                       */
/*        md5sum= md5sum to match with (md5sum can be null)                    */
/*        dest_addr: address to send result (active mode), can be NULL (=passiv*/
/*                (if dest_nick==NULL, dest_addr is the destination)           */
/*                (if dest_addr==NULL, dest_nick is the destination)           */
/*******************************************************************************/
void search_in_db(int sck, char *dest_nick, int file_type, SIZE_ATTRIB size_matter, unsigned long size, char *pattern, unsigned char *md5sum, struct sockaddr_in *dest_addr)
{
	int i;
	int nb_result=0;
	GPtrArray *decpat;

	decpat=g_ptr_array_new();
	
	decpat=decode_pattern(decpat,pattern);

	if(decpat->len!=0)
	{	/* something to search ? */
		G_LOCK(shared_info);
		if(shared_info!=NULL)
		{
			for(i=0;i<shared_info->len;i++)
			{
				DB_ENTRY *de;
	
				/* DB_ENTRY to compare */
				de=&(g_array_index(shared_info,DB_ENTRY,i));
	
				if(!size_match(size_matter,size, de->filesize))	/* size match ? */
					continue;
	
				if(!type_match(file_type, de->filetype))			/* type match ? */
					continue;
	
				if(!pattern_match(decpat, de->filename))			/* valid file ? */
				{
					/* the filename does not match but perhaps the sum does */
					if((md5sum==NULL)||
						(!md5sum_match(md5sum,de->md5sum)))
						continue;
				}
	
				send_a_db_result(sck,dest_nick, de,dest_addr,md5sum);
				nb_result++;
				if(nb_result==5)
					break;
			}
		}
		G_UNLOCK(shared_info);

		for(i=0;i<decpat->len;i++)
		{
			g_string_free(g_ptr_array_index(decpat,i),TRUE);
		}
	}

	g_ptr_array_free(decpat,TRUE);
}

/*************************************************************/
/* check if the given filename exists in the shared database */
/*************************************************************/
/* output: 0=no, 1=yes                                   */
/* if yes, *virtual ==1 if the file is virtual else == 0 */
/*********************************************************/
int file_in_db(char *filename, int *virtual)
{
	int i;

	G_LOCK(shared_info);
	if(shared_info!=NULL)
	{
		for(i=0;i<shared_info->len;i++)
		{
			DB_ENTRY *de;

			/* DB_ENTRY to compare */
			de=&(g_array_index(shared_info,DB_ENTRY,i));
			if(!strcmp(de->filename,filename))
			{
				*virtual=de->virtual;
				G_UNLOCK(shared_info);
				return 1;
			}
		}
	}
	G_UNLOCK(shared_info);
	return 0;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ------------------------- search in DB by CRC ---------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/*******************************************************************************************/
/* search a file having given md4 CRC (MD4_DIGEST_LENGTH bytes) with the given file_length */
/* in the MD DB and if not found, search inside the shared file db                         */
/*******************************************************************************************/
void find_file_with_crc_and_length_and_reply(int output_sck, const char *searching_nick, const guint8 *md4, guint32 file_length)
{
	MD_INFO *mi;
	int i;
	GString *str;
	GString *adapted;
	int out;

	mi=get_md4(md4,file_length);
	if(mi!=NULL)
	{
		str=g_string_new("$MD4Set ");
		g_string_append(str,searching_nick);
		g_string_append_c(str,' ');
		append_MD4_to_str(str,mi->global_crc);
		g_string_sprintfa(str," %lu ",(unsigned long)mi->filesize);
		for(i=0;i<mi->nb_seg;i++)
		{
			append_MD4_to_str(str,mi->partial_crc+i*MD4_DIGEST_LENGTH);
		}
		g_string_append_c(str,' ');

		adapted=g_string_new(mi->filename);
		adapted=convert_path(adapted);
		g_string_append(str,adapted->str);
		g_string_free(adapted,TRUE);

		g_string_append_c(str,'|');
		
		out=send(output_sck,str->str,str->len,MSG_NOSIGNAL);
		if(out==-1)
		{
			out=errno;
			disp_msg(ERR_MSG,"find_file_with_crc_and_length_and_reply","send error:","|d",(int)out,NULL);
		}
	
		g_string_free(str,TRUE);
		free_md_info(mi);
		return;
	}

	/* well, the CRC:filesize is not in the database       */
	/* let's try to find a file with the matching filesize */
	G_LOCK(shared_info);
	if(shared_info!=NULL)
	{
		for(i=0;i<shared_info->len;i++)
		{
			DB_ENTRY *de;
	
			/* DB_ENTRY to compare */
			de=&(g_array_index(shared_info,DB_ENTRY,i));

			if(de->filesize==file_length)
			{
				/* we have a file with the same size, perhaps it is already inside the database */
				if(!md4_is_inside(file_length,de->filename))
				{
					/* no, let's compute the CRC of this file */
					guint8 comp_g_crc[MD4_DIGEST_LENGTH];
					guint8 *comp_l_crc;
					size_t comp_length;
					guint32 nb_seg;
					int ret_comp;
					
					{
						GString *full_path;

						full_path=g_string_new(de->filename);

						/* we always prepend a '/' at the beginning of the fullpathname */
						/* this is done in a very dirty way. Due to the fact that the command is always before the fullname */
						/* and we won't use it anymore, we "accidentally" trash it :) */
						g_string_prepend_c(full_path,'/');

						ret_comp=compute_file_crcs(full_path->str,comp_g_crc,&comp_l_crc,&comp_length);
						g_string_free(full_path,TRUE);
					}

					if(ret_comp==0)
					{
						nb_seg=(comp_length+PARTSIZE-1)/PARTSIZE;
						md4_add(comp_g_crc,comp_length,comp_l_crc,nb_seg,de->filename);

						if(!memcmp(comp_g_crc,md4,MD4_DIGEST_LENGTH))
						{
							str=g_string_new("$MD4Set ");
							g_string_append(str,searching_nick);
							g_string_append_c(str,' ');
							append_MD4_to_str(str,comp_g_crc);
							g_string_sprintfa(str," %lu ",(unsigned long)comp_length);
							for(i=0;i<nb_seg;i++)
							{
								append_MD4_to_str(str,comp_l_crc+i*MD4_DIGEST_LENGTH);
							}
							g_string_append_c(str,' ');

							adapted=g_string_new(de->filename);
							adapted=convert_path(adapted);
							g_string_append(str,adapted->str);
							g_string_free(adapted,TRUE);
					
							g_string_append_c(str,'|');
		
							out=send(output_sck,str->str,str->len,MSG_NOSIGNAL);
							if(out==-1)
							{
								out=errno;
								disp_msg(ERR_MSG,"find_file_with_crc_and_length_and_reply","send error:","|d",(int)out,NULL);
							}
						
							g_string_free(str,TRUE);
							free(comp_l_crc);
							G_UNLOCK(shared_info);
							return;
						}
						free(comp_l_crc);
					}
				}
			}
		}
	}
	G_UNLOCK(shared_info);
}

