# Makefile for Setup and Installation of dbengine
# Compatible with Linux and Windows
#
# Version 1.2
# <c> 2001 Ingo Ciechowski
# see closer information at http://www.cis-computer.com/dbengine

BEGIN {
	require 5.003;
	unshift @INC, "lib";
	unshift @INC, "lib/auto" if (-d "lib/auto");	# this is needed to load mysql DBD; see lib.pm docs for details
}

use Config;
use Test::Harness;	# tests use it so check it's loadable early
use Cwd;		# added these two modules to support
use Sys::Hostname;	# Windows machines
use ExtUtils::MakeMaker qw(prompt);

my $options = { 'prompt' => 1 };

# Configuration files
my $config_file	= "base/confdat";
my $config_out	= "base/dbengine.cfg";
my $base_cgi	= "base/dbengine.cgi";
my $output_cgi	= "base/dbengine.cgi.out";

my $mode	= $ARGV[0];
my $pwd		= getcwd;
my $host	= hostname;
my %supported_drivers = (
				'Oracle'	=> 'oracle',
				'Postgres'	=> 'pg',
				'MySQL'		=> 'mysql',
);
	
	
#
# check perl version
if ($] < 5.005) {
	print "\n*** YOU ARE USING A VERY OLD VERSION OF PERL. ***\n\a";
	print "Perl 5.005 and later fixes many bugs in older versions and is highly recommended.\n";
	print "You may see warnings from 'pod2man' due to the old version you have.\n";
	print "You can ignore them, or, better still, upgrade to the latest release\n";
	print "of Perl (5.6.1 at the time this was written).\n\n";
	sleep 3;
}


#
# check if required modules are installed
my @missing;

eval "use CGI;";
if ($@) {
	push @missing, 'CGI';
	print "\a",<<'MSG';
*** ERROR:
	The required CGI package is not installed on your system.
	It it usually part of any perl source code distribution and
	will be installed when you run perls 'make install'.

MSG
	sleep 2;
}

eval "use Sys::Syslog;";
if($@ && ($^O !~ /win/i || $^O =~ /darwin/i)) {
	push @missing, 'CGI';
	print "\a",<<'MSG';
*** ERROR:
	The required Sys::Syslog package is not installed on your system.

MSG
	sleep 2;
}

eval "use Carp;";
if ($@) {
	push @missing, 'CGI';
	print "\a",<<'MSG';
*** ERROR:
	The required Carp package is not installed on your system.

MSG
	sleep 2;
}

eval "use DBI 1.14 ();";
if ($@) {
	push @missing, 'CGI';
	print "\a",<<'MSG';
*** ERROR:
	The required DBI package is not installed on your system.
	You need version 1.15 or later.

MSG
	sleep 2;
}


# Check to see if any of the required DBD drivers were missing
if (@missing) {
	print &print_module_error_msg;
	exit 1;
}


#
# read options file
init_options();

#
# check for DBD driver for selected database
if ($db eq 'oracle') {
	eval "use DBD::Oracle 0.59 ();";
	if ($@) {
		push @missing, 'CGI';
		print "\a",<<'MSG';
*** ERROR:
	The DBD-Oracle package is too old or not properly installed on
	your system. You need version 0.59 or later if you want to
	access an Oracle database using dbengine.

MSG
		sleep 2;
	}
}

if ($db eq 'postgres') {
	eval "use DBD::Pg 0.90 ();";
	if ($@) {
		push @missing, 'CGI';
		print "\a",<<'MSG';
*** ERROR:
	The DBD-Pg package is too old or not properly installed on
	your system. You need version 0.90 or later if you want to
	access a PostgreSQL database using dbengine.

MSG
		sleep 2;
	}
}
	
if ($db eq 'mysql') {
	eval "use DBD::mysql 2.04 ();";
	if ($@) {
		push @missing, 'CGI';
		print "\a",<<'MSG';
*** ERROR:
	The Msql-Mysql modules is too old or not properly installed on
	your system. You need version v1.22.16 or later if you want to
	access a mySQL database using dbengine.

MSG
		sleep 2;
	}
}

#
# Check to see if any of the require DBD drivers were missing
if (@missing) {
	print &print_module_error_msg;
	exit 1;
}

#
# Else, figure out step to take next
if($mode eq "create") {
	&create();

} elsif($mode eq "test") {
	&test();

} elsif($mode eq "install") {
	&install();

#
# ask for options and write Makefile
} else {
	&configure();
} # end if


exit 0;





#######################################################################
#######################################################################

#
# create the dbengine.cgi file from the dbengine.base
# customize settings according to values in configuration file
sub create {
	open (OUTCONF, ">$config_out");
	print OUTCONF "[CONFIG]\n";
	while (($desc, $varname, $value) = split /\|/,shift @configure) {
		if($varname =~ /config_file/) {
			$replace_config_file = $value;
			next;
		}
		if($varname =~ /path_to_local_modules/) {
			$replace_path_to_local_modules = $value;
			next;
		}
		next if($varname =~ /xgroup/ || $varname =~ /xuser/);
		print OUTCONF "# $desc\n";
		my $txt = sprintf("%-20s %s\n\n", $varname, $value);
		printf OUTCONF "$txt";
	}
	print OUTCONF "\n\n";
	close OUTCONF;

	open (BASEFILE, "<$base_cgi");
	open (OUTFILE, ">$output_cgi");
	print OUTFILE "#!".$Config{'perlpath'}."\n";
	#
	# scan default dbengine.cgi and create new copy with user-defined variable settings
	while(<BASEFILE>) {
		next if(/\#\!\/usr\/bin\/perl/);
		if(/\$config_file\s+=/) {
			print OUTFILE "\t\$config_file = \"$replace_config_file\";\n";
		} elsif(/\@path_to_local_modules\s+/) {
			print OUTFILE "\t"."my \@path_to_local_modules = qw[ $replace_path_to_local_modules ];\n";
		} else {
			print OUTFILE $_;
		}
	}
	close BASEFILE;
	close OUTFILE;
	print "dbEngine and config file successfully built. Continue with `make test` or `make install`.\n\n";
} # create


#
# testing
sub test {
	if ($^O !~ /darwin/i && $^O =~ /win/i && $params{dbasetype} ne 'mysql' ) {
		print "Testing is not supported under Windows with Oracle or Postgres yet.";
		exit 1;
	}

	print "\ndbengine test initialisation\nUSER ACCESS TO DATABASE MUST BE PERMITTED !!\n\n";
	print "name[/password if Oracle] of test database [$params{defdb}] : "; $in=getline(); if(length($in)>0) { $params{defdb} = $in; }
	print "name[/password if Oracle] of describing test database [$params{descdb}] : "; $in=getline(); if(length($in)>0) { $params{descdb} = $in; }
	print "\n\n";

	print "Database: " . $params{defdb} . "\nDescDb: " . $params{descdb} . "\n";


	#
	# if Oracle
	if($params{dbasetype} eq 'oracle') {
		$ENV{TWO_TASK}		= $params{ oraTWOTASK };
		$ENV{ORACLE_HOME}	= $params{ oraHOME };
		$ENV{ORACLE_SID}	= $params{ oraSID };

		print "\nMake sure $params{defdb} and $params{descdb} have access to a database\n";
		print "using ORACLE_SID=$params{ oraSID } and TWO_TASK=$params{ oraTWOTASK } (Press Return)\n";
		$in = getline();

		#
		# load basic tables and test data
		print "\nstarting sqlplus...\n";
		`sqlplus $params{descdb} < util/test/Oracle/descdb.sql`;
		`sqlplus $params{defdb} < util/test/Oracle/sample_data.sql`;
		`sqlplus $params{descdb} < util/test/Oracle/sample_desc.sql`;

	#
	# if PostgreSQL
	} elsif ($params{dbasetype} eq 'postgres') {
		do { print "shall $params{defdb} and $params{descdb} be removed and recreated (y/n)? "; $in= lc getline(); } while ($in ne "y" && $in ne "n");
		if ( $in eq "y" ) {
			if($> != 0) {
				#
				# destroy old databses
				print `dropdb $params{defdb}`;
				print `dropdb $params{descdb}`;
				#
				# initialize them from scratch
				print `createdb $params{defdb}`;
				print `createdb $params{descdb}`;
			} else {
				#
				# destroy old databses
				print `su - $params{user} -c "destroydb $params{defdb}"`;
				print `su - $params{user} -c "destroydb $params{descdb}"`;
				#
				# initialize them from scratch
				print `su - $params{user} -c "createdb $params{defdb}"`;
				print `su - $params{user} -c "createdb $params{descdb}"`;
			}
		}

		#
		# load basic tables and test data
		if($> != 0) {
			`psql $params{descdb} < util/test/PostgreSQL/descdb.sql`;
			`psql $params{defdb}  < util/test/PostgreSQL/sample_data.sql`;
			`psql $params{descdb} < util/test/PostgreSQL/sample_desc.sql`;
		} else {
			`su - $params{user} -c "psql $params{descdb} < $pwd/util/test/PostgreSQL/descdb.sql"`;
			`su - $params{user} -c "psql $params{defdb}  < $pwd/util/test/PostgreSQL/sample_data.sql"`;
			`su - $params{user} -c "psql $params{descdb} < $pwd/util/test/PostgreSQL/sample_desc.sql"`;
		}

	# if MySQL
	} elsif ($params{dbasetype} eq 'mysql') {
		# MySQL tests
			print "For these tests to work, you need to have the path to the mysql runtime properly set and have permissions to create and drop tables. If you receive errors, check the database permissions for the user and password you entered in the configuration section ($params{user}).\n";
		do { print "Shall $params{defdb} and $params{descdb} be removed and recreated (y/n)? "; $in= lc getline(); } while ($in ne "y" && $in ne "n");
		#
		# Under Win*, the EUID is 0; 
		# The following tests have only been conducted under Win*
		if ( $in eq "y" ) {
			if($> != 0) {
				#
				# destroy old databses
				print `mysql -u $params{user} -p$params{passwd} -e "drop database if exists $params{defdb}"`;
				print `mysql -u $params{user} -p$params{passwd} -e "drop database if exists $params{descdb}"`;
				#
				# initialize them from scratch
				print `mysql -u $params{user} -p$params{passwd} -e "create database $params{defdb}"`;
				print `mysql -u $params{user} -p$params{passwd} -e "create database $params{descdb}"`;
			} else {
					#
					# Check for Win*
					if ($^O !~ /darwin/i && $^O =~ /win/i) {
				  #
				  # destroy old databases
				  print `mysql -u $params{user} -p$params{passwd} -e "drop database if exists $params{defdb}"`;
				  print `mysql -u $params{user} -p$params{passwd} -e "drop database if exists $params{descdb}"`;
				  #
				  # create new databases
				  print `mysql -u $params{user} -p$params{passwd} -e "create database $params{defdb}"`;
				  print `mysql -u $params{user} -p$params{passwd} -e "create database $params{descdb}"`;
  					}
				else {
				  #
				  # destroy old databses
				  print `su - $params{user} -c "mysql -u $params{user} -p$params{passwd} -e 'drop database if exists $params{defdb}'`;
				  print `su - $params{user} -c "mysql -u $params{user} -p$params{passwd} -e 'drop database if exists $params{descdb}'"`;
				  #
				  # initialize them from scratch
				  print `su - $params{user} -c "mysql -u $params{user} -p$params{passwd} -e 'create database $params{defdb}'"`;
				  print `su - $params{user} -c "mysql -u $params{user} -p$params{passwd} -e 'create database $params{descdb}'"`;
				}
			}
			print "Databases rebuilt.\n";
		}

		#
		# load basic tables and test data for MySQL
		# You may get errors from MySQL if you try running make test multiple times without dropping the tables first; I should probably add a check to see if the tables already exist!
		# only tested under WinNt...
		print "Loading basic tables and test data...\n";
		if($> != 0) { # not tested
		  print `mysql -u $params{user} -p$params{passwd} $params{descdb} < $pwd/util/test/MySQL/descdb.sql`;
		  print `mysql -u $params{user} -p$params{passwd} $params{defdb} < $pwd/util/test/MySQL/sample_data.sql`;
		  print `mysql -u $params{user} -p$params{passwd}  $params{descdb} < $pwd/util/test/MySQL/sample_desc.sql`;
		} else {
		  #
		  # Check for Win* which, to my knowledge, always reports EID of 0
		  if ($^O !~ /darwin/i && $^O =~ /win/i) {
			print "Creating description database $params{descdb}...";
			print `mysql -u $params{user} -p$params{passwd} $params{descdb} < $pwd/util/test/MySQL/descdb.sql`;
			print "done.\nImporting sample data into $params{defdb}...";
			print `mysql -u $params{user} -p$params{passwd} $params{defdb} < $pwd/util/test/MySQL/sample_data.sql`;
			print "done.\nImporting sample data into $params{descdb}...";
			print `mysql -u $params{user} -p$params{passwd}  $params{descdb} < $pwd/util/test/MySQL/sample_desc.sql`;
			print "done.\n";
		  }
		  else {  # not tested
			`su - $params{user} -c "mysql -u $params{user} -p$params{passwd} $params{descdb} < $pwd/util/test/MySQL/descdb.sql"`;
			`su - $params{user} -c "mysql -u $params{user} -p$params{passwd} $params{defdb} < $pwd/util/test/MySQL/sample_data.sql"`;
			`su - $params{user} -c "mysql -u $params{user} -p$params{passwd}  $params{descdb} < $pwd/util/test/MySQL/sample_desc.sql"`;
		  }
		}
		print "Tables loaded.\n";
	# end if MySQL

	}

	print "\n\n";
	do {	my $tmp = "$params{htdocs}$params{tmp}"; $tmp =~ s|//|/|g; # remove duplicate slashes
		print "shall a sample template HTML file be installed in $params{templ}\n";
		print "and navigation.gif in $tmp (y/n)? "; $in= lc getline(); } while ($in ne "y" && $in ne "n");
	if ( $in eq "y" ) {
	  if ($^O !~ /darwin/i && $^O =~ /win/i) {
		`copy samples/template/articles.html $params{templ}`;
		`copy samples/template/real_life/navigation.gif $params{htdocs}$params{tmp}`;
	  }
	  else {
		`mkdir -p $params{templ}`;
		`mkdir -p $params{htdocs}$params{tmp}`;
		`cp samples/template/articles.html $params{templ}`;
		`cp samples/template/real_life/navigation.gif $params{htdocs}$params{tmp}`;
		`chown $params{xuser}:$params{xgroup} $params{templ}/articles.html`;
		`chown $params{xuser}:$params{xgroup} $params{htdocs}$params{tmp}/navigation.gif`;
		`chmod -R 754 $params{templ}/articles.html`;
		`chmod -R 775 $params{htdocs}$params{tmp}/navigation.gif`;
	  }
	}

	print "\n\n";
	print "*** Testdata is in place now. Go ahead with 'make install' if not already done.\n";
	print "*** After that you'll be able to access your database with the following URL:\n";
	print "*** http://".$host."$params{cgis}dbengine.cgi\n";
	print "\n\n";
} # test

#
# install
sub install {
	if ($^O !~ /darwin/i && $^O =~ /win/i) {
		my $abs_cgi = $params{abs_cgi};
		$abs_cgi =~ s|/|\\|g;
		my $templ = $params{templ};
		$templ =~ s|/|\\|g;
		my $tmp = "$params{htdocs}$params{tmp}";
		$tmp =~ s|/|\\|g;
		$output_cgi =~ s|/|\\|g;
		my $cfg_in = $config_out;
		$cfg_in =~ s|/|\\|g;
		my $cfg_out = $params{config_file};
		$cfg_out =~ s|/|\\|g;
		my $util = "$abs_cgi\\util";
#		my $cgi_dir = params{abs_cgi};
		`md $util` unless (-d $util);
		`md $tmp` unless (-e $tmp);  # perl isn't recognizing this as a dir but as a file-why? wlm
		`md $templ` unless (-d $templ);
		`copy $cfg_in $cfg_out`;
		`copy $output_cgi $abs_cgi\dbengine.cgi`;
		die "\terror copying $output_cgi to $abs_cgi" unless (-e "$abs_cgi/dbengine.cgi");
		`copy util\\LOG.pm $util`;
		`copy util\\base.pl $util`;
		#
		# Copy database libraries
		`copy util\\$params{dbasetype}.pl $util\\`;
	} else {
		`mkdir -p $params{abs_cgi}/util`;
		`mkdir -p $params{templ}`;
		`mkdir -p $params{htdocs}$params{tmp}`;
		`cp $output_cgi $params{abs_cgi}/dbengine.cgi`;
		`cp $config_out $params{config_file}`;
		`cp util/LOG.pm $params{abs_cgi}/util/`;
		`cp util/base.pl $params{abs_cgi}/util/`;
		`chown -R $params{xuser}:$params{xgroup} $params{abs_cgi}`;
		`chown $params{xuser}:$params{xgroup} $params{templ}`;
		`chown $params{xuser}:$params{xgroup} $params{htdocs}$params{tmp}`;
		`chmod -R 754 $params{abs_cgi}`;
		`chmod -R 754 $params{templ}`;
		`chmod -R 775 $params{htdocs}$params{tmp}`;
		#
		# Copy database libraries
		`cp util/$params{dbasetype}.pl $params{abs_cgi}/util/`;
	}

	&loadBasicDescDB();
		
	print "\n\n";
	print "*** You now can access your database with the following URL:\n";
	print "*** http://".$host."$params{cgis}dbengine.cgi\n";
	print "\n\n";
} # install


sub loadBasicDescDB {
	#
	# now we're filling the describing database with basic tables and content
	if ($params{dbasetype} eq 'oracle') {
		$ENV{TWO_TASK}		= $params{ oraTWOTASK };
		$ENV{ORACLE_HOME}	= $params{ oraHOME };
		$ENV{ORACLE_SID}	= $params{ oraSID };
		print "\n\n$params{descdb} < util/test/descdb.sql to load basic describing database...\n";
		`sqlplus $params{descdb} < util/test/Oracle/descdb.sql`;
	} elsif ($params{dbasetype} eq 'postgres') {
		if($> != 0) {
			print "\n\n$params{descdb} < util/test/descdb.sql to load basic describing database...\n";
			`psql $params{descdb} < util/test/PostgreSQL/descdb.sql`;
		} else {
			print "\n\n$params{descdb} < util/test/descdb.sql to load basic describing database...\n";
			`su - $params{user} -c "psql cisinfo < $pwd/util/test/PostgreSQL/descdb.sql"`;
		}
	} elsif ($params{dbasetype} eq 'mysql') {
		# MySQL install
	}
} # loadBasicDescDB


#
# configure installation
sub configure {
	do {
		init_options();

		#
		# If this is a Windows OS, tell user to use '/' dir separators
		if ($^O !~ /darwin/i && $^O =~ /win/i) {
			print "Windows OS detected: Remember to use a slash (/) to separate directory names when configuring the options.";
		}
		#
		# ask for configuration information and save it
		open (CONFOUT, ">$config_file");
		foreach $line (@configure) {
			($desc, $varname, $default, $option, $original) = split /\|/,$line;
			# Check for database type
			if ($varname eq 'dbasetype') {
				$value = &get_database_type($default);
			} else {
				$original = $default unless ($original);
				$localdefault = $default?$default:$original;
				print "\n$desc ".($option?"($option) ":'')."\n[$localdefault] : ";
				$value = getline();
			}
			$value = $localdefault unless (length($value)>0);
			$value = $original if($value eq "d");

			print CONFOUT "$desc|$varname|$value|$option|$original\n";
		}
		close CONFOUT;

		init_options();
		print "\n\nThese are the saved values:\n";
		foreach $line (@configure) {
			($desc, $varname, $value, $option, $original) = split /\|/,$line;
			print "$desc: $value\n";
		}
		
		do {
			print "\nAre these settings ok (y/n) ? ";
			$in = lc getline();
		} while ($in ne "y" && $in ne "n");

	} while ($in eq "n");
	open (MAKEFILE, ">Makefile");
	print MAKEFILE <<MAKEFILE;
$output_cgi:
	perl Makefile.PL create

all: $output_cgi

install:
	perl Makefile.PL install

test:
	perl Makefile.PL test

clean:
	rm -f $output_cgi
	mv Makefile Makefile.old

realclean: clean
	rm -f Makefile Makefile.old
	rm -f $config_file
MAKEFILE
} # configure


#
# read options file and fill 
sub init_options {
	#
	# initialize default options - order must be same as in dbengine.pl
	@configure = (
		'Database type (postgres,mysql,oracle)|dbasetype|',
		'Domain name of the server where your database lives (PostgreSQL/MySQL)|server|localhost',
		'Port number of the database server (PostgreSQL/MySQL)|port|5432',
		'User to access database with (PostgreSQL/MySQL)|user|postgres',
		'Password (if any) to access the database (PostgreSQL/MySQL)|passwd',
		'Name of default database (PostgreSQL/MySQL) [/<access password>] (Oracle)]|defdb|cis',
		'Name of default database for display information (PostgreSQL/MySQL) [/<access password>] (Oracle)]|descdb|cisinfo',
		'TWO_TASK environment variable (Oracle)|oraTWOTASK|',
		'ORACLE_SID environment variable (Oracle)|oraSID|CIS',
		'ORACLE_HOME environment variable (Oracle)|oraHOME|/opt/oracle/actual/product/8.0.5',
		'Absolute Filename including full Path of external Configuration file|config_file|/usr/local/httpd/dbengine.conf',
		'Absolute path to the directory where your local Perl libs have been installed|path_to_local_modules|/usr/local/lib/perl5/lib',
		'Absolute path to document-root directory of your WWW Server (ending slash required)|htdocs|/usr/local/httpd/htdocs/',
		'Absolute path to folder where dbengine.cgi will live (ending slash required)|abs_cgi|/usr/local/httpd/cgi-bin/dbengine/',
		'Absolute path to folder where template .html files are stored for processing (ending slash required)|templ|/usr/local/httpd/cgi-bin/dbengine/template/',
		'Absolute http path to folder for temporary .html files (ending slash required)|tmp|/dbengine/tmp/',
		'Absolute http path to folder where dbengine.cgi lives (ending slash required)|cgis|/dbengine/cgi-bin/',
		'Language|language|english',
		'List output using tables (1: true - 0: false)|smarttable|1',
		'Maximum width of an automatically generated text field|maxwidth|45',
		'Optional path to optional background images (ending slash required)|bgPath|/bgimages/',
		'Optional default extension for background images|bgType|.jpeg',
		'Optional default background image for the table list page|menuBackground|menuBG.jpeg',
		'Optional default background image for for generated pages|mainBackground|mainBG.jpeg',
		'Background color for generated pages|bgcol|#C4DDFF',
		'Table header background color|headerbgcol|#555555',
		'Table header text color|headertxtcol|#FFFFFF',
		'Table item background color|tablebgcol|#CDC7C2',
		'Table item text color|tabletxtcol|#000000',
		'UNIX-User running httpd|xuser|wwwrun',
		'UNIX-Group for www files|xgroup|wwwadmin',
	);

	#
	# if saved options exist, use those first...
	if (-e $config_file) {
		@zwerg = @configure;
		undef @configure;
		open (CONFIN, "<$config_file");
		while (my $line = <CONFIN>) {
			chomp $line;
			my ($desc, $varname, $value, $option, $original) = split /\|/, $line;
			push @configure, "$desc|$varname|$value";
		}
		close CONFIN;

		#
		# extend @configure with those options that were missing in the file
		foreach my $line (@zwerg) {
			($desc, $varname, $value) = split /\|/, $line;
			my $exists = 0;
			foreach my $xline (@configure) {
				my ($desc, $xvarname, $value, $option, $original) = split /\|/, $xline;
				$exists = 1 if($varname eq $xvarname);
			}
			push @configure, $line unless($exists == 1);
		}

		foreach my $line (@configure) {
			my ($desc, $varname, $value) = split /\|/,$line;
			$params{$varname} = $value;
		}
		undef @zwerg;
	}
}


#
# read a single line from STDIN and return it
# without trailing <CR>
sub getline {
	# For some reason, chomp won't work properly from the commandline 
	#	under Windows unless the input record separator is explicitly 
	#	set to \r\n; this seems like a bug but I'm not sure since Win*
	#	is abnormal in many regards - wlm
	local $/="\r\n" if ($^O !~ /darwin/i && $^O =~ /win/i);
	
	local($line) = "";
	local($in);
	do {	# similar to $line = <STDIN>
		$in = getc;
		$line .= $in;
	} while ($in ne "\n");
	chomp $line;
	$line =~ s/[\r\n]//g;
	$line;
}


#
# generate an error message if the user does not
# have a required module installed
sub print_module_error_msg {
	my $msg = q!
	Modules are available from any CPAN mirror, in particular
	http://www.perl.com/CPAN/modules/by-module
	http://www.perl.org/CPAN/modules/by-module
	ftp://ftp.funet.fi/pub/languages/perl/CPAN/modules/by-module
	!;
}

#
# return a configuration string based on user input which
# contains the type of database for which dbEngine is being configured
sub get_database_type {
	my ($choice) = @_;
	my @available_drivers = DBI->available_drivers;
	
	#print "The following drivers are supported by dbEngine:\n";
	#print map {"\t$_\n"} keys %supported_drivers;
	#print "You have the following DBI drivers installed on this system:\n";
	#print map {"\t$_\n"} @available_drivers;

	foreach $dbd (keys %supported_drivers) {
		my $driver_name = $supported_drivers{$dbd};
		my $avail = (grep { $_ =~ /$driver_name/i } @available_drivers) ? "(available)" : "(unavailable)";
		push @choices, "$dbd $avail";
	}

	while ($choice < 1 || $choice > 3) {
		if($choice eq "postgres") {
			$choice = 1;
		} elsif($choice eq "oracle") {
			$choice = 2;
		} else {
			$choice = 3;
		}
		$choice = prompt
			("\nWhich database do you want to install for dbEngine?\n"
			 . " 1)  $choices[0]\n"
			 . " 2)  $choices[1]\n"
			 . " 3)  $choices[2]\n"
			 . "Enter the appropriate number: ", $choice);
	}
	
	if ($choice == 1)	{ $db = 'postgres' }
	elsif ($choice == 2)	{ $db = 'oracle' }
	elsif ($choice == 3)	{ $db = 'mysql' }
	else			{ print "Invalid choice.\n" }

	$db;
}
