%  Copyright (C) 2002-2004 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; see the file COPYING.  If not, write to
%  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
%  Boston, MA 02110-1301, USA.

\begin{code}
module Darcs.Patch.TouchesFiles ( look_touch, choose_touching,
                      select_touching,
                      deselect_not_touching, select_not_touching,
                    ) where
import Data.List ( sort )

import Darcs.Patch.Choices ( PatchChoices, patch_choices, tag,
                             get_first_choice, get_middle_choice, get_last_choice,
                      force_firsts, force_lasts, tp_patch,
                    )
import Darcs.Patch ( Patchy, apply_to_filepaths, list_touched_files )
import Darcs.Patch.Ordered ( FL(..), mapFL_FL, (+>+) )
\end{code}

\begin{code}
select_touching :: Patchy p => [FilePath] -> PatchChoices p -> PatchChoices p
select_touching [] pc = pc
select_touching files pc = force_firsts xs pc
    where ct _ NilFL = []
          ct fs (tp:>:tps) = case look_touch fs (tp_patch tp) of
                             (True, fs') -> tag tp:ct fs' tps
                             (False, fs') -> ct fs' tps
          xs = ct (map fix files) (get_middle_choice pc +>+ get_last_choice pc)

deselect_not_touching :: Patchy p => [FilePath] -> PatchChoices p -> PatchChoices p
deselect_not_touching [] pc = pc
deselect_not_touching files pc = force_lasts xs pc
    where ct _ NilFL = []
          ct fs (tp:>:tps) = case look_touch fs (tp_patch tp) of
                             (True, fs') -> ct fs' tps
                             (False, fs') -> tag tp:ct fs' tps
          xs = ct (map fix files) (get_first_choice pc +>+ get_middle_choice pc)

select_not_touching :: Patchy p => [FilePath] -> PatchChoices p -> PatchChoices p
select_not_touching [] pc = pc
select_not_touching files pc = force_firsts xs pc
    where ct _ NilFL = []
          ct fs (tp:>:tps) = case look_touch fs (tp_patch tp) of
                             (True, fs') -> ct fs' tps
                             (False, fs') -> tag tp:ct fs' tps
          xs = ct (map fix files) (get_first_choice pc +>+ get_middle_choice pc)

fix :: FilePath -> FilePath
fix f | take 1 (reverse f) == "/" = fix $ reverse $ drop 1 $ reverse f
fix "" = "."
fix "." = "."
fix f = "./" ++ f

choose_touching :: Patchy p => [FilePath] -> FL p -> FL p
choose_touching [] p = p
choose_touching files p =
    mapFL_FL tp_patch $ get_first_choice $ select_touching files $ patch_choices p

look_touch :: Patchy p => [FilePath] -> p -> (Bool, [FilePath])
look_touch fs p = (orr affects (list_touched_files p) fs
                   || fs' /= fs, fs')
    where orr :: (a -> b -> Bool) -> [a] -> [b] -> Bool
          orr fun (a:as) bs = any (fun a) bs || orr fun as bs
          orr _ [] _ = False
          affects touched f | touched == f = True
          affects t f = case splitAt (length f) t of
                        (t', '/':_) -> t' == f
                        _ -> case splitAt (length t) f of
                             (f', '/':_) -> f' == t
                             _ -> False
          fs' = sort $ apply_to_filepaths p fs
\end{code}
