%  Copyright (C) 2003-2004 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\subsection{darcs tag}
\begin{code}
module Tag ( tag ) where
import Monad ( liftM, when )

import DarcsCommands
import DarcsArguments
import DarcsUtils ( askUser )
import Repository
import Patch
import PatchInfo
import Depends
import Record ( get_date )
import Lock ( withLock )
\end{code}
\begin{code}
tag_description :: String
tag_description =
 "Tag the contents of a repo with a given version name."
\end{code}

\options{tag}

\haskell{tag_help} Tag differs from record in that it doesn't record any
new changes, and it always depends on all patches residing in the
repository when it is tagged.  This means that one can later reproduce this
version of the repository by calling, for example:
\begin{verbatim}
% darcs get --tag "darcs 3.14" REPOLOCATION
\end{verbatim}

\begin{code}
tag_help :: String
tag_help =
 "Tag is used to name a version of the tree.\n"
\end{code}
\begin{code}
tag :: DarcsCommand
tag = DarcsCommand {command_name = "tag",
                    command_help = tag_help,
                    command_description = tag_description,
                    command_extra_args = 0,
                    command_extra_arg_help = [],
                    command_command = tag_cmd,
                    command_prereq = am_in_repo,
                    command_get_arg_possibilities = return [],
                    command_argdefaults = nodefaults,
                    command_darcsoptions = [patchname_option, author,
                                            checkpoint, nocompress,
                                            pipe_interactive, verbose]}
\end{code}
\begin{code}
tag_cmd :: [DarcsFlag] -> [String] -> IO ()
tag_cmd opts _ = withLock "./_darcs/lock" $ do
    date <- get_date opts
    the_author <- get_author opts
    deps <- liftM get_tags_right $ read_repo "."
    name <- get_patchname opts
    let mypatch = namepatch date name the_author [] null_patch
        myinfo = patchinfo date name the_author []
       in do
       write_patch opts $ adddeps mypatch deps
       add_to_inventory "." myinfo
       when (CheckPoint `elem` opts) $ write_recorded_checkpoint myinfo
       putStrLn $ "Finished tagging patch '"++name++"'"
\end{code}
Each tagged version has a version name.
\begin{code}
get_patchname :: [DarcsFlag] -> IO String
get_patchname (PatchName n:_) = return $ "TAG "++n
get_patchname (_:flags) = get_patchname flags
get_patchname [] = do
    n <- askUser "What is the version name? "
    if n == "" then get_patchname []
               else return $ "TAG "++n
\end{code}
The version is also flagged with the person who tagged it (taken by default
from the `DARCS\_EMAIL' or `EMAIL' environment variable).  The date is also
included in the version information.

A tagged version automatically depends on all patches in the repo.  This
allows you to later reproduce precisely that version.  The tag does this by
depending on all patches in the repo, except for those which are depended
upon by other tags already in the repo.  In the common case of a sequential
series of tags, this means that the tag depends on all patches since the
last tag, plus that tag itself.

\begin{options}
--checkpoint
\end{options}
The \verb!--checkpoint! option allows the tag be used later with the
\verb!--partial! flag to \verb!get! or \verb!check!.

A partial repository only contains patches from after the checkpoint.  A
partial repository works just like a normal repository, but any command that
needs to look at the contents of a missing patch will complain and abort.

\begin{options}
--pipe
\end{options}

If you run tag with the \verb!--pipe! option, you will be prompted for the
tag name and date.  This interface is intended for scripting darcs, in
particular for writing repository conversion scripts.  The prompts are
intended mostly as useful guide (since scripts won't need them), to help
you understand the format in which to provide the input. Here's an example
of what the \verb!--pipe! prompts looks like:

\begin{verbatim}
 What is the date? Mon Nov 15 13:38:01 EST 2004
 Who is the author? David Roundy
 What is the version name? 3.0
 Finished tagging patch 'TAG 3.0'
\end{verbatim}

Using \verb!tag! creates an entry in the repo history just like \verb!record!.
It will show up with \verb!darcs changes! appearing in the format:

\begin{verbatim}
 tagged My Tag Name
\end{verbatim}

Because the word 'tagged' is always prepended to the tag name, you can search
for tag names by simply passing the output of \verb!darcs changes! through \verb!grep!:

\begin{verbatim}
 darcs changes | grep tagged
\end{verbatim}

The above example would display all the tag names in use in the repo.

