%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

\begin{code}
module PatchRead ( readPatchPS )
             where

import Prelude hiding ( pi )
import Control.Monad ( liftM, unless )

import PatchInfo ( PatchInfo, readPatchInfoPS )
import FastPackedString ( PackedString, unpackPS, packString,
                          tailPS, headPS, nullPS, dropWhilePS,
                          mylexPS, breakOnPS, concatPS,
                          fromHex2PS, dropWhitePS,
                          readIntPS, nilPS, breakFirstPS
                        )
import FileName ( fn2fp, ps2fn )
import AntiMemo ( AntiMemo, antimemoize, readAntiMemo )
import PatchCore ( Patch(..), DirPatchType(..), FilePatchType(..),
                   invert, hunk, binary )
import PatchCommute ( merger )
#include "impossible.h"
\end{code}

\begin{code}
readPatchPS :: PackedString -> Maybe (Patch,PackedString)
readPatchPS s = case (unpackPS . fst) `liftM` mylexPS s of
                Just "{" -> readComPPS s -- }
                Just "(" -> readSplitPS s -- )
                Just "hunk" -> readHunkPS s
                Just "replace" -> readTokPS s
                Just "binary" -> readBinaryPS s
                Just "addfile" -> readAddFilePS s
                Just "adddir" -> readAddDirPS s
                Just "rmfile" -> readRmFilePS s
                Just "rmdir" -> readRmDirPS s
                Just "move" -> readMovePS s
                Just "changepref" -> readChangePrefPS s
                Just "merger" -> readMergerPS True s
                Just "regrem" -> readMergerPS False s
                Just ('[':_) -> readNamedPS s -- ]
                _ -> Nothing
\end{code}

\begin{code}
readComPPS :: PackedString -> Maybe (Patch,PackedString)
readComPPS s =
    case mylexPS s of
    Just (start,t) ->
        case read_patchesPS t of
        Just (ps,w) ->
            case mylexPS w of
            Just (end,x) -> if unpackPS end == "}" && unpackPS start == "{"
                            then Just (ComP ps, dropWhitePS x)
                            else Nothing
            Nothing -> impossible
        Nothing -> impossible
    Nothing -> impossible

read_patchesPS :: PackedString -> Maybe ([Patch],PackedString)
read_patchesPS s =
    case readPatchPS s of
    Nothing -> Just ([],s)
    Just (p,s') ->
        case read_patchesPS s' of
        Just (ps,s'') -> Just (p:ps,s'')
        Nothing -> impossible
\end{code}

\begin{code}
readSplitPS :: PackedString -> Maybe (Patch,PackedString)
readSplitPS = parseGP $ do
  start <- GenP mylexPS
  assertGP $ start == packString "("
  ps <- GenP read_patchesPS
  end <- GenP mylexPS
  assertGP $ end == packString ")"
  return $ Split ps
\end{code}

\begin{code}
readHunkPS :: PackedString -> Maybe (Patch,PackedString)
readHunkPS = parseGP $ do
  hun <- GenP mylexPS
  assertGP $ hun == packString "hunk"
  fi <- GenP mylexPS
  l <- GenP readIntPS
  skipGP tailPS -- skipping the newline...
  _ <- GenP $ lines_starting_withPS ' ' -- skipping context
  old <- GenP $ lines_starting_withPS '-'
  new <- GenP $ lines_starting_withPS '+'
  _ <- GenP $ lines_starting_withPS ' ' -- skipping context
  return $ hunk (fn2fp $ ps2fn fi) l old new
\end{code}

\begin{code}
readTokPS :: PackedString -> Maybe (Patch,PackedString)
readTokPS = parseGP $ do
  rep <- GenP $ mylexPS
  assertGP $ rep == packString "replace"
  f <- GenP $ mylexPS
  regstr <- GenP $ mylexPS
  o <- GenP $ mylexPS
  n <- GenP $ mylexPS
  return $ FP (ps2fn f) $ TokReplace (drop_brackets $ unpackPS regstr)
                          (unpackPS o) (unpackPS n)
    where drop_brackets = init . tail
\end{code}

\paragraph{Binary file modification}

Modify a binary file
\begin{verbatim}
binary FILENAME
oldhex
*HEXHEXHEX
...
newhex
*HEXHEXHEX
...
\end{verbatim}
\begin{code}
-- This is a generic parser monad for convenience...
newtype GP a b = GenP (a -> Maybe (b,a))
instance Monad (GP a) where
    m >>= k          = GenP $ parse_then m k
    return x         = GenP (\a -> Just (x,a))
    fail _           = GenP (\_ -> Nothing)
parse_then :: GP a b -> (b -> GP a c) -> a -> Maybe (c,a)
parse_then (GenP f) g a = case f a of
                                 Nothing -> Nothing
                                 Just (b,x) -> parseGP (g b) x
parseGP :: GP a b -> a -> Maybe (b,a)
parseGP (GenP p) a = p a
skipGP :: (a -> a) -> GP a ()
skipGP s = GenP $ \a -> Just ((), s a)
assertGP :: Bool -> GP a ()
assertGP b = unless b $ fail ""

readBinaryPS :: PackedString -> Maybe (Patch,PackedString)
readBinaryPS = parseGP $ do
  bin <- GenP mylexPS
  assertGP $ bin == packString "binary"
  fi <- GenP mylexPS
  _ <- GenP mylexPS
  skipGP dropWhitePS
  old <- GenP $ lines_starting_withPS '*'
  _ <- GenP mylexPS
  skipGP dropWhitePS
  new <- GenP $ lines_starting_withPS '*'
  return $ binary (fn2fp $ ps2fn fi) (fromHex2PS $ concatPS $ readAntiMemo old)
                                     (fromHex2PS $ concatPS $ readAntiMemo new)
\end{code}

\begin{code}
readAddFilePS :: PackedString -> Maybe (Patch,PackedString)
readAddFilePS s =
    case mylexPS s of
    Just (_,s') -> case mylexPS s' of
                   Just (f,s'') -> Just (FP (ps2fn f) AddFile, s'')
                   Nothing -> impossible
    Nothing -> impossible
\end{code}

\begin{code}
readRmFilePS :: PackedString -> Maybe (Patch,PackedString)
readRmFilePS s =
    case mylexPS s of
    Just (_,s') -> case mylexPS s' of
                   Just (f,s'') -> Just (FP (ps2fn f) RmFile, s'')
                   Nothing -> impossible
    Nothing -> impossible
\end{code}

\begin{code}
readMovePS :: PackedString -> Maybe (Patch,PackedString)
readMovePS s =
    case mylexPS s of
    Just (_,s') ->
        case mylexPS s' of
        Just (d,s'') ->
            case mylexPS s'' of
            Just (d',s''') -> Just (Move (ps2fn d) (ps2fn d'), s''')
            Nothing -> impossible
        Nothing -> impossible
    Nothing -> impossible
\end{code}

\begin{code}
readChangePrefPS :: PackedString -> Maybe (Patch,PackedString)
readChangePrefPS s =
    case mylexPS s of
    Just (_,s') ->
        case mylexPS s' of
        Just (p,s'') ->
            case breakOnPS '\n' $ tailPS $ dropWhilePS (==' ') s'' of
            (f,s''') ->
                case breakOnPS '\n' $ tailPS s''' of
                (t,s4) -> Just (ChangePref (u p) (u f) (u t), tailPS s4)
                where u = unpackPS
        Nothing -> impossible
    Nothing -> impossible
\end{code}

\begin{code}
readAddDirPS :: PackedString -> Maybe (Patch,PackedString)
readAddDirPS s =
    case mylexPS s of
    Just (_,s') -> case mylexPS s' of
                   Just (f,s'') -> Just (DP (ps2fn f) AddDir, s'')
                   Nothing -> impossible
    Nothing -> impossible
\end{code}

\begin{code}
readRmDirPS :: PackedString -> Maybe (Patch,PackedString)
readRmDirPS s =
    case mylexPS s of
    Just (_,s') -> case mylexPS s' of
                   Just (f,s'') -> Just (DP (ps2fn f) RmDir, s'')
                   Nothing -> impossible
    Nothing -> impossible
\end{code}

\begin{code}
readMergerPS :: Bool -> PackedString -> Maybe (Patch,PackedString)
readMergerPS b s =
    case mylexPS $ snd $ fromJust $ mylexPS s of
    Just (g,s1) ->
        case mylexPS s1 of
        Just (start,s2) ->
            case readPatchPS s2 of
            Just (p1, s3) ->
                case readPatchPS s3 of
                Just (p2, s4) ->
                    case mylexPS s4 of
                    Just (end,s5) ->
                        if (unpackPS start) == "(" && (unpackPS end) == ")"
                        then if b
                             then Just (merger (unpackPS g) p1 p2, s5)
                             else Just (invert $ merger (unpackPS g) p1 p2, s5)
                        else Nothing
                    Nothing -> impossible
                Nothing -> bug "readMergerPS 1"
            Nothing -> bug "readMergerPS 2"
        Nothing -> impossible
    Nothing -> impossible
\end{code}

\begin{code}
readNamedPS :: PackedString -> Maybe (Patch, PackedString)
readNamedPS s =
    case readPatchInfoPS s of
    Nothing -> bug "readNamedPS 1"
    Just (n,s2) ->
        case read_dependsPS s2 of
        Nothing -> bug "readNamedPS 2"
        Just (d, s3) ->
            case readPatchPS s3 of
            Nothing -> Nothing
            Just (p, s4) -> Just (NamedP n d p, s4)
read_dependsPS :: PackedString -> Maybe ([PatchInfo], PackedString)
read_dependsPS s = case mylexPS s of
                   Just (st,s') -> if unpackPS st == "<" then read_pisPS s'
                                   else Just ([],s)
                   Nothing -> impossible
read_pisPS :: PackedString -> Maybe ([PatchInfo], PackedString)
read_pisPS s = case readPatchInfoPS s of
               Just (pi,s') ->
                   case read_pisPS s' of
                   Just (pis,s'') -> Just (pi:pis,s'')
                   Nothing -> impossible
               Nothing -> Just ([],tailPS $ dropWhilePS (/='>') s)
\end{code}

\begin{code}
lines_starting_withPS :: Char -> PackedString
                      -> Maybe (AntiMemo [PackedString],PackedString)
lines_starting_withPS c thes =
    case lsw `antimemoize` thes of
    thetwo -> Just (fst `fmap` thetwo, snd $ readAntiMemo thetwo)
    where lsw s | nullPS s || headPS s /= c = ([], s)
          lsw s = let s' = tailPS s
                  in case breakFirstPS '\n' s' of
                     Just (l, r) -> case lsw r of
                                    (ls, rest) -> (l:ls, rest)
                     Nothing -> ([s'], nilPS)
\end{code}
