% Various utility functions that do not belong anywhere else.

\begin{code}

module DarcsUtils ( catchall, ortryrunning, bug, bugDoc,
                    putStrLnError, putDocLnError,
                    withCurrentDirectory, askUser, stripCr ) where

import System ( ExitCode(..) )
import System.IO ( hFlush, hPutStrLn, stderr, stdout )
import Control.Exception ( bracket )
import Directory ( setCurrentDirectory )
import Workaround ( getCurrentDirectory )
import Monad ( when )
import Printer ( Doc, errorDoc, hPutDocLn, text, ($$) )

#ifdef WIN32
import Monad ( liftM )
#endif

catchall :: IO a -> IO a -> IO a
a `catchall` b = a `catch` (\_ -> b)

ortryrunning :: Monad m => m ExitCode -> m ExitCode -> m ExitCode
a `ortryrunning` b = do ret <- a
                        if ret == ExitSuccess
                           then return ret
                           else b

bug :: String -> a
bug s = error $ "bug in darcs!\n" ++ s ++
                "\nPlease report this to bugs@darcs.net," ++
                "\nIf possible include the output of 'darcs --exact-version'."

bugDoc :: Doc -> a
bugDoc s = errorDoc $ text "bug in darcs!"
                   $$ s
                   $$ text "Please report this to bugs@darcs.net"
                   $$ text "If possible include the output of 'darcs --exact-version'."

putStrLnError :: String -> IO ()
putStrLnError = hPutStrLn stderr

putDocLnError :: Doc -> IO ()
putDocLnError = hPutDocLn stderr

withCurrentDirectory :: FilePath -> IO a -> IO a
withCurrentDirectory name m =
    bracket
        (do cwd <- getCurrentDirectory
            when (name /= "") (setCurrentDirectory name)
            return cwd)
        (\oldwd -> do setCurrentDirectory oldwd)
        (const m)

askUser :: String -> IO String
askUser prompt = do putStr prompt
                    hFlush stdout
#ifndef WIN32
                    getLine
#else
                    liftM stripCr getLine
#endif

stripCr :: String -> String
stripCr ""     = ""
stripCr "\r"   = ""
stripCr (c:cs) = c : stripCr cs

\end{code}
