/*
 * ClutterCairo.
 *
 * An simple Clutter Cairo 'Drawable'.
 *
 * Authored By Matthew Allum  <mallum@openedhand.com>
 *
 * Copyright (C) 2006 OpenedHand
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/**
 * SECTION:clutter-cairo
 * @short_description: Actor for displaying 
 *
 * #ClutterCairo is a #ClutterTexture that displays the contents
 * of a Cairo context.
 *
 * #ClutterCairo will provide a #cairo_t context with the
 * clutter_cairo_create() and clutter_cairo_create_region() functions; as
 * soon as the context is destroyed with cairo_destroy(), the contents will
 * be uploaded into the #ClutterCairo actor.
 */
#include "clutter-cairo.h"
#include <string.h>

G_DEFINE_TYPE (ClutterCairo, clutter_cairo, CLUTTER_TYPE_TEXTURE);

enum
{
  PROP_0,

  PROP_SURFACE_WIDTH,
  PROP_SURFACE_HEIGHT
};

#define CLUTTER_CAIRO_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), CLUTTER_TYPE_CAIRO, ClutterCairoPrivate))

struct _ClutterCairoPrivate
{
  cairo_format_t   format;

  cairo_surface_t *cr_surface;
  guchar          *cr_surface_data;
  gboolean         initialised;
  gint             width;
  gint             height;
  gint             rowstride;
};

typedef struct
{
  gint x;
  gint y;
  guint width;
  guint height;
} ClutterCairoRectangle;

typedef struct
{
  ClutterCairo *cairo;
  ClutterCairoRectangle rect;
} ClutterCairoContext;

static const cairo_user_data_key_t clutter_cairo_surface_key;
static const cairo_user_data_key_t clutter_cairo_context_key;

static void
clutter_cairo_surface_destroy (void *data)
{
  ClutterCairo *cairo = data;

  cairo->priv->cr_surface = NULL;
}

static void
clutter_cairo_set_property (GObject      *object,
                            guint         prop_id,
                            const GValue *value,
                            GParamSpec   *pspec)
{
  ClutterCairo        *cairo;
  ClutterCairoPrivate *priv;

  cairo = CLUTTER_CAIRO(object);
  priv = cairo->priv;

  switch (prop_id) 
    {
    case PROP_SURFACE_WIDTH:
      priv->width = g_value_get_int (value);
      break;
    case PROP_SURFACE_HEIGHT:
      priv->height = g_value_get_int (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
clutter_cairo_get_property (GObject    *object,
                            guint       prop_id,
                            GValue     *value,
                            GParamSpec *pspec)
{
  ClutterCairoPrivate *priv = CLUTTER_CAIRO (object)->priv;

  switch (prop_id) 
    {
    case PROP_SURFACE_WIDTH:
      g_value_set_int (value, priv->width);
      break;
    case PROP_SURFACE_HEIGHT:
      g_value_set_int (value, priv->height);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    } 
}

static void 
clutter_cairo_finalize (GObject *object)
{
  ClutterCairoPrivate *priv = CLUTTER_CAIRO (object)->priv;

  if (priv->cr_surface)
    {
      cairo_surface_t *surface = priv->cr_surface;

      cairo_surface_finish (priv->cr_surface);
      cairo_surface_set_user_data (priv->cr_surface,
                                   &clutter_cairo_surface_key,
                                   NULL, NULL);
      cairo_surface_destroy (surface);

      priv->cr_surface = NULL;
    }

  if (priv->cr_surface_data)
    {
      g_free (priv->cr_surface_data);
      priv->cr_surface_data = NULL;
    }
  
  G_OBJECT_CLASS (clutter_cairo_parent_class)->finalize (object);
}

static GObject*
clutter_cairo_constructor (GType                  gtype,
                           guint                  n_properties,
                           GObjectConstructParam *properties)
{
  GObjectClass        *parent_class;
  GObject             *obj;
  ClutterCairo        *cairo;
  ClutterCairoPrivate *priv;

  parent_class = G_OBJECT_CLASS (clutter_cairo_parent_class);
  obj = parent_class->constructor (gtype, n_properties, properties);

  /* Now all of the object properties are set */
  cairo = CLUTTER_CAIRO (obj);
  priv = cairo->priv;

  if (!priv->width || !priv->height)
    {
      g_warning ("Unable to create the Cairo surface: invalid size (%dx%d)",
                 priv->width,
                 priv->height);
      return obj;
    }

#if CAIRO_VERSION > 106000
  priv->rowstride = cairo_format_stride_for_width (priv->format, priv->width);
#else
  /* poor man's version of cairo_format_stride_for_width() */
  switch (priv->format)
    {
    case CAIRO_FORMAT_ARGB32:
    case CAIRO_FORMAT_RGB24:
      priv->rowstride = priv->width * 4;
      break;
    case CAIRO_FORMAT_A8:
    case CAIRO_FORMAT_A1: 
      priv->rowstride = priv->width;
      break;
    default:
      g_assert_not_reached ();
      break;
    }
#endif

  priv->cr_surface_data = g_malloc0 (priv->height * priv->rowstride);
  priv->cr_surface = 
    cairo_image_surface_create_for_data (priv->cr_surface_data,
                                         priv->format,
                                         priv->width,
                                         priv->height,
                                         priv->rowstride);
  
  cairo_surface_set_user_data (priv->cr_surface, &clutter_cairo_surface_key,
                               cairo, clutter_cairo_surface_destroy);

  return obj;
}

static void
clutter_cairo_get_preferred_width (ClutterActor *actor,
                                   ClutterUnit   for_height,
                                   ClutterUnit  *min_width,
                                   ClutterUnit  *natural_width)
{
  ClutterCairoPrivate *priv = CLUTTER_CAIRO (actor)->priv;

  if (min_width)
    *min_width = 0;

  if (natural_width)
    *natural_width = CLUTTER_UNITS_FROM_DEVICE (priv->width);
}

static void
clutter_cairo_get_preferred_height (ClutterActor *actor,
                                    ClutterUnit   for_width,
                                    ClutterUnit  *min_height,
                                    ClutterUnit  *natural_height)
{
  ClutterCairoPrivate *priv = CLUTTER_CAIRO (actor)->priv;

  if (min_height)
    *min_height = 0;

  if (natural_height)
    *natural_height = CLUTTER_UNITS_FROM_DEVICE (priv->height);
}

static void
clutter_cairo_class_init (ClutterCairoClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *actor_class = CLUTTER_ACTOR_CLASS (klass);

  gobject_class->finalize     = clutter_cairo_finalize;
  gobject_class->set_property = clutter_cairo_set_property;
  gobject_class->get_property = clutter_cairo_get_property;
  gobject_class->constructor  = clutter_cairo_constructor;

  actor_class->get_preferred_width = clutter_cairo_get_preferred_width;
  actor_class->get_preferred_height = clutter_cairo_get_preferred_height;

  g_type_class_add_private (gobject_class, sizeof (ClutterCairoPrivate));

#define PARAM_FLAGS (G_PARAM_CONSTRUCT_ONLY |                   \
                     G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |\
                     G_PARAM_STATIC_BLURB |                     \
                     G_PARAM_READABLE | G_PARAM_WRITABLE)

  g_object_class_install_property (gobject_class,
                                   PROP_SURFACE_WIDTH,
                                   g_param_spec_int ("surface-width",
                                                     "Surface-Width",
                                                     "Surface Width",
                                                     0, G_MAXINT,
                                                     0,
                                                     PARAM_FLAGS));
  g_object_class_install_property (gobject_class,
                                   PROP_SURFACE_HEIGHT,
                                   g_param_spec_int ("surface-height",
                                                     "Surface-Height",
                                                     "Surface Height",
                                                     0, G_MAXINT,
                                                     0,
                                                     PARAM_FLAGS));
#undef PARAM_FLAGS
}

static void
clutter_cairo_init (ClutterCairo *self)
{
  ClutterCairoPrivate *priv;

  self->priv = priv = CLUTTER_CAIRO_GET_PRIVATE (self);

  /* XXX - we are hardcoding the format; it would be good to have
   * a :surface-format construct-only property for creating
   * textures with a different format
   */
  priv->format = CAIRO_FORMAT_ARGB32;
}

/**
 * clutter_cairo_new
 * @width: clutter cairo surface width
 * @height: clutter cairo surface height
 *
 * Creates a new #ClutterCairo texture.
 *
 * Return value: a #ClutterCairo texture
 */
ClutterActor*
clutter_cairo_new (guint width,
                   guint height)
{
  return g_object_new (CLUTTER_TYPE_CAIRO, 
                       "surface-width", width,
                       "surface-height", height,
                       NULL);
}

static void
clutter_cairo_context_destroy (void *data)
{
  ClutterCairoContext *ctxt = data;
  ClutterCairo        *cairo = ctxt->cairo;
  ClutterCairoPrivate *priv;
  GError              *error = NULL;

  gint    cairo_width, cairo_height, cairo_rowstride;
  gint    surface_width, surface_height;
  guchar *pixbuf_data, *dst, *cairo_data;
  guint  *src, pixbuf_rowstride;
  gint    x, y;

  priv = CLUTTER_CAIRO_GET_PRIVATE (cairo);

  if (!priv->cr_surface)
    return;

  surface_width = cairo_image_surface_get_width (priv->cr_surface);
  surface_height = cairo_image_surface_get_height (priv->cr_surface);

  cairo_width = MIN (ctxt->rect.width, surface_width);
  cairo_height = MIN (ctxt->rect.height, surface_height);

  if (!cairo_width || !cairo_height)
    {
      g_free (ctxt);
      return;
    }

  cairo_rowstride  = priv->rowstride;
  cairo_data       = priv->cr_surface_data;
  pixbuf_data      = g_malloc (cairo_width * cairo_height * 4);
  pixbuf_rowstride = cairo_width * 4;

  /* BAH BAH BAH ! un-pre-multiply alpha... 
   * FIXME: Need to figure out if GL has a premult texture 
   *        format... or go back to battling glitz
  */
  for (y = 0; y < cairo_height; y++)
    {
      src = (unsigned int *) (cairo_data 
			      + ((y + ctxt->rect.y) * cairo_rowstride)
			      + (ctxt->rect.x * 4));
      dst = pixbuf_data + y * pixbuf_rowstride;

      for (x = 0; x < cairo_width; x++) 
        {
          guchar alpha = (*src >> 24) & 0xff;

          if (alpha == 0)
            dst[0] = dst[1] = dst[2] = dst[3] = alpha;
          else
            {
#if G_BYTE_ORDER == G_LITTLE_ENDIAN
              dst[0] = (((*src >> 16) & 0xff) * 255 ) / alpha;
              dst[1] = (((*src >> 8) & 0xff) * 255 ) / alpha; 
              dst[2] = (((*src >> 0) & 0xff) * 255 ) / alpha;
              dst[3] = alpha;
#elif G_BYTE_ORDER == G_BIG_ENDIAN
              dst[0] = alpha;
              dst[1] = (((*src >> 0) & 0xff) * 255 ) / alpha;
              dst[2] = (((*src >> 8) & 0xff) * 255 ) / alpha; 
              dst[3] = (((*src >> 16) & 0xff) * 255 ) / alpha;
#else /* !G_LITTLE_ENDIAN && !G_BIG_ENDIAN */
#error unknown ENDIAN type
#endif /* !G_LITTLE_ENDIAN && !G_BIG_ENDIAN */
            }
          dst += 4;
          src++;
        }
    }

  if (ctxt->rect.x == 0 &&
      ctxt->rect.y == 0 &&
      cairo_width == priv->width &&
      cairo_height == priv->height)
    {
      /* Dealing with the whole area */
      clutter_texture_set_from_rgb_data (CLUTTER_TEXTURE (cairo),
                                         pixbuf_data,
                                         TRUE,
                                         cairo_width, cairo_height,
                                         pixbuf_rowstride,
                                         4, 0,
                                         &error);
    }
  else
    {
      if (!priv->initialised)
        {
          guchar *init_pixbuf_data = g_malloc (priv->width * priv->height * 4);
          clutter_texture_set_from_rgb_data (CLUTTER_TEXTURE (cairo),
                                             init_pixbuf_data,
                                             TRUE, priv->width, priv->height,
                                             priv->width * 4,
                                             4, 0, &error);
          g_free (init_pixbuf_data);
          priv->initialised = TRUE;
        }
      
      clutter_texture_set_area_from_rgb_data (CLUTTER_TEXTURE (cairo),
                                              pixbuf_data,
                                              TRUE,
                                              ctxt->rect.x,
                                              ctxt->rect.y,
                                              cairo_width, cairo_height,
                                              pixbuf_rowstride,
                                              4, 0,
                                              &error);
    }

  g_free (pixbuf_data); 
  g_free (ctxt);
  
  if (CLUTTER_ACTOR_IS_VISIBLE (cairo))
    clutter_actor_queue_redraw (CLUTTER_ACTOR (cairo));
}

static void
intersect_rectangles (ClutterCairoRectangle *a,
		      ClutterCairoRectangle *b,
		      ClutterCairoRectangle *inter)
{
  gint dest_x, dest_y;
  gint dest_width, dest_height;

  dest_x = MAX (a->x, b->x);
  dest_y = MAX (a->y, b->y);
  dest_width = MIN (a->x + a->width, b->x + b->width) - dest_x;
  dest_height = MIN (a->y + a->height, b->y + b->height) - dest_y;

  if (dest_width > 0 && dest_height > 0)
    {
      inter->x = dest_x;
      inter->y = dest_y;
      inter->width = dest_width;
      inter->height = dest_height;
    }
  else
    {
      inter->x = 0;
      inter->y = 0;
      inter->width = 0;
      inter->height = 0;
    }
}

/**
 * clutter_cairo_create_region
 * @cairo: A #ClutterCairo texture.
 * @x_offset:
 * @y_offset:
 * @width:
 * @height:
 *
 * Creates a new cairo context that will update the region defined by
 * @x_offset,@y_offset,@width,@height.
 *
 * Return value: A newly created cairo context.
 */
cairo_t *
clutter_cairo_create_region (ClutterCairo *cairo,
                             gint          x_offset,
			     gint          y_offset,
			     guint         width,
			     guint         height)
{
  ClutterCairoContext *ctxt;
  ClutterCairoRectangle region, area, inter;
  cairo_t *cr;

  g_return_val_if_fail (CLUTTER_IS_CAIRO (cairo), NULL);

  ctxt = g_new0 (ClutterCairoContext, 1);
  ctxt->cairo = cairo;

  region.x = x_offset;
  region.y = y_offset;
  region.width = width;
  region.height = height;

  area.x = 0;
  area.y = 0;
  area.width = cairo->priv->width;
  area.height = cairo->priv->height;

  /* Limit the region to the visible rectangle */
  intersect_rectangles (&area, &region, &inter);

  ctxt->rect.x = inter.x;
  ctxt->rect.y = inter.y;
  ctxt->rect.width = inter.width;
  ctxt->rect.height = inter.height;

  cr = cairo_create (cairo->priv->cr_surface);
  cairo_set_user_data (cr, &clutter_cairo_context_key, 
		       ctxt, clutter_cairo_context_destroy);

  return cr;
}

/**
 * clutter_cairo_create
 * @cairo:  A #ClutterCairo texture.
 *
 * Creates a new cairo context #ClutterCairo texture. 
  *
 * Return value: a newly created cairo context. Free with cairo_destroy() 
 * when you are done drawing.  
 */
cairo_t *
clutter_cairo_create (ClutterCairo *cairo)
{
  cairo_t *cr;

  g_return_val_if_fail (CLUTTER_IS_CAIRO (cairo), NULL);

  cr = clutter_cairo_create_region (cairo,
                                    0, 0, 
				    cairo->priv->width,
                                    cairo->priv->height);
  return cr;
}

/**
 * clutter_cairo_set_source_color:
 * @cr: a #cairo_t
 * @color: a #ClutterColor
 *
 * Sets @color as the source color for the cairo context.
 */
void
clutter_cairo_set_source_color (cairo_t            *cr,
                                const ClutterColor *color)
{
  g_return_if_fail (cr != NULL);
  g_return_if_fail (color != NULL);

  if (color->alpha == 0xff)
    cairo_set_source_rgb (cr,
                          color->red / 255.0,
                          color->green / 255.0,
                          color->blue / 255.0);
  else
    cairo_set_source_rgba (cr,
                           color->red / 255.0,
                           color->green / 255.0,
                           color->blue / 255.0,
                           color->alpha / 255.0);
}

/**
 * clutter_cairo_surface_resize:
 * @cairo: A #ClutterCairo texture
 * @width: The new width of the texture
 * @height: The new height of the texture
 *
 * Resizes the cairo surface to @width and @height.
 */
void
clutter_cairo_surface_resize (ClutterCairo *cairo,
			      guint         width,
			      guint         height)
{
  ClutterCairoPrivate *priv;
  gboolean notify_width, notify_height;

  g_return_if_fail (CLUTTER_IS_CAIRO (cairo));
  
  priv = cairo->priv;

  if (width == priv->width && height == priv->height)
    return;

  if (priv->cr_surface)
    {
      cairo_surface_t *surface = priv->cr_surface;
      
      cairo_surface_finish (surface);
      cairo_surface_set_user_data (surface, &clutter_cairo_surface_key, 
				   NULL, NULL);
      cairo_surface_destroy (surface);

      priv->cr_surface = NULL;
    }

  if (priv->cr_surface_data)
    {
      g_free (priv->cr_surface_data);
      priv->cr_surface_data = NULL;
    }

  if (priv->width != width)
    {
      priv->width = width;
      notify_width = TRUE;
    }
  else
    notify_width = FALSE;
  
  if (priv->height != height)
    {
      priv->height = height;
      notify_height = TRUE;
    }
  else
    notify_height = FALSE;

#if CAIRO_VERSION > 106000
  priv->rowstride = cairo_format_stride_for_width (priv->format, priv->width);
#else
  /* poor man's version of cairo_format_stride_for_width() */
  switch (priv->format)
    {
    case CAIRO_FORMAT_ARGB32:
    case CAIRO_FORMAT_RGB24:
      priv->rowstride = priv->width * 4;
      break;
    case CAIRO_FORMAT_A8:
    case CAIRO_FORMAT_A1: 
      priv->rowstride = priv->width;
      break;
    default:
      g_assert_not_reached ();
      break;
    }
#endif

  priv->cr_surface_data = g_malloc0 (priv->height * priv->rowstride);
  priv->cr_surface =
    cairo_image_surface_create_for_data (priv->cr_surface_data,
                                         priv->format,
                                         priv->width, priv->height,
                                         priv->rowstride);

  cairo_surface_set_user_data (priv->cr_surface, &clutter_cairo_surface_key,
			       cairo, clutter_cairo_surface_destroy);
  priv->initialised = FALSE;
  
  if (notify_width)
    g_object_notify (G_OBJECT (cairo), "surface-width");

  if (notify_height)
    g_object_notify (G_OBJECT (cairo), "surface-height");
}

