/*
	$Id: provider_ppm.cpp,v 1.1.1.1 2000/04/09 12:18:02 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------

	File purpose:
		PPM SurfaceProvider
*/

//  It should be 100% ASCII PPM and RAW/BINARY PPM competable.

#include "Core/precomp.h"
#include "API/Core/Display/pixelformat.h"
#include "API/Core/Display/palette.h"
#include "API/Core/IOData/inputsource.h"
#include "API/Core/IOData/inputsource_provider.h"
#include "API/Core/SurfaceProviders/provider_ppm.h"
#include "API/Core/System/error.h" 
#include "API/Core/System/cl_assert.h"
#include "API/Core/Display/res_surface.h"
#include "API/Core/Resources/resourceoptions.h"


CL_Surface *CL_PPMProvider::create(
	std::string handle, 
	CL_InputSourceProvider *provider, 
	bool transparent,
	unsigned char trans_red, 
	unsigned char trans_green, 
	unsigned char trans_blue)
{
	return CL_Surface::create(new CL_PPMProvider(
		handle, 
		provider, 
		transparent,
		trans_red,
		trans_green,
		trans_blue), true);
}

CL_PPMProvider::CL_PPMProvider(
	std::string _name, 
	CL_InputSourceProvider *_provider, 
	bool _transparent,
	unsigned char _trans_red, 
	unsigned char _trans_green, 
	unsigned char _trans_blue)
{
	provider = _provider != NULL ? _provider->clone() : 
								CL_InputSourceProvider::create_file_provider(".");
	width = height = 0;

	buffer = NULL;
	lock_refs = 0;
	transparent = _transparent;

	trans_red = ~0; 
	trans_green = ~0; 
	trans_blue = ~0;


	if (transparent)
	{
		trans_red = _trans_red; 
		trans_green = _trans_green; 
		trans_blue = _trans_blue;
	}
	

	load_lock=false;
	
	name = _name;
	//palette = NULL;
	//image = NULL;
	//pixelformat = PAL8;
}

CL_PPMProvider::~CL_PPMProvider()
{
	delete[] buffer;
}

unsigned int CL_PPMProvider::get_pitch() const
{
	return 4*get_width();
}

unsigned int CL_PPMProvider::get_width() const
{
	return width;
}

unsigned int CL_PPMProvider::get_height() const
{
	return height;
}

unsigned int CL_PPMProvider::get_num_frames() const
{
	return 1;
}
/*
EPixelFormat CL_PPMProvider::get_pixel_format() const
{
	return RGBA8888;
}
*/

unsigned int CL_PPMProvider::get_depth() const
{
	return 32;
}

unsigned int CL_PPMProvider::get_red_mask() const
{
	return 0xff000000;
}

unsigned int CL_PPMProvider::get_green_mask() const
{
	return 0x00ff0000;
}

unsigned int CL_PPMProvider::get_blue_mask() const
{
	return 0x0000ff00;
}

unsigned int CL_PPMProvider::get_alpha_mask() const
{
	return 0x000000ff;
}

bool CL_PPMProvider::is_indexed() const
{
	return false;
}

CL_Palette *CL_PPMProvider::get_palette() const
{
	return NULL; // no palette used by the PPM surface provider.
}

bool CL_PPMProvider::uses_src_colorkey() const
{
	return false;
}

unsigned int CL_PPMProvider::get_src_colorkey() const
{
	unsigned int color=0;

	if(transparent)
		color = ( ((unsigned int)trans_red<<16)|
				  ((unsigned int)trans_green<<8)|
				  ((unsigned int)trans_blue) );

	return color; 
}

void *CL_PPMProvider::get_data() const
{
	return buffer;
}

void CL_PPMProvider::lock()
{
	if (buffer != NULL) return;
	
	cl_assert(provider != NULL);
	CL_InputSource *datafile = provider->open_source(name.c_str());
	cl_assert(datafile != NULL);



///////////// - Read PPM header - ///////////

	// read the magic signatur.
	bool is_PPM_P3 = false;
	bool is_PPM_P6 = false;

	//is_PPM_P3,is_PPM_P6=false;
	char magic[2];
	
	magic[0] = datafile->read_char8();
	magic[1] = datafile->read_char8();
	
	if (strncmp(magic, "P6", 2) == 0)//P6 is the PAW PPM format, which is not yet supported
		is_PPM_P6=true;
		
	else if (strncmp(magic, "P3", 2) == 0)//P3 is the ASCII PPM format,
		is_PPM_P3=true;
	
	else
		std::cout << name.c_str() << " is not a PPM image." << std::endl;
		
	width = my_get_value(datafile);//read width
	height = my_get_value(datafile);//read height
	max_val = my_get_value(datafile);//read max_val

/*
	std::cout << std::endl << name.c_str() << " is a " << magic << " PPM file."
		<< std::endl << "The width is: " << width
		<< std::endl << "The height is: " << height 
		<< std::endl << "The max_val is: " << max_val
		<< std::endl;
*/

///////////// - Read PPM image - ///////////
	
	if (lock_refs == 0)
	{
		buffer = new unsigned char[get_pitch()*get_height()];
		
		for (int y=0; y<height; y++)
		{
			for (int x=0; x<width; x++)
			{
				//read P3 - ASCII PPM
				if (is_PPM_P3)
				{
					unsigned int R=0,G=0,B=0,A=255;

					R = my_get_value(datafile);
					G = my_get_value(datafile);
					B = my_get_value(datafile);

					if (R==trans_red && G==trans_green && B==trans_blue)
						A=0;
					else
						A=255;

					if(max_val==255)// If the max_val color component is 255, just read and put! :-)
					{
						buffer[(x+y*width)*4 + 0] = A; // alpha component of pixel
						buffer[(x+y*width)*4 + 3] = R; // red component
						buffer[(x+y*width)*4 + 2] = G; // green component
						buffer[(x+y*width)*4 + 1] = B; // blue component
					}
					else// If the max_val color component is not 255, we have to do some calculation! :-(
					{
						buffer[(x+y*width)*4 + 0] = A; // alpha component of pixel
						buffer[(x+y*width)*4 + 3] = ( R * 255 /max_val); // red component
						buffer[(x+y*width)*4 + 2] = ( G * 255 /max_val); // green component
						buffer[(x+y*width)*4 + 1] = ( B * 255 /max_val); // blue component
					}
				}//end read P3 - ASCII PPM


				//read P6 - RAW/BINARY PPM
				else if (is_PPM_P6)
				{
					unsigned char input;

					unsigned int R=0,G=0,B=0,A=255;

					input = datafile->read_char8();//read from file
					R = ( ((unsigned int)input) );//put red component from input

					input = datafile->read_char8();//read from file
					G = ( ((unsigned int)input) );//put green component from input

					input = datafile->read_char8();//read from file
					B = ( ((unsigned int)input) );//put blue component from input

					if (R==trans_red && G==trans_green && B==trans_blue)
						A=0;
					else
						A=255;
						
			
					if(max_val==255)// If the max_val color component is 255, just read and put! :-)
					{
						buffer[(x+y*width)*4 + 0] = A; // alpha component of pixel
						buffer[(x+y*width)*4 + 3] = R; // red component
						buffer[(x+y*width)*4 + 2] = G; // green component
						buffer[(x+y*width)*4 + 1] = B; // blue component
					}
					else// If the max_val color component is not 255, we have to do some calculation! :-(
					{
						buffer[(x+y*width)*4 + 0] = A; // alpha component of pixel
						buffer[(x+y*width)*4 + 3] = ( R * 255 /max_val); // red component
						buffer[(x+y*width)*4 + 2] = ( G * 255 /max_val); // green component
						buffer[(x+y*width)*4 + 1] = ( B * 255 /max_val); // blue component
					}
				}//end read P6 - RAW PPM
			}
		}
	}
	
	lock_refs++;
}

void CL_PPMProvider::unlock()
{
	lock_refs--;

	if (lock_refs == 0)
	{
		delete[] buffer;
		buffer = NULL;
	}
}

int CL_PPMProvider::my_get_value(CL_InputSource *datafile)
{
	int value = 0;
	unsigned char ascii_letter;

	// Skip whitespace
	do
	{
		ascii_letter = datafile->read_char8();
				
		// Skip comments as whitespace
		if ( ascii_letter == '#' )/* Comment is '#' to end of line */
			do
			{
				ascii_letter = datafile->read_char8();//read out the line;
				
			} while ( (ascii_letter != '\r') && (ascii_letter != '\n') );//stop when end a new line.
		
	} while ( isspace(ascii_letter) );

	//read a value
	do 
	{
		value *= 10;
		value += ascii_letter-'0';

		ascii_letter = datafile->read_char8();
				
	} while ( isdigit(ascii_letter) );
	
//	std::cout << std::endl << "value = " << value;


	return value;

}




///////////////// - The PPM file format DESCRIPTION - ///////////////////
/*

  ppm(5)                    FILE FORMATS                     ppm(5)



NAME
     ppm - portable pixmap file format

DESCRIPTION
     The portable pixmap format is a  lowest  common  denominator
     color image file format.  The definition is as follows:

     - A "magic number" for identifying the  file  type.   A  ppm
       file's magic number is the two characters "P3".

     - Whitespace (blanks, TABs, CRs, LFs).

     - A width, formatted as ASCII characters in decimal.

     - Whitespace.

     - A height, again in ASCII decimal.

     - Whitespace.

     - The maximum color-component value, again in ASCII decimal.

     - Whitespace.

     - Width * height pixels, each  three  ASCII  decimal  values
       between 0 and the specified maximum value, starting at the
       top-left  corner  of  the  pixmap,  proceeding  in  normal
       English  reading  order.   The three values for each pixel
       represent red, green, and blue, respectively; a value of 0
       means  that color is off, and the maximum value means that
       color is maxxed out.

     - Characters from a "#" to the next end-of-line are  ignored
       (comments).

     - No line should be longer than 70 characters.

     Here is an example of a small pixmap in this format:
     P3
     # feep.ppm
     4 4
     15
      0  0  0    0  0  0    0  0  0   15  0 15
      0  0  0    0 15  7    0  0  0    0  0  0
      0  0  0    0  0  0    0 15  7    0  0  0
     15  0 15    0  0  0    0  0  0    0  0  0

     Programs that read this format should be as lenient as  pos-
     sible, accepting anything that looks remotely like a pixmap.

     There is also a variant on the format, available by  setting
     the  RAWBITS  option  at  compile  time.   This  variant  is
     different in the following ways:

     - The "magic number" is "P6" instead of "P3".

     - The pixel values are stored as  plain  bytes,  instead  of
       ASCII decimal.

     - Whitespace is not allowed in the pixels area, and  only  a
       single  character  of  whitespace (typically a newline) is
       allowed after the maxval.

     - The files are smaller and many times faster  to  read  and
       write.

     Note that this raw format can only be used for maxvals less
     than or equal to 255.
*/