/*
	$Id: provider_fli.cpp,v 1.1.1.1 2000/04/09 12:18:02 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------

	File purpose:
		Fli/Flc reader.

*/

#include "Core/precomp.h"
#include <stdio.h>
#include "API/Core/SurfaceProviders/provider_fli.h"
#include "API/Core/Display/palette.h"
#include "API/Core/Display/res_surface.h"
#include "API/Core/Resources/resourceoptions.h"
#include "API/Core/System/clanstring.h"

CL_FLIProvider::CL_FLIProvider(const char *fli_id, int _start_frame)
{
	start_frame = _start_frame;

	FILE *fli_file = fopen(fli_id, "rb");
	
	fseek(fli_file, 0, SEEK_END);
	int file_size = ftell(fli_file);

	fli_start = fli_buffer = new unsigned char[file_size];
	if (fli_buffer == NULL)
	{
		return;		// throw 42;
	}

	fseek(fli_file, 0, SEEK_SET);
	fread(fli_buffer, 1, file_size, fli_file);
	fclose(fli_file);

	// Read the fli header, and search past the expansion part
	//
	cur_frame = start_frame;

	header = (SFLIHeader *) fli_buffer;
//	cout << "Number of frames: " << header->num_frames << endl;
	fli_buffer += 128;

	output_buffer = new unsigned char[header->width*header->height];

	unsigned char *tmp = new unsigned char[768];
	memset(tmp, 0, 768);
	pal = new CL_Palette(tmp);
	delete[] tmp;

	// If this file isn't a .FLI or .FLC abort
	//
	if (header->magic != 0xAF11 && header->magic != 0xAF12)
	{
		delete[] fli_start;
		fli_start = NULL;
		return;
	}

	// Examine if this file is a .flc - in that case skip the first frame
	//
	SFLIFrameHeader *frame_header = (SFLIFrameHeader *) fli_buffer;
	if (frame_header->magic == 0x00A1)
	{
		fli_buffer += frame_header->size;
	}

/*	// If a specific frame has been requested, skip the previous frames
	//
	for (int i=0;i<start_frame;i++)
	{
		frame_header = (SFLIFrameHeader *) fli_buffer;
		cout << "Skipping frame: " << frame_header->size << endl;
		fli_buffer += frame_header->size;
	}

	// If a specific frame has been requested, make sure we don't save
	// the entire animation in memory
	//
	if (start_frame > 0)
	{
		frame_header = (SFLIFrameHeader *) fli_buffer;
		unsigned char *temp = new unsigned char[frame_header->size];
		memcpy(temp, fli_buffer, frame_header->size);
		delete[] fli_start;
		fli_buffer = fli_start = temp;
	}
*/
	loop_pos = fli_buffer;
}

CL_FLIProvider::~CL_FLIProvider()
{
	delete[] fli_start;
	delete[] output_buffer;
	delete pal;
}

unsigned int CL_FLIProvider::get_width() const
{
	return header->width;
}

unsigned int CL_FLIProvider::get_height() const
{
	return header->height;
}

unsigned int CL_FLIProvider::get_num_frames() const
{
	return 1;
}
/*
EPixelFormat CL_FLIProvider::get_pixel_format() const
{
	return PAL8;
}
*/

unsigned int CL_FLIProvider::get_pitch() const
{
	return get_width();
}

unsigned int CL_FLIProvider::get_depth() const
{
	return 8;
}

unsigned int CL_FLIProvider::get_red_mask() const
{
	return 0;
}

unsigned int CL_FLIProvider::get_green_mask() const
{
	return 0;
}

unsigned int CL_FLIProvider::get_blue_mask() const
{
	return 0;
}

unsigned int CL_FLIProvider::get_alpha_mask() const
{
	return 0;
}

bool CL_FLIProvider::is_indexed() const
{
	return true;
}

CL_Palette *CL_FLIProvider::get_palette() const
{
	return pal;
}

bool CL_FLIProvider::uses_src_colorkey() const
{
	return false;
}

unsigned int CL_FLIProvider::get_src_colorkey() const
{
	return 0;
}

void *CL_FLIProvider::get_data() const
{
	return output_buffer;
}

void CL_FLIProvider::perform_lock()
{
/*	if (cur_frame >= header->num_frames || start_frame != 0)
	{
		fli_buffer = loop_pos;
		cur_frame = start_frame;
	}
*/
/*	unsigned short fd = 0xF1FA;
	cout << "WATCH FOR: " << ((fd)&0xff) << "-" << ((fd>>8)&0xff) << endl;
	cout << "DUMP: ";
	for (unsigned int i=0;i<sizeof(SFLIFrameHeader);i++)
	{
		cout << int(*(fli_buffer+i)) << "-";
	}
	cout << endl;*/

	SFLIFrameHeader *frame_header = (SFLIFrameHeader *) fli_buffer;
	fli_buffer += 16;
	cur_frame++;
//	cout << "Frame size: " << frame_header->size << " (chunks: " << frame_header->num_chunks << ")" << endl;

	if (frame_header->magic != 0xF1FA &&
		frame_header->magic != 0xF1)
	{
		return;
	}

	for (int cur_chunk=0;cur_chunk<frame_header->num_chunks;cur_chunk++)
	{
//		unsigned char *start_pos = fli_buffer;

		SFLIChunkHeader *chunk_header = (SFLIChunkHeader *) fli_buffer;
//		cout << "Size of chunk: " << chunk_header->size << "(type = " << chunk_header->type << ")" << endl;
		fli_buffer += 6;
		switch (chunk_header->type)
		{
		case CHUNK_FLI_COLOR:
			read_color(2);
			break;
		
		case CHUNK_FLI_LC:
			read_lc();
			break;
		
		case CHUNK_FLI_BLACK:
			read_black();
			break;
		
		case CHUNK_FLI_BRUN:
			read_brun();
			break;

		case CHUNK_FLI_COPY:
			read_copy();
			break;

		case CHUNK_FLC_DELTA:
			read_delta();
			break;

		case CHUNK_FLC_256COLOR:
			read_color(0);
			break;
		
		case CHUNK_FLC_MINI:
			fli_buffer += chunk_header->size-6;
			break;
		}
//		cout << "chunk read" << endl;
//		unsigned char *end_pos = fli_buffer;
//		fli_buffer = start_pos + chunk_header->size;
	}
//	cout << "Lock complete" << endl;
}

void CL_FLIProvider::perform_unlock()
{
}

/***************************************************************
				 FLI specific decoding functions
***************************************************************/

void CL_FLIProvider::read_color(int shift_amount)
{
	unsigned short num_packets = *((unsigned short *)	 fli_buffer);
	fli_buffer += 2;

	int cur_col = 0;
	for (int i=0;i<num_packets;i++)
	{
		unsigned char skip_count = *fli_buffer++;
		cur_col += skip_count;
		unsigned char change_count = *fli_buffer++;
		if (!change_count) change_count = 255;
		for (int c=0;c<=change_count;c++)
		{
			unsigned char r, g, b;
			pal->palette[(cur_col+c)*3] = r = *fli_buffer++ << shift_amount;
			pal->palette[(cur_col+c)*3+1] = g = *fli_buffer++ << shift_amount; 
			pal->palette[(cur_col+c)*3+2] = b = *fli_buffer++ << shift_amount;
		}
		cur_col += change_count;
	}
}

void CL_FLIProvider::read_lc()
{
	unsigned short cur_line = *((unsigned short *) fli_buffer);
	fli_buffer += 2;
//	cout << "Line: " << cur_line << endl;

	unsigned short change_line = *((unsigned short *) fli_buffer);
	fli_buffer += 2;
//	cout << "Num_lines: " << change_line << endl;

	int line_count = cur_line * header->width;
	for (int l=cur_line;l<cur_line+change_line;l++)
	{
//		cout << "Line number: " << l << endl;
		unsigned char num_packets = *fli_buffer++;
//		cout << "Number of packets: " << int(num_packets) << endl;
		int cur_x = 0;
		for (int n=0;n<num_packets;n++)
		{
			unsigned char skip_count = *fli_buffer++;
//			cout << "Skip_count: " << int(skip_count) << endl;
			cur_x += skip_count;

			char size_count = *fli_buffer++;
//			cout << "S=" << int(size_count);
			if (size_count > 0)
			{
//				cout << ">";
				memcpy(output_buffer+line_count+cur_x, fli_buffer, size_count);
				fli_buffer += size_count;
				cur_x += size_count;
			}
			else if (size_count < 0)
			{
//				cout << "<";
				memset(output_buffer+line_count+cur_x, *fli_buffer++, -size_count);
				cur_x -= size_count;
			}
//			cout << "Cur_X: " << cur_x << endl;
		}
		line_count += header->width;
	}
}

void CL_FLIProvider::read_black()
{
	memset(output_buffer, 0, header->width*header->height);
}

void CL_FLIProvider::read_brun()
{
	int line_count = 0;
	for (int l=0;l<header->height;l++)
	{
		unsigned char num_packets = *fli_buffer++;

		int cur_x = 0;
		for (int p=0;p<num_packets;p++)
		{
			char size_count = *fli_buffer++;
			if (size_count > 0)
			{
				memset(output_buffer+line_count+cur_x, *fli_buffer++, size_count);
				cur_x += size_count;
			}
			else if (size_count < 0)
			{
				memcpy(output_buffer+line_count+cur_x, fli_buffer, -size_count);
				fli_buffer -= size_count;
				cur_x -= size_count;
			}
		}
		line_count += header->width;
	}
//	fli_buffer++; // this is something religious, and probably wrong
}

void CL_FLIProvider::read_copy()
{
	memcpy(output_buffer, fli_buffer, header->width*header->height);
	fli_buffer += header->width*header->height;
}

void CL_FLIProvider::read_delta()
{
	unsigned short num_lines = *((unsigned short *) fli_buffer);
	fli_buffer += 2;

	int cur_line = 0;
	for (int l=0;l<num_lines;l++)
	{
		short num_packets = *((unsigned short *) fli_buffer);
		fli_buffer += 2;

		if (num_packets < 0)
		{
			cur_line -= num_packets*header->width;
		}
		else if (num_packets > 0)
		{
			int cur_x = 0;
			for (int p=0;p<num_packets;p++)
			{
				unsigned char skip_count = *fli_buffer++;
				cur_x += skip_count;
				char size_count = *fli_buffer++;
				if (size_count > 0)
				{
					memcpy(output_buffer+cur_line+cur_x, fli_buffer, size_count<<1);
					fli_buffer += size_count << 1;
					cur_x += size_count << 1;
				}
				else if (size_count < 0)
				{
					for (int c=0;c<-size_count;c++)
					{
						*(((unsigned short *) ((unsigned char *) (output_buffer+cur_line+cur_x)))+c) = *((unsigned short *) fli_buffer);
					}
					fli_buffer += 2;
					cur_x -= size_count << 1;
				}
			}
			cur_line += header->width;
		}
	}
}
