/*
	$Id: input_mouse.cpp,v 1.1.1.1 2000/04/09 12:18:01 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------
*/

#include "Core/precomp.h"

#include "API/Core/Display/display.h"
#include "input_mouse.h"
#include "API/Core/System/cl_assert.h"

#include "Core/Display/DirectDraw/displaycard_win32compatible.h"

/******************************
				CL_Mouse_Win32
******************************/

#define NUM_MBUTTONS 3

CL_Mouse_Win32::CL_Mouse_Win32()
{
	cursor = new CL_InputCursor_Mouse_Win32();
	axes = new CL_InputAxis_Mouse_Win32[2];

	buttons = new CL_InputButton_Mouse_Win32*[NUM_MBUTTONS];
	for (int i=0; i<NUM_MBUTTONS; i++) buttons[i] = NULL;

	CL_System_Win32::add_listener(this);
}

CL_Mouse_Win32::~CL_Mouse_Win32()
{
	CL_System_Win32::remove_listener(this);

	delete cursor;

	for (int i=0; i<NUM_MBUTTONS; i++) delete buttons[i];
	delete[] buttons;
}

char *CL_Mouse_Win32::get_name() const
{
	return "Standard mouse";
}

int CL_Mouse_Win32::get_num_buttons() const
{
	return NUM_MBUTTONS;
}

CL_InputButton *CL_Mouse_Win32::get_button(int button_num)
{
	if (buttons[button_num] == NULL)
		buttons[button_num] = new CL_InputButton_Mouse_Win32(button_num);

	return buttons[button_num];
}

int CL_Mouse_Win32::get_num_axes() const
{
	return 2;
}

CL_InputAxis *CL_Mouse_Win32::get_axis(int axis_num)
{
	cl_assert(((unsigned int) axis_num)<2);
	return &axes[axis_num];
}

int CL_Mouse_Win32::get_num_hats() const
{
	return 0;
}

CL_InputHat *CL_Mouse_Win32::get_hat(int hat_num)
{
	return NULL;
}

int CL_Mouse_Win32::get_num_cursors() const
{
	return 1;
}

CL_InputCursor *CL_Mouse_Win32::get_cursor(int cursor_num)
{
	return cursor;
}

void CL_Mouse_Win32::keep_alive()
{
	if (CL_Display::get_current_card() == NULL) return;

	POINT cursor_pos;
	GetCursorPos(&cursor_pos);

	CL_DisplayCard_Win32Compatible *win32card =
		(CL_DisplayCard_Win32Compatible *) CL_Display::get_current_card();

	BOOL res = ScreenToClient(win32card->get_hwnd(), &cursor_pos);

	if (cursor->x != cursor_pos.x ||
		cursor->y != cursor_pos.y)
	{
		cursor->x = cursor_pos.x;
		cursor->y = cursor_pos.y;

		CL_Input::chain_mouse_move.on_mouse_move(this);
	}

	axes[0].center = float(CL_Display::get_width())/2.0f;
	axes[1].center = float(CL_Display::get_height())/2.0f;
	
	axes[0].pos = cursor_pos.x;
	axes[1].pos = cursor_pos.y;
}


bool CL_Mouse_Win32::received_event(UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	if (CL_Display::get_current_card() == NULL) return false;

	bool up = false;
	switch (uMsg)
	{
	case WM_LBUTTONUP:
	case WM_MBUTTONUP:
	case WM_RBUTTONUP:
		up = true;
		// fall through
	case WM_LBUTTONDOWN:
	case WM_MBUTTONDOWN:
	case WM_RBUTTONDOWN:
		{
			int button_states = wParam;
			for (int i=0; i<NUM_MBUTTONS; i++)
			{
				if (buttons[i] == NULL) continue;

				bool new_state = (button_states&(1<<i)) ? true : false;
				if (new_state != buttons[i]->button_state)
				{
					buttons[i]->button_state = new_state;

					CL_Key key;
					key.ascii = -1;
					key.id = i;
					key.x = cursor->x;
					key.y = cursor->y;

					if (up == false)
					{
						key.state = CL_Key::Pressed;
						CL_Input::chain_button_press.on_button_press(this, key);
					}
					else
					{
						key.state = CL_Key::Released;
						CL_Input::chain_button_release.on_button_release(this, key);
					}
				}
			}
		}
		break;

	case WM_MOUSEMOVE:
		{
			int x = LOWORD(lParam);
			int y = HIWORD(lParam);

			if (cursor->x != x ||
				cursor->y != y)
			{
				cursor->x = x;
				cursor->y = y;

				CL_Input::chain_mouse_move.on_mouse_move(this);
			}

			axes[0].center = float(CL_Display::get_width())/2.0f;
			axes[1].center = float(CL_Display::get_height())/2.0f;
			
			axes[0].pos = x;
			axes[1].pos = y;
		}
		break;

	default:
		return false;
	}

	return true;
}

/*******************************
	 CL_InputButton_Mouse_Win32
*******************************/

CL_InputButton_Mouse_Win32::CL_InputButton_Mouse_Win32(int _key)
{
	key = (1<<_key);
	button_state = false;
}

CL_InputButton_Mouse_Win32::~CL_InputButton_Mouse_Win32()
{
}

bool CL_InputButton_Mouse_Win32::is_pressed()
{
	return button_state;
}

/*******************************
	 CL_InputCursor_Mouse_Win32
*******************************/

CL_InputCursor_Mouse_Win32::CL_InputCursor_Mouse_Win32()
{
	x = 0;
	y = 0;
}

CL_InputCursor_Mouse_Win32::~CL_InputCursor_Mouse_Win32()
{
}

float CL_InputCursor_Mouse_Win32::get_x()
{
	POINT cursor_pos;
	GetCursorPos(&cursor_pos);

	CL_DisplayCard_Win32Compatible *win32card =
		(CL_DisplayCard_Win32Compatible *) CL_Display::get_current_card();

	ScreenToClient(win32card->get_hwnd(), &cursor_pos);

	return cursor_pos.x;
}

float CL_InputCursor_Mouse_Win32::get_y()
{
	POINT cursor_pos;
	GetCursorPos(&cursor_pos);

	CL_DisplayCard_Win32Compatible *win32card =
		(CL_DisplayCard_Win32Compatible *) CL_Display::get_current_card();

	ScreenToClient(win32card->get_hwnd(), &cursor_pos);

	return cursor_pos.y;
}

float CL_InputCursor_Mouse_Win32::get_max_x()
{
	if (CL_Display::get_current_card() == NULL) return -1;
	return CL_Display::get_width();
}

float CL_InputCursor_Mouse_Win32::get_max_y()
{
	if (CL_Display::get_current_card() == NULL) return -1;
	return CL_Display::get_height();
}

/*******************************
	 CL_InputAxis_Mouse_Win32
*******************************/

CL_InputAxis_Mouse_Win32::CL_InputAxis_Mouse_Win32()
{
	pos = 0;
	center = 1;
}

CL_InputAxis_Mouse_Win32::~CL_InputAxis_Mouse_Win32()
{
}

float CL_InputAxis_Mouse_Win32::get_pos()
{
	return (pos - center) / center;
}