/*

	------------------------------------------------------------------------
	PNG-SurfaceProvider Class
	written by Karsten-Olaf Laux

	in order to works with
	ClanLib, the platform independent game SDK.

	This file is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------

	File purpose:
		PNG (.png) surface provider.
*/

//! component="SurfaceProviders"

#ifndef header_PNGprovider
#define header_PNGprovider

#include "../core.h"
#include <png.h>

class CL_PNGProvider : public CL_SurfaceProvider_Generic
//: Surface provider that can load PNG (.png) files.
{
public:
	CL_InputSource* get_input_source() {return input_source; };
	//:returns Pointer to CL_InputSource
	//:(used by libpng-callback CL_PNGProvider::pngread_file() )

	static CL_Surface *create(CL_String handle, 
				  CL_InputSourceProvider *provider=NULL, 
				  bool transparent=true,
				  bool ignore_alphachannel=false);

	//: Loads the PNG file 'handle' from the inputsource provider 'provider. 
	//: Creates a CL_Surface using the PNG image and returns it.

	CL_PNGProvider(CL_String name, 
			 CL_InputSourceProvider *provider = NULL,
			 bool transparent=true,
			 bool ignore_alphachannel=false);
	//: Constructs a surface provider that can read PNG files.
	//!param: name - Name of the PNG file to load.
	//!param: provider - Input source provider that delivers the PNG file.
	//!param: transparent - True if a transparency color should be used.
	//!param: ignore_alphachannel - True if the alpha component should be ignored.
	//!param: trans_red - Red component of the transparency color.
	//!param: trans_green - Green component of the transparency color.
	//!param: trans_blue - blue component of the transparency color.

	virtual ~CL_PNGProvider();

	virtual unsigned int get_pitch() const { return pitch; }
	//: Returns the pitch of the image (bytes per line).

	virtual unsigned int get_width() const { return width; }
	//: Returns the width of the image.

	virtual unsigned int get_height() const { return height; }
	//: Returns the height of the image.
	
	virtual unsigned int get_num_frames() const { return no_sprs; }
	//: Returns the number of subsprites in the image.

	virtual EPixelFormat get_pixel_format() const { return RGBA8888; }
	//: Returns the pixelformat used by the image.

	virtual CL_Palette *get_palette() const { return NULL; }
	//: Returns the palette used by the image. NULL if system palette.

	virtual unsigned int get_src_colorkey() const { return trans_col; }
	//: Returns the transparency color used.

	virtual bool uses_src_colorkey() const { return m_uses_src_colorkey; }
	//: Returns whether a source colorkey is used.

	virtual bool is_indexed() const { return false; }
	//: Returns whether the target uses an indexed color mode or not.
	
	virtual unsigned int get_red_mask() const;
	//: Returns the red color mask used by the target.

	virtual unsigned int get_green_mask() const;
	//: Returns the green color mask by the target.

	virtual unsigned int get_blue_mask() const;
	//: Returns the blue color mask by the target.

	virtual unsigned int get_alpha_mask() const;
	//: Returns the alpha mask by the target.

	virtual void *get_data() const;
	//: Returns the image data. Provider must be locked before pointer is valid.

	virtual void perform_lock();
	//: Locks the surface provider.

	virtual void perform_unlock();
	//: Unlocks the surface provider.
	
	static void pngread_file(png_structp png_ptr,
				 png_bytep data, 
				 png_size_t length)
	//: Callback used by libpng to retrieve the filedata. 
	//: (calls get_input_source()->read_uchar8())
	  {
	    //since this method is static, we need to know who we are ...
	    CL_PNGProvider *instance =  (CL_PNGProvider *)png_get_io_ptr(png_ptr);
	    //no error-checking here ....
		std::cerr << ".";
	    png_size_t i;
	    for(i=0; i< length; i++)
	      data[i]=instance->get_input_source()->read_uchar8();
	  }

private:
	CL_String filename;
	bool locked;

	bool m_is_indexed, m_uses_src_colorkey;
	
	unsigned char *image;

	unsigned short map_length;
	unsigned char *color_map;

	unsigned char datatype;

	int pitch, height, no_sprs;
	int width;

	int bpp;

	bool transparent, ignore_alphachannel, use_alphapixels;

	int trans_num;
	int trans_col;
	unsigned char trans_redcol, trans_greencol, trans_bluecol;

	int pos;

	void read_data();

	void read_header();



	CL_InputSourceProvider *provider;
	CL_InputSource *input_source;

	//PNGlib stuff:
	png_structp png_ptr;
	png_infop info_ptr;
	png_infop end_info;

};

class CL_SetupPNG
{
public:
	static void init();
	static void deinit();
};

#endif
