#ifndef header_bsp
#define header_bsp

// a little math is always good:
class CL_Plane
{
public:
	double a,b,c,d;
	CL_Vector s, e0, e1;

	CL_Plane() { return; }

	CL_Plane(const CL_Vector &pos, const CL_Vector &v1, const CL_Vector &v2)
	{
		CL_Vector cross = v1.cross(v2);
		a = cross.x;
		b = cross.y;
		c = cross.z;
		d = -a*pos.x - b*pos.y - c*pos.z;
		s = pos;
		e0 = v1;
		e1 = v2;
	}

	CL_Vector plan_coords(const CL_Vector &a)
	{
		double t0;
		double t1;

		double div1 = ( e0.x*e1.y - e0.y*e1.x);
		double div2 = ( e0.y*e1.z - e0.z*e1.y);
		double div3 = ( e0.z*e1.x - e0.x*e1.z);

		// make sure we don't divide with zero:
		cl_assert(
			div1 != 0 ||
			div2 != 0 ||
			div3 != 0);

		if (fabs(div1) > fabs(div2) &&
			fabs(div1) > fabs(div3))
		{
			t1 = ( e0.x*(a.y-s.y) - e0.y*(a.x-s.x) ) / div1;
		}
		else if (fabs(div2) > fabs(div1) &&
				 fabs(div2) > fabs(div3))
		{
			t1 = ( e0.y*(a.z-s.z) - e0.z*(a.y-s.y) ) / div2;
		}
		else
		{
			t1 = ( e0.z*(a.x-s.x) - e0.x*(a.z-s.z) ) / div3;
		}

		double efx = fabs(e0.x);
		double efy = fabs(e0.y);
		double efz = fabs(e0.z);

		if (
			efx >= efy &&
			efx >= efz)
		{
			t0 = ( a.x-s.x-e1.x*t1 ) / e0.x;
		}
		else if (
			efy >= efx &&
			efy >= efz)
		{
			t0 = ( a.y-s.y-e1.y*t1 ) / e0.y;
		}
		else
		{
			t0 = ( a.z-s.z-e1.z*t1 ) / e0.z;
		}

		return CL_Vector(t0, t1);
	}
};

class CL_Line
{
public:
	CL_Vector pos, dir;

	CL_Line() { return; }

	CL_Line(const CL_Vector &p, const CL_Vector &d)
	{
		pos = p;
		dir = d;
	}

	double intersect_time(const CL_Plane &plan)
	{
		double div = (plan.a*dir.x+plan.b*dir.y+plan.c*dir.z);
		if (div == 0) 
		{
			throw CL_Error("CL_Plane is parallel with line.");
		}

		return
			(-plan.a*pos.x-plan.b*pos.y-plan.c*pos.z-plan.d) / div;
	}

	CL_Vector intersect_pos(const CL_Plane &plan)
	{
		double t = intersect_time(plan);
		return pos+dir*t;
	}
};

class CL_Rectangle
{
public:
	float x1, y1, x2, y2;
	CL_Rectangle(float _x1, float _y1, float _x2, float _y2) : x1(_x1), y1(_y1), x2(_x2), y2(_y2) {}
	CL_Rectangle(float *xvals, float *yvals, int num_vals_x, int num_vals_y, int xstride, int ystride)
	{
		set_x1(xvals, num_vals_x, xstride);
		set_y1(yvals, num_vals_y, ystride);
		set_x2(xvals, num_vals_x, xstride);
		set_y2(yvals, num_vals_y, ystride);
	}
	CL_Rectangle() { ; }

	void set_x1(float *vals, int num_vals=1, int stride=1)
	{
		x1 = get_min(vals, num_vals, stride);
	}
	
	void set_y1(float *vals, int num_vals=1, int stride=1)
	{
		y1 = get_min(vals, num_vals, stride);
	}
	
	void set_x2(float *vals, int num_vals=1, int stride=1)
	{
		x2 = get_max(vals, num_vals, stride);
	}

	void set_y2(float *vals, int num_vals=1, int stride=1)
	{
		y2 = get_max(vals, num_vals, stride);
	}

	bool get_intersection(const CL_Rectangle &rect, CL_Rectangle *res)
	{
		if (x1 > rect.x1) res->x1 = x1; else res->x1 = rect.x1;
		if (x2 < rect.x2) res->x2 = x2; else res->x2 = rect.x2;
		if (y1 > rect.y1) res->y1 = y1; else res->y1 = rect.y1;
		if (y2 < rect.y2) res->y2 = y2; else res->y2 = rect.y2;

		return res->x1 < res->x2 && res->y1 < res->y2;
	}

	void limit_region(const CL_Rectangle &r)
	{
		if (x1 < r.x1) x1 = r.x1;
		if (y1 < r.y1) y1 = r.y1;
		if (x2 > r.x2) x2 = r.x2;
		if (y2 > r.y2) y2 = r.y2;
	}

private:
	float get_min(float *vals, int num_vals=1, int stride=1)
	{
		float min_val = vals[0];
		float *vp = &vals[stride];
		for (int i=1;i<num_vals;i++)
		{
			if (*vp < min_val) min_val = *vp;
			vp += stride;
		}
		return min_val;
	}
	float get_max(float *vals, int num_vals=1, int stride=1)
	{
		float max_val = vals[0];
		float *vp = &vals[stride];
		for (int i=1;i<num_vals;i++)
		{
			if (*vp > max_val) max_val = *vp;
			vp += stride;
		}
		return max_val;
	}
};

template<class T>
class CL_BlockAlloc
{
public:
	CL_BlockAlloc()
	{
		cur_chunk = NULL;
		items_left = 0;
	}

	~CL_BlockAlloc()
	{
		clear();
	}

	void clear()
	{
		for (std::list<T *>::iterator it = chunks.begin();it!=chunks.end();it++)
		{
			delete[] *it;
		}
		chunks.clear();
		delete[] cur_chunk;
		cur_chunk = NULL;
		items_left = 0;
	}

	T *get()
	{
		if (!free_chunks.empty())
		{
			T *ret = free_chunks.back();
			free_chunks.pop_back();
			return ret;
		}
		if (items_left == 0)
		{
			if (cur_chunk != NULL)
			{
				chunks.push_back(cur_chunk);
			}
			cur_chunk = new T[1024];
			items_left = 1024;
		}

		T *ret = &cur_chunk[1024-items_left];
		items_left--;

		return ret;
	}

	void put(T *item)
	{
		free_chunks.push_back(item);
	}
private:
	std::list<T *> chunks;
	std::list<T *> free_chunks;
	T *cur_chunk;
	unsigned int items_left;
};

class CL_SceneRenderer
{
public:
	CL_SceneRenderer();
	virtual ~CL_SceneRenderer();

	void begin_cube(int cube_id);
	void end_cube();

	void begin_portal(int connect_id);
	void end_portal();

	void begin_facelist();
	void end_facelist();
	
	void begin_tstriplist();
	void end_tstriplist();
	
	void add_vertex(float x, float y, float z);
	void set_texture(CL_Texture *texture);
	void set_color(float r, float g, float b, float a);
	void set_texcoord(float u, float v);
	bool generate();

	void show(const CL_Vector &pos, const CL_Vector &dir, const CL_Vector &up);

//	--------------
//	Implementation

private:
	CL_Texture *cur_texture;
	float cur_cols[4];
	float cur_texcoords[2];

	void clear();
	struct SFace
	{
		CL_Texture *texture;
		float vdata[(4+2+3)*3];
	};

	struct STStrip
	{
		STStrip(const STStrip &s)
		{
			vdata = NULL;
			if (s.vcount > 0)
			{
				vdata = new float[s.vcount*9];
				memcpy(vdata, s.vdata, sizeof(float)*s.vcount*9);
			}
			texture = s.texture;
			vcount = s.vcount;
		}
		STStrip()
		{
			texture = NULL;
			vdata = NULL;
			vcount = 0;
		}
		~STStrip()
		{
			delete[] vdata;
			vdata = NULL;
		}
		CL_Texture *texture;
		float *vdata;
		int vcount;
	};

	struct SCube
	{
		SCube()
		{
			for (int i=0;i<6;i++)
			{
				neighbors[i] = -1;
			}
		}
		SCube(const SCube &c) : faces(c.faces), strips(c.strips)
		{
			memcpy(neighbors, c.neighbors, sizeof(int)*6);
			memcpy(portal_coords, c.portal_coords, sizeof(float)*6*3);
		}

		SCube &operator=(const SCube &c)
		{
			faces = c.faces;
			strips = c.strips;
			memcpy(neighbors, c.neighbors, sizeof(int)*6);
			memcpy(portal_coords, c.portal_coords, sizeof(float)*6*3);
			return *this;
		}
		std::vector<SFace *> faces;
		std::vector<STStrip *> strips;
		int neighbors[6];
		float portal_coords[6*3*4];
	};

	int cur_state;
	SFace *cur_face;
	int cur_vertex;
	STStrip *cur_strip;

	SCube *cur_cube;
	int cur_cube_id;
	int cur_portal_vertex;

	SCube **cubes;
	int cubes_size;

	struct SBSPartition
	{
		CL_Vector plane_normal;
		float pos[3];
		float plane_d;

		std::vector<int> cubes;

		SBSPartition *front;
		SBSPartition *back;
	};

	inline CL_Vector find_plane_normal(float *vdata);
	inline CL_Vector find_portal_plane_normal(float *vdata);
	inline float get_plane_distance(float *pos, const CL_Vector &plane_normal, float plane_d);
	int get_split_delta(
		const CL_Vector &n,
		float d,
		bool copy,
		SBSPartition *p,
		SBSPartition *front,
		SBSPartition *back);

	void find_even_partition(SBSPartition *p, SBSPartition *front, SBSPartition *back);
	bool partition_rec(SBSPartition *cur);

	float *pos;
	CL_Vector dir;
	CL_Vector up;
	int face_count;
	inline void show_facelist(const std::vector<SFace *> &faces);
	inline void show_trianglestrip(const std::vector<STStrip *> &strips);
	bool inside_cube(int cube_id, const CL_Vector &pos);
	int find_cube_location(SBSPartition *p, const CL_Vector &pos);

	bool *showed_cubes;
	void show_rec(SBSPartition *partition);
	void show_cubes_rec(int cube_id, int connect_cube, float *coords, int rcount);
	void show_bsp(SBSPartition *cur, float col);

	CL_BlockAlloc<SBSPartition> partition_alloc;
	std::vector<SBSPartition *> partitions;
	SBSPartition *root;
};

#endif