/**
  \class CVideoDeviceLinux
  \brief A device wrapper for Video4Linux devices

  This class wraps itself around a Video4Linux device; it supports
  webcams and grabber cards equally well. This device can be 'opened'
  multiple times, so more than one class can use this device (of course
  they all get the same picture).

  The class can return video images in RGB, YUV or both formats; it also
  has an optional buffer scheme for the images; this can prevent the
  otherwise deep copy that is needed when, for example, image diffs
  need to be calculated.

  The class will use mmap() when it is supported by the device driver. Also
  some device support select(); if not, a QTimer is used.
*/

#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

#include <qdatetime.h>

#include "VideoDeviceLinux.h"
#include "VideoDeviceInput.h"
#include "VideoAudioInput.h"

#include "ccvt.h"
#include "pwc-ioctl.h"

/**
  \param node_name The /dev/video* device name of the video device

  The constructor will do some basic checks to see if this is a valid device.
  If yes, \ref IsValid() will return TRUE, otherwise FALSE.
 */
CVideoDeviceLinux::CVideoDeviceLinux(const QString &node_name)
{
   struct video_capability vcap;
   struct pwc_probe Probe;
   struct pwc_serial Serial;
   int i;

   qDebug(">> CVideoDeviceLinux::CVideoDeviceLinux(%s)", (const char *)node_name);

   m_CamFD = -1;
   m_Palette = 0;
   m_Framerate = 10;
   vid_io_buffer = NULL;
   vid_io_buffer_size = vid_io_image_size = 0;
   m_pSettings = new CVideoDeviceOptions();
   pSettingsDlg = 0;

   m_NodeName = node_name;
   m_HasDisplayDialog = true;
   m_HasFormatDialog = true;
   m_HasSourceDialog = true;
   m_HasTunerDialog = false;

   m_VideoInputs.setAutoDelete(true);
   m_AudioInputs.setAutoDelete(true);
   m_CurrentVideoInput = -1;
   m_CurrentTuner = -1;
   m_CurrentAudioInput = -1;
   memset(&m_VMBuf, 0, sizeof(m_VMBuf));

   // Do a small test
   m_CamFD = ::open((const char *)node_name, O_RDWR);
   if (m_CamFD >= 0) {
     if (ioctl(m_CamFD, VIDIOCGCAP, &vcap) < 0) {
       qDebug("CVideoDevice::CVideoDevice() could not query capabilities; is this really a video device?");
     }
     else {
       m_Validated = true;
       m_MinSize.setWidth(vcap.minwidth);
       m_MinSize.setHeight(vcap.minheight);
       m_MaxSize.setWidth(vcap.maxwidth);
       m_MaxSize.setHeight(vcap.maxheight);
       vcap.name[31] = '\0';
       m_IntfName = vcap.name;

       /* Test for Philips cam */
       memset(&Probe, 0, sizeof(Probe));
       if (ioctl(m_CamFD, VIDIOCPWCPROBE, &Probe) == 0)
       {
         if (m_IntfName == Probe.name)
         {
           /* succes. Try serial number */
           if (ioctl(m_CamFD, VIDIOCPWCGSERIAL, &Serial) == 0)
           {
             m_SerialNumber = Serial.serial;
           }
         }
       }

       /* Query current video status */
       if (ioctl(m_CamFD, VIDIOCGPICT, &m_VPic) < 0)
       {
         qWarning("Could not query video picture... duh!?");
       }

       /* Query video inputs... */
       m_VideoInputs.clear();
       if (vcap.channels > 0) {
         m_VideoInputs.resize(vcap.channels);
         for (i = 0; i < vcap.channels; i++) {
            CVideoDeviceInput *input = 0;
            int no_tuners = 0;

            input = new CVideoDeviceInput(this, i);
            m_VideoInputs.insert(i, input);
            if (input != 0) {
              connect(input, SIGNAL(Selected(int)), this, SLOT(VideoInputSwitched(int)));

              /* Query tuners. Note that this class also maintains a list of
                 tuners, which is the combined list of all tuners of all inputs.
                 Therefor, the number given to a tuner here does not necessarely
                 reflect the (internal) number of the tuner with an input.
               */
              no_tuners = input->GetNumberOfTuners();
              if (no_tuners > 0) {
                int tc;

                m_HasTunerDialog = true;
                tc = m_Tuners.count();
                m_Tuners.resize(tc + no_tuners);
                for (int j = 0; j < no_tuners; j++) {
                   m_Tuners.insert(tc + j, input->GetTuner(j));
                }
              }
            }
         }
         // Set defaults
         if (vcap.channels == 1)
           m_CurrentVideoInput = 0;
         if (m_Tuners.count() == 1)
           m_CurrentTuner = 0;
       }
       /* ...and audio inputs */
       m_AudioInputs.clear();
       if (vcap.audios > 0) {
         m_AudioInputs.resize(vcap.audios);
         for (i = 0; i < vcap.audios; i++)
            m_AudioInputs.insert(i, new CVideoAudioInput(this, i));
       }
     }

     pSettingsDlg = new CVideoSettingsDlg(this);
     ::close(m_CamFD);
   }
   m_CamFD = -1;

   qDebug("<< CVideoDeviceLinux::CVideoDeviceLinux()");
}


CVideoDeviceLinux::~CVideoDeviceLinux()
{
   qDebug(">> CVideoDeivceLinux::~CVideoDeviceLinux()");
   Exit();
   delete m_pSettings;
   delete pSettingsDlg;
   qDebug("<< CVideoDeivceLinux::~CVideoDeviceLinux()");
}

// private

bool CVideoDeviceLinux::TryPalette(int palette, int depth)
{
   if (m_CamFD < 0)
     return false;
   m_VPic.palette = palette;
   m_VPic.depth = depth;
   if (ioctl(m_CamFD, VIDIOCSPICT, &m_VPic) < 0)
     return false;
   /* Sigh. It was to be expected. The OV511 and IBMCam don't pay attention to the palette field */
   if (ioctl(m_CamFD, VIDIOCGPICT, &m_VPic) < 0)
     return false;
   if (m_VPic.palette == palette) {
     m_Palette = palette;
     return true;
   }
   return false;
}

void CVideoDeviceLinux::SetPalette()
{
   /* Determine most preferable palette... With more and more
      color conversion going into the apps, we must try quite a few
      palettes before we hit one that we like.
      In case we want both YUV and RGB output, we prefer the YUV output.
    */
   char *pal_names[17] = { "", "grey", "hi240", "rgb565" ,"rgb24", "rgb32",
                           "rgb555", "yuv422", "yuyv" , "uyvy", "yuv420",
                           "yuv411", "raw", "yuv422p", "yuv411p", "yuv420p",
                           "yuv410p" };
   m_Palette = 0;
   if (m_CamFD < 0)
     return;

   if (m_PalYUV > 0) {
     TryPalette(VIDEO_PALETTE_YUV420P, 12) ||
     TryPalette(VIDEO_PALETTE_YUYV, 16)    ||
     TryPalette(VIDEO_PALETTE_YUV422, 16)  ||
     TryPalette(VIDEO_PALETTE_RGB32, 32)   ||
     TryPalette(VIDEO_PALETTE_RGB24, 24)   ||
     TryPalette(VIDEO_PALETTE_GREY, 8);
   }
   else if (m_PalRGB > 0) {
     TryPalette(VIDEO_PALETTE_RGB32, 32)   ||
     TryPalette(VIDEO_PALETTE_RGB24, 24)   ||
     TryPalette(VIDEO_PALETTE_YUV422, 16)  ||
     TryPalette(VIDEO_PALETTE_YUYV, 16)    ||
     TryPalette(VIDEO_PALETTE_YUV420P, 12) ||
     TryPalette(VIDEO_PALETTE_GREY, 8);
   }
   qDebug("CVideoDeviceLinux::SetPalette picked palette %d [%s]", m_Palette, pal_names[m_Palette]);

}

void CVideoDeviceLinux::CreateImagesRGB()
{
   int w, h;
   uint u;

qDebug(">> CVideoDeviceLinux::CreateImagesRGB()");
   m_RGB.resize(m_Buffers);
   w = m_ImageSize.width();
   h = m_ImageSize.height();
   if (m_Palette == VIDEO_PALETTE_RGB32) {
     // This is easy... map directly onto images
     for (u = 0; u < m_Buffers; u++) {
        m_RGB.insert(u, new QImage(vid_io_buffer + vid_io_offsets[u], w, h, 32, 0, 0, QImage::IgnoreEndian));
     }
   }
   else {
     for (u = 0; u < m_Buffers; u++) {
        m_RGB.insert(u, new QImage(w, h, 32));
     }
   }
qDebug("<< CVideoDeviceLinux::CreateImagesRGB()");
}

void CVideoDeviceLinux::DeleteImagesRGB()
{
qDebug(">> CVideoDeviceLinux::DeleteImagesRGB()");
   m_RGB.resize(0);
qDebug("<< CVideoDeviceLinux::DeleteImagesRGB()");
}

void CVideoDeviceLinux::CreateImagesYUV()
{
   int i, n;
   int w, h;

qDebug(">> CVideoDeviceLinux::CreateImagesYUV()");
   m_Y.resize(m_Buffers);
   m_U.resize(m_Buffers);
   m_V.resize(m_Buffers);
   w = m_ImageSize.width();
   h = m_ImageSize.height();
   n = w * h;

   if (m_Palette == VIDEO_PALETTE_YUV420P) {
     // This is easy... map directly onto images
     for (i = 0; i < m_Buffers; i++) {
        m_Y.insert(i, new QImage(vid_io_buffer + vid_io_offsets[i]            , w    , h    , 8, m_GrayScale, 256, QImage::IgnoreEndian));
        m_U.insert(i, new QImage(vid_io_buffer + vid_io_offsets[i] + n        , w / 2, h / 2, 8, m_GrayScale, 256, QImage::IgnoreEndian));
        m_V.insert(i, new QImage(vid_io_buffer + vid_io_offsets[i] + n + n / 4, w / 2, h / 2, 8, m_GrayScale, 256, QImage::IgnoreEndian));
     }
   }
   else {
     for (i = 0; i < m_Buffers; i++) {
        m_Y.insert(i, new QImage(w    , h    , 8, 256));
        m_U.insert(i, new QImage(w / 2, h / 2, 8, 256));
        m_V.insert(i, new QImage(w / 2, h / 2, 8, 256));
     }
   }
qDebug("<< CVideoDeviceLinux::CreateImagesYUV()");
}

void CVideoDeviceLinux::DeleteImagesYUV()
{
qDebug(">> CVideoDeviceLinux::DeleteImagesYUV()");
   m_Y.resize(0);
   m_U.resize(0);
   m_V.resize(0);
qDebug("<< CVideoDeviceLinux::DeleteImagesYUV()");
}


// private slots
void CVideoDeviceLinux::VideoInputSwitched(int n)
{
   m_CurrentVideoInput = n;
}

void CVideoDeviceLinux::NewFrequency(float frequency)
{
  CVideoDeviceTuner *tuner = 0;

  if (m_CurrentTuner >= 0) {
    tuner = m_Tuners[m_CurrentTuner];
    if (tuner != 0) {
      tuner->SetFrequency(frequency);
    }
  }
}


// protected

/**
  \brief Open device, set defaults, prepare for capture
*/
bool CVideoDeviceLinux::Init()
{
   struct video_window vwin;
   struct video_audio Audio;
   QSize RequestedSize;
   QDomNode v;
   uint i;

   qDebug(">> CVideoDeviceLinux::Init()");
   m_CamFD = ::open((const char *)m_NodeName, O_RDWR);
   if (m_CamFD < 0)
     return false;

   /* Check for Philips cams */
   m_HasFramerate = false;
   if (m_IntfName.find("Philips") == 0) {
     m_HasFramerate = true;
   }
   else {
     struct pwc_probe Probe;

     if (ioctl(m_CamFD, VIDIOCPWCPROBE, &Probe) == 0) {
       if (m_IntfName == Probe.name) {
         m_HasFramerate = true;
       }
     }
   }
   if (m_HasFramerate)
     qDebug("Device controls its own framerate.");

   /* Probe audio functionality */
   if (ioctl(m_CamFD, VIDIOCGAUDIO, &Audio) == 0) {
     m_Mutable = (Audio.flags & VIDEO_AUDIO_MUTABLE);
     if (m_Mutable)
       qDebug("Device can be muted (usually TV card).");
   }
   else
     qWarning("Cannot query audio capabilities of video device.");

   /* See if device has mmap(); */
   m_VMBuf.size = 0;
   m_VMBuf.frames = 0;
   vid_io_buffer_size = 0;
   if (ioctl(m_CamFD, VIDIOCGMBUF, &m_VMBuf) == 0) {
     qDebug("Using mmap(), size = %d", m_VMBuf.size);
     vid_io_buffer_size = m_VMBuf.size;
     if (m_RequestedBuffers > 0) {
       // User specified a number of buffers; see if we can match that.
       if (m_VMBuf.frames < m_RequestedBuffers) {
         m_Buffers = m_VMBuf.frames;
         qWarning("There are more buffers requested than device can provide. Limiting to %d buffers.", m_Buffers);
       }
       else
         m_Buffers = m_RequestedBuffers;
     }
     else // We grab everything we can.
       m_Buffers = m_VMBuf.frames;
   }
   else {
     m_VMBuf.size = 0; // Just to be sure....
     m_VMBuf.frames = 0;
   }

   /* See if we can actually mmap() the memory */
   if (m_VMBuf.size > 0) {
     vid_io_buffer = (uchar *)mmap(NULL, vid_io_buffer_size, PROT_READ | PROT_WRITE, MAP_SHARED, m_CamFD, 0);
     if (vid_io_buffer == (uchar *)-1) {
       qWarning("mmap() failed (%d). Falling back to non-mmap()ed mode.", errno);
       m_VMBuf.size = 0;
       vid_io_buffer = NULL;
     }
     else {
       qDebug("mmap()ed %d buffers.", m_Buffers);
       vid_io_offsets.resize(m_Buffers);
       for (i = 0; i < m_Buffers; i++)
          vid_io_offsets[i] = m_VMBuf.offsets[i];
     }
   }

   if (m_VMBuf.size == 0) { // No mmap or failed: allocate a buffer
     qDebug("Allocating own buffer.");
     m_Buffers = m_RequestedBuffers;
     if (m_Buffers < 0)
       m_Buffers = 4; // In case the user didn't specify a buffer size, we make one.
     vid_io_buffer_size = m_Buffers * m_MaxSize.width() * m_MaxSize.height() * 4;
     vid_io_buffer = new uchar[vid_io_buffer_size];
     vid_io_offsets.resize(m_Buffers);
     for (i = 0; i < m_Buffers; i++)
        vid_io_offsets[i] = i * m_MaxSize.width() * m_MaxSize.height() * 4;
   }
   if (vid_io_buffer == NULL) {
     qWarning("Failed to mmap/malloc memory!");
     return false;
   }

   // Try to set default image size (from config)
   memset(&vwin, 0, sizeof(vwin));
   RequestedSize = m_pSettings->GetSize();
   if (!RequestedSize.isNull())
   {
     vwin.width = RequestedSize.width();
     vwin.height = RequestedSize.height();
     if (m_HasFramerate) {
       vwin.flags |= (m_pSettings->GetFramerate() << PWC_FPS_SHIFT) & PWC_FPS_FRMASK;
     }
     if (ioctl(m_CamFD, VIDIOCSWIN, &vwin) < 0) {
       qWarning("Failed to restore image size (%d, %d)", vwin.width, vwin.height);
     }
   }

   // Query current image size
   if (ioctl(m_CamFD, VIDIOCGWIN, &vwin) < 0) {
     qWarning("Failed to query current image size.");
   }
   else {
     m_ImageSize.setWidth(vwin.width);
     m_ImageSize.setHeight(vwin.height);

     //emit SizeChanged(m_ImageSize);
     if (m_HasFramerate)
     {
        m_Framerate = (vwin.flags & PWC_FPS_FRMASK) >> PWC_FPS_SHIFT;
       qDebug("Initial image size = (%d, %d), framerate = %d", vwin.width, vwin.height, m_Framerate);
     }
     else
       qDebug("Initial image size = (%d, %d)", vwin.width, vwin.height);
   }

   // Restore brightness, contrast, etc.
   m_VPic.brightness = m_pSettings->GetBrightness();
   m_VPic.contrast = m_pSettings->GetContrast();
   m_VPic.colour = m_pSettings->GetSaturation();
   m_VPic.hue = m_pSettings->GetHue();
   m_VPic.whiteness = m_pSettings->GetGamma();
   if (ioctl(m_CamFD, VIDIOCSPICT, &m_VPic) < 0)
     qWarning("Failed to restore video controls.");

   // Restore input/tuner
   SetInput(m_CurrentVideoInput);
   SetTuner(m_CurrentTuner);

   qDebug("<< CVideoDeviceLinux::Init()");
   return true;
}

void CVideoDeviceLinux::Exit()
{
   if (m_VMBuf.size > 0) {
     munmap(vid_io_buffer, vid_io_buffer_size);
     m_VMBuf.size = 0;
   }
   else
     delete [] vid_io_buffer;
   vid_io_buffer = 0;
   vid_io_buffer_size = 0;
   vid_io_image_size = 0;

   m_ImageSize.setWidth(0);
   m_ImageSize.setHeight(0);

   if (m_CamFD >= 0)
     ::close(m_CamFD);
   m_CamFD = -1;
}




bool CVideoDeviceLinux::StartCapture()
{
   int n;

   if (m_CamFD < 0) {
     qWarning("CVideoDeviceLinux::StartCapture() Cannot capture a closed device!");
     return false;
   }

   if (running()) {
     qWarning("Capture thread already running.");
     return false;
   }

   qDebug(">> CVideoDeviceLinux::StartCapture()");
   SetPalette();

   if (m_VMBuf.size > 0) {
     m_VMMap.format = m_Palette;
     m_VMMap.width  = m_ImageSize.width();
     m_VMMap.height = m_ImageSize.height();
     // Use all buffers
     for (int i = 0; i < m_Buffers; i++)
     {
        m_VMMap.frame = i;
        if (ioctl(m_CamFD, VIDIOCMCAPTURE, &m_VMMap) < 0) {
          qWarning("VIDIOCMCAPTURE failed (%s)", strerror(errno));
        }
     }
   }

   /* calculate image size */
   n = m_ImageSize.width() * m_ImageSize.height();
   switch (m_Palette)
   {
     case VIDEO_PALETTE_GREY   : vid_io_image_size = n;           break;
     case VIDEO_PALETTE_YUV420P: vid_io_image_size = (n * 3) / 2; break;
     case VIDEO_PALETTE_YUYV   : vid_io_image_size = n * 2;       break;
     case VIDEO_PALETTE_YUV422 : vid_io_image_size = n * 2;       break;
     case VIDEO_PALETTE_RGB24  : vid_io_image_size = n * 3;       break;
     case VIDEO_PALETTE_RGB32  : vid_io_image_size = n * 4;       break;
     default:
       qWarning("Unknown palette");
       vid_io_image_size = 0;
       break;
   }

   CreateVideoFrames();

   m_StopCapture = false;
   start(); // start our own thread
   qDebug("<< CVideoDeviceLinux::StartCapture()");
   return false;
}


void CVideoDeviceLinux::StopCapture()
{
   if (m_CamFD < 0)
     return;

   qDebug(">> CVideoDeviceLinux::StopCapture()");
   if (running()) {
     qDebug("Waiting for capture thread to stop...");
     m_StopCapture = true;
     m_CaptureFinished.wait();
     qDebug("Capture thread stopped.");
   }

   // Clear all buffers
   if (m_VMBuf.size > 0) {
     for (int i = 0; i < m_Buffers; i++)
        ioctl(m_CamFD, VIDIOCSYNC, &i);
   }
   DeleteVideoFrames();
   qDebug("<< CVideoDeviceLinux::StopCapture()");
}

void CVideoDeviceLinux::run()
{
   CVideoFrame *frame = 0;
   uchar *src = 0, *dst = 0;
   uchar *dsty = 0, *dstu = 0, *dstv = 0;
   int BufNum = 0;
   long TimeDays = 0, TimeNow = 0, TimeLast = 0;
   QTime TimeTick;

   qDebug(">> CVideoDeviceLinux::run()...");
   TimeTick.start();
   while (!m_StopCapture) {
      // Acquire empty frame; the loop is somewhat nasty, but the best at the moment.
      // Need to implement a QWaitCondition somewhere
      while (frame == 0) {
         frame = GetFillFrame();
         if (frame == 0) {
           // Oops, no frame available: wait a bit.
           m_FrameDropped++;
           if (!m_HasFramerate)
             usleep(1000000 / m_Framerate);
           else
             usleep(15000); // sleep 15 ms
         }
      }

      BufNum = frame->GetNumber();
      src = vid_io_buffer + vid_io_offsets[BufNum];
      if (m_VMBuf.size > 0) {
        // using mmap
        if (ioctl(m_CamFD, VIDIOCSYNC, &BufNum) < 0) {
          qWarning("run(): VIDIOCSYNC(%d) failed (%s)", BufNum, strerror(errno));
          usleep(100000); // wait 100ms
          // Retry buffer
          m_VMMap.frame = BufNum;
          if (ioctl(m_CamFD, VIDIOCMCAPTURE, &m_VMMap) < 0)
            qWarning("VDLinux::run() VIDIOCMCAPTURE failed (%s)", strerror(errno));
          continue;
        }
      }
      else {
        // normal read
        if (read(m_CamFD, src, vid_io_image_size) < 0) {
          qWarning("run(): read() failed (%s)", strerror(errno));
          usleep(100000); // wait 100ms
          continue;
        }
      }

      /* Note: there is something to be said against doing conversion here
         in this thread. The loop should be as thight as possible. However,
         we then loose the information about which palette the original data
         is. OTOH, conversion is probably better in CVideoFrame anyway.
         Decisions, decisions....
       */

      // We now have the frame ready in our buffer; do necessary conversion
      switch (m_Palette)
      {
        case VIDEO_PALETTE_YUV420P:
          if (m_PalRGB > 0) {
           dst = frame->GetRGB()->bits();
           ccvt_420p_bgr32(m_ImageSize.width(), m_ImageSize.height(), src, dst);
         }
         break;

        case VIDEO_PALETTE_RGB24:
          if (m_PalRGB > 0) {
            dst = frame->GetRGB()->bits();
            //ccvt_bgr24_bgr32(m_ImageSize.width(), m_ImageSize.height(), src, dst);
            ccvt_rgb24_bgr32(m_ImageSize.width(), m_ImageSize.height(), src, dst);
          }
          if (m_PalYUV)
          {
            qWarning("// TODO: rgb24 to yuv420p conversion");
          }
          break;

        case VIDEO_PALETTE_RGB32:
          if (m_PalYUV)
          {
            qWarning("// TODO: rgb32 to yuv420p conversion");
          }
          break;

        case VIDEO_PALETTE_YUV422:
        case VIDEO_PALETTE_YUYV:
          if (m_PalRGB > 0)
          {
            dst = frame->GetRGB()->bits();
            ccvt_yuyv_bgr32(m_ImageSize.width(), m_ImageSize.height(), src, dst);
          }
          if (m_PalYUV > 0)
          {
            dsty = frame->GetY()->bits();
            dstu = frame->GetU()->bits();
            dstv = frame->GetV()->bits();
            ccvt_yuyv_420p(m_ImageSize.width(), m_ImageSize.height(), src, dsty, dstu, dstv);
          }
          break;

        default:
          qWarning("Unsupported palette %d", m_Palette);
          break;
      }

      // set sequence and timestamp
      frame->SetSequence(m_FrameCount);
      TimeNow = TimeTick.elapsed();
      if (TimeNow < TimeLast)
        TimeDays += 86400000; // We have a wrap around after 24 hours; TimeDays will wrap around after nearly 50 days...
      frame->SetTimeStamp(TimeDays + TimeNow);
      TimeLast = TimeNow;
      // Frame is available for use
      ReturnFillFrame();
      frame = 0;

      // Notify whoever is waiting for us; we cannot use a regular signal since
      // we are in a thread
      SendSignal(frame_ready);

      if (m_VMBuf.size > 0) {
        // Setup buffer again for mmap
        m_VMMap.frame = BufNum;
        if (ioctl(m_CamFD, VIDIOCMCAPTURE, &m_VMMap) < 0)
          qWarning("VDLinux::run() VIDIOCMCAPTURE failed (%s)", strerror(errno));
      }
      m_FrameCount++;
   }
   m_CaptureFinished.wakeAll();
   qDebug("<< CVideoDeviceLinux::run()...");
}



// public

void CVideoDeviceLinux::SetConfiguration(const QDomNode &node)
{
  QDomNode v = node.namedItem("settings");
  CVideoDevice::SetConfiguration(node);

  delete m_pSettings;
  m_pSettings = new CVideoDeviceOptions();
  m_pSettings->SetXML(v);
  // Initialize dialog
  if (pSettingsDlg != 0) {
    pSettingsDlg->SetConfiguration(v);
  }
}

void CVideoDeviceLinux::GetConfiguration(QDomNode &node) const
{
  CVideoDevice::GetConfiguration(node);
  // Append our settings to the XML node
  if (m_pSettings != 0) {
    QDomNode v = node.namedItem("settings");
    m_pSettings->GetXML(v);
  }
}

long CVideoDeviceLinux::GetDescriptor() const
{
   return m_CamFD;
}

void CVideoDeviceLinux::Mute(bool on) const
{
   struct video_audio Audio;

   if (m_CamFD < 0 || !m_Mutable)
     return;
   if (ioctl(m_CamFD, VIDIOCGAUDIO, &Audio) < 0) {
     qWarning("VIDIOCGAUDIO() failed.");
   }
   else {
     if (on)
       Audio.flags |= VIDEO_AUDIO_MUTE;
     else
       Audio.flags &= ~VIDEO_AUDIO_MUTE;
     if (ioctl(m_CamFD, VIDIOCSAUDIO, &Audio) < 0) {
       qWarning("VIDIOCSAUDIO failed.");
     }
   }
}


QSize CVideoDeviceLinux::GetMinSize() const
{
   return m_MinSize;
}

QSize CVideoDeviceLinux::GetMaxSize() const
{
   return m_MaxSize;
}

/**
  \brief Try new size
*/
void CVideoDeviceLinux::SetSize(const QSize &new_size)
{
   struct video_window vwin;

   qDebug(">> CVideoDeviceLinux::SetSize(%dx%d)", new_size.width(), new_size.height());
   if (m_CamFD < 0)
     return;

   StopCapture();
   memset(&vwin, 0, sizeof(vwin)); // clear structure
   vwin.width = new_size.width();
   vwin.height = new_size.height();
   if (ioctl(m_CamFD, VIDIOCSWIN, &vwin) < 0) {
     qDebug("SWIN failed");
   }
   else {
     /* Read back; it might be the hardware decides a bit differently
        (for example, multiples of 8)
      */
     if (ioctl(m_CamFD, VIDIOCGWIN, &vwin) < 0) {
       qDebug("GWIN failed");
     }
     else {
       qDebug("returned size = %dx%d", vwin.width, vwin.height);
       if (vwin.width == 0 && vwin.height == 0) {
         emit Error(SizeChangeFailed);
       }
       else {
         m_ImageSize.setWidth(vwin.width);
         m_ImageSize.setHeight(vwin.height);
         m_pSettings->SetSize(m_ImageSize);
         emit SizeChanged(m_ImageSize);
       }
     }
   }
   StartCapture();
   qDebug("<< CVideoDeviceLinux::SetSize");
}


bool CVideoDeviceLinux::HasFramerate() const
{
   return m_HasFramerate;
}


int CVideoDeviceLinux::GetFramerate() const
{
   struct video_window vwin;

   if (m_HasFramerate) {
     memset(&vwin, 0, sizeof(vwin)); // clear structure
     if (ioctl(m_CamFD, VIDIOCGWIN, &vwin) < 0) {
       qDebug("SetFramerate: GWIN failed");
       return -1;
     }
     return (vwin.flags & PWC_FPS_FRMASK) >> PWC_FPS_SHIFT;
   }
   return m_Framerate;
}

void CVideoDeviceLinux::SetFramerate(int fps)
{
   struct video_window vwin;

   if (fps < 1 || fps > 60)
     return;
   if (m_CamFD < 0)
     return;

   /* There's no need to stop capture. */
   m_Framerate = fps;
   if (m_HasFramerate) {
     qDebug("SetFramerate: setting fps in device.");
     memset(&vwin, 0, sizeof(vwin)); // clear structure
     if (ioctl(m_CamFD, VIDIOCGWIN, &vwin) < 0) {
       qDebug("SetFramerate: GWIN failed");
       return;
     }

     vwin.flags &= ~PWC_FPS_FRMASK;
     vwin.flags |= (fps << PWC_FPS_SHIFT);

     if (ioctl(m_CamFD, VIDIOCSWIN, &vwin) < 0) {
       qDebug("SetFramerate: SWIN failed");
       return;
     }
   }
   m_pSettings->SetFramerate(fps);
   emit FramerateChanged(fps);
}

int CVideoDeviceLinux::GetBrightness() const
{
   return m_VPic.brightness;
}

bool CVideoDeviceLinux::SetBrightness(int val)
{
   if (m_CamFD < 0)
     return false;

   m_VPic.brightness = val & 0xffff;
   if (ioctl(m_CamFD, VIDIOCSPICT, &m_VPic) < 0)
     return false;
   m_pSettings->SetBrightness(val);
   return true;
}

int CVideoDeviceLinux::GetContrast() const
{
   return m_VPic.contrast;
}

bool CVideoDeviceLinux::SetContrast(int val)
{
   if (m_CamFD < 0)
     return false;

   m_VPic.contrast = val & 0xffff;
   if (ioctl(m_CamFD, VIDIOCSPICT, &m_VPic) < 0)
     return false;
   m_pSettings->SetContrast(val);
   return true;
}

int CVideoDeviceLinux::GetHue() const
{
   return m_VPic.hue;
}

bool CVideoDeviceLinux::SetHue(int val)
{
   if (m_CamFD < 0)
     return false;

   m_VPic.hue = val & 0xffff;
   if (ioctl(m_CamFD, VIDIOCSPICT, &m_VPic) < 0)
     return false;
   m_pSettings->SetHue(val);
   return true;
}

int CVideoDeviceLinux::GetColour() const
{
   return m_VPic.colour;
}

bool CVideoDeviceLinux::SetColour(int val)
{
   if (m_CamFD < 0)
     return false;

   m_VPic.colour = val & 0xffff;
   if (ioctl(m_CamFD, VIDIOCSPICT, &m_VPic) < 0)
     return false;
   m_pSettings->SetSaturation(val);
   return true;
}

int CVideoDeviceLinux::GetWhiteness() const
{
   return m_VPic.whiteness;
}

bool CVideoDeviceLinux::SetWhiteness(int val)
{
   if (m_CamFD < 0)
     return false;

   m_VPic.whiteness = val & 0xffff;
   if (ioctl(m_CamFD, VIDIOCSPICT, &m_VPic) < 0)
     return false;
   m_pSettings->SetGamma(val);
   return true;
}

/**
  \brief Get number of inputs

  This functions returns the number of video inputs on the video device.
  TV cards usually have more than one input; webcams only one.

  The numbering starts at 0.
*/

int CVideoDeviceLinux::GetNumberOfInputs() const
{
   return (int)m_VideoInputs.count();
}

void CVideoDeviceLinux::SetInput(int input)
{
  CVideoDeviceInput *pInput = 0;

  if (input < 0 || input >= (int)m_VideoInputs.count())
    return;
  pInput = m_VideoInputs[input];
  if (pInput!= 0) {
    m_CurrentVideoInput = input;
    if (m_OpenCount > 0) {
      if (!pInput->Select()) {
        m_CurrentVideoInput - -1;
      }
    }
  }
}

int CVideoDeviceLinux::GetInput() const
{
  return m_CurrentVideoInput;
}

/**
  \brief Get number of tuners

  This function returns ALL tuners belonging to this video device.

  \note It is unlikely a TV card has more than one tuner.

*/

int CVideoDeviceLinux::GetNumberOfTuners() const
{
  return (int)m_Tuners.count();
}

void CVideoDeviceLinux::SetTuner(int tuner)
{
  CVideoDeviceTuner *pTuner = 0;

  if (tuner < 0 || tuner >= (int)m_Tuners.count())
    return;
  pTuner = m_Tuners[tuner];
  if (pTuner != 0) {
    m_CurrentTuner = tuner;
    if (m_OpenCount > 0) {
      if (!pTuner->Select()) {
        m_CurrentTuner = -1;
      }
    }
  }
}

int CVideoDeviceLinux::GetTuner() const
{
  return m_CurrentTuner;
}


CVideoDeviceInput *CVideoDeviceLinux::GetVideoInput(int number) const
{
  if (number < 0)
    number = m_CurrentVideoInput;
  if (number < 0 || number >= (int)m_VideoInputs.count())
    return 0;
  return m_VideoInputs.at(number);
}



CVideoDeviceTuner *CVideoDeviceLinux::GetVideoTuner(int number) const
{
  if (number < 0)
    number = m_CurrentTuner;
  if (number < 0 || number >= (int)m_Tuners.count())
    return 0;
  return m_Tuners.at(number);
}



// public slots

void CVideoDeviceLinux::ShowDisplayDialog()
{
   if (pSettingsDlg != 0) {
     pSettingsDlg->ShowDisplayTab();
   }
}

void CVideoDeviceLinux::ShowFormatDialog()
{
   if (pSettingsDlg != 0) {
     pSettingsDlg->ShowFormatTab();
   }
}

void CVideoDeviceLinux::ShowSourceDialog()
{
   if (pSettingsDlg != 0) {
     pSettingsDlg->ShowSourceTab();
   }
}


