/**
  \class CBasicPanel
  \brief A simple panel that displays 1 or 3 images (RGB/YUV style)
  
  This panel can take 1 to N images and will display them when asked
  for it. It can be used to display intermediate results or just an image
  that needs to be registered with a base class.

  It overloads SetSize() so it will create images of the correct size and
  depth;, but these images will be destroyed with SetImage();
*/

#include "BasicPanel.h"


/**
  \brief Constructor
  \param panel_type Panel type; only CCamPanel::RGB or CCamPanel::YUV420 is useful
  
  This creates the panel and reserves space for the images.
*/ 
CBasicPanel::CBasicPanel(const char *panel_name, const char *desc, int panel_type, QWidget *parent, const char *name)
	: CCamPanel(panel_name, desc, panel_type, TRUE, parent, name)
{
}


/**
  \fn CBasicPanel::~CBasicPanel()
  \brief Destructor
  
  Destroys objects. The QImage objects it contains are destroyed as well, 
  but this does not affect operations as these images are just copies.
*/
CBasicPanel::~CBasicPanel()
{
}


// public

/** 
  \fn void CBasicPanel::SetImage(int n, const QImage &new_image, bool deep)
  \brief Set an image of BasicPanel
  \param n Image number
  \param new_image The QImage class
  \param deep Force deep copy
  
  This class makes a shallow or deep copy of the image and sets it in 
  its internal list of image. For RGB panels, <b>n</b> is not used;
  for YUV420 panels, if <b>n</b> is out of bounds, the action is ignored.
 */
void CBasicPanel::SetImage(int n, const QImage &new_image, bool deep)
{
   QImage *pImage = NULL;
   
   switch(GetPanelType()) {
     case RGB: 
       pImage = &ImgRGB;
       break;
     case YUV420:
       switch(n) {
         case 0: pImage = &ImgY; break;
         case 1: pImage = &ImgU; break;
         case 2: pImage = &ImgV; break;
       }
       break;
   }
   if (pImage != NULL) {
     if (deep)
       *pImage = new_image.copy();
     else
       *pImage = new_image;
   }
   update();
}

/**
  \fn void CBasicPanel::SetSize(const QSize &ns)
  \brief [overloaded] Sets size and initializes images
*/


void CBasicPanel::SetSize(const QSize &ns)
{
   CCamPanel::SetSize(ns);
   CreateImages();
}
