/*  audiodevs: Abstraction layer for audio hardware & samples
    Copyright (C) 2003-2004 Nemosoft Unv.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    For questions, remarks, patches, etc. for this program, the author can be
    reached at camstream@smcc.demon.nl.
*/

#ifndef AUDIODEVICE_HPP
#define AUDIODEVICE_HPP

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <qglobal.h>

#if (QT_VERSION >= 0x030000)
#include <qmutex.h>
#endif

#include <qdom.h>
#include <qobject.h>
#include <qthread.h>

#include "AudioRingBuffer.h"
#include "SoundAttributes.h"

class CAudioDevice: public QObject, public QThread
{
friend class CAudioCollector;
   Q_OBJECT
public:
   enum Mode { Capture, Playback };

private:
   Mode m_Mode;
   int m_OpenCount;
   int m_CaptureCount, m_PlaybackCount;

private slots:
   void EnableCapture();
   void DisableCapture();
   void EnablePlayback();
   void DisablePlayback();

protected:
   QString m_ShortName, m_LongName;
   QString m_NodeName;
   bool m_Validated;

   SoundAttributes m_CurrentSoundAttr;
   CRingBuffer m_CaptureBuffer, m_PlaybackBuffer;

   virtual bool Init() = 0;
   virtual void Exit() = 0;

   virtual int StartCapture() = 0;
   virtual void StopCapture() = 0;
   virtual int StartPlayback() = 0;
   virtual void StopPlayback() = 0;

   Mode GetMode() const { return m_Mode; };
   int GetCaptureCount() const { return m_CaptureCount; };
   int GetPlaybackCount() const { return m_PlaybackCount; };

public:
   CAudioDevice();
   virtual ~CAudioDevice();

   QString GetName() const;
   QString GetLongName() const;
   QString GetNodeName() const;

   bool IsValid() const;
   bool Open(Mode m);
   void Close();
   bool IsOpen() const;

   bool SetSoundAttributes(const SoundAttributes &);
   SoundAttributes GetSoundAttributes() const;

   CAudioRingBufferReader *CreateReader();
   CRingBufferWriter *CreateWriter();

   virtual unsigned long GetPlaybackPointer() const = 0;

   virtual void SetBufferLength(unsigned int len, unsigned int chunk_length = 0) = 0;
   virtual void SetBufferTime(unsigned int ms, unsigned int chunk_length = 0) = 0;
//   virtual int WriteDirectly(const void *, unsigned int len) = 0;

///   virtual void ShowSettingsDialog() = 0;
   virtual void ShowMixerControls() = 0;

   virtual void GetMixerSettings(QDomNode &) const = 0;
   virtual void SetMixerSettings(const QDomNode &) const = 0;

signals:
   void Opened();
   void Closed();

   /**
     \brief Let the world know we (are about to) change the sound attributes

     This signal is emitted when the new attributes are set in the device.
     This signal can be emitted multiple times in a row between stop and start
     of capture/playback, depending on whether the new SoundAttributes are
     accepted. See also \ref SetSoundAttributes()
   */
   void SoundAttributesChanged(const SoundAttributes &);
};


#endif
