#ifndef VIDEO_DEVICE_HPP
#define VIDEO_DEVICE_HPP

#include <sys/ioctl.h>
#include <sys/types.h>
#include <linux/videodev.h>

#include <qimage.h>
#include <qobject.h>
#include <qsocketnotifier.h>
#include <qtimer.h>
#include <qvector.h>

#include "VideoDeviceInput.h"

class CVideoDevice: public QObject
{
friend class CVideoDeviceInput;
friend class CVideoDeviceTuner;
   Q_OBJECT
private:
   bool validated;
   QString NodeName, IntfName;
   int CamFD, Opened;
   bool UseSelect, HasFramerate;
   int Palette, FrameRate;
   int PalRGB, PalYUV;
   struct video_picture VPic;
   struct video_mbuf VMBuf;
   int Buffers, CurBuffer;

   QVector <CVideoDeviceInput>Inputs;
   int CurrentInput;

   QImage *RGB, *Y, *U, *V;
   int image_w, image_h;
   int max_w, max_h;
   int min_w, min_h;
   uchar *vid_buffer;
   int vid_buffer_size, image_buffer_size;

   QSocketNotifier *pImageSocket;
   QTimer *pImageTimer;

   void Init(bool First = FALSE);
   void CleanUp();
   bool TryPalette(int pal, int depth);
   void SetPalette();
   
   int MCapture(int buf);
   int MSync();

   void CreateImagesRGB();
   void ResetImagesRGB();
   void CreateImagesYUV();
   void ResetImagesYUV();

private slots:
   void LoadImage();

public:
   CVideoDevice(const QString &node_name);
   ~CVideoDevice();

   bool IsValid();

   int Open(int buffers = 1);
   void Close();
   int GetDescriptor() const;
   
   void EnableRGB(bool isOn);
   void EnableYUV(bool isOn);

   QString GetNodeName() const;
   QString GetIntfName() const;
   
   QSize GetMinSize() const;
   QSize GetSize();
   QSize GetMaxSize() const;
   bool  SetSize(int width, int height);
   bool  SetSize(const QSize &new_size);
   int   GetFramerate() const;
   bool  SetFramerate(int fps, bool snap = FALSE);
   int   GetMBuffers() const;

   int   GetInputs() const;
   int   GetCurrentInput() const;
   CVideoDeviceInput *GetInput(int number) const;
   bool SelectInput(int number);
   
   int GetBrightness() const;
   bool SetBrightness(int val);
   int GetContrast() const;
   bool SetContrast(int val);
   int GetHue() const;
   bool SetHue(int val);
   int GetColour() const;
   bool SetColour(int val);
   int GetWhiteness() const;
   bool SetWhiteness(int val);
   
   int ReadImage();

   QImage *GetRGB(int offset = 0) const;
   QImage *GetY(int offset = 0) const;
   QImage *GetU(int offset = 0) const;
   QImage *GetV(int offset = 0) const;

signals:   
   /** A new image is available. */
   void Notify();
   /** The size has been changed. */
   void Resized(const QSize &new_size);
   /** The input channel has changed. */
   void ChangedInput(int input);
   /** The frequency or norm changed. */
   void ChangedTunerNorm(int tuner, int norm);
   /** An error occured. err_num is one of the errno values. */
   void Error(int err_num);
};

#endif
