/*
 * options.h - configuration settings
 * Copyright (c) 2008, NLnet Labs. All rights reserved.
 * This software is open source.
 * For license see doc/LICENSE.
 */

#ifndef OPTIONS_H
#define OPTIONS_H

#include <config.h>

#include "trust_anchors.h"

typedef struct strlist_struct strlist_t;
struct strlist_struct {
	const char*		str;
	strlist_t*		next;
};

typedef struct options_struct options_t;
struct options_struct {
	char*		statefile;		/* filename that stores DNSKEY states */
	char*		workingdir;		/* working directory */
	char*		logfile;		/* log filename */
	char*		pidfile;		/* pid filename */
	char*		roothints;      /* filename that stores root hints */
	char*		resolvconf;     /* resolver configuration filename */
	char*		unboundconf;	/* configure resolving unbound-like */

	uint32_t	add_holddown;   /* add holdddown time */
	uint32_t	del_holddown;   /* del holdddown time */
	uint32_t	keep_missing;   /* how long keep MISSING DNSKEYs */

	/* filename that stores resolver pid and reload cmd */
	strlist_t*	resolvers;
	strlist_t*	reloadcmd;
	/* configured trust anchors */
	strlist_t*	trustanchors;
	strlist_t*	trustanchorfiles;
	strlist_t*	trustedkeysfiles;

	uint8_t		verbosity;	/* verbosity level */
	uint8_t		verbosity_set;	/* verbosity set */
	uint8_t		use_syslog; /* use syslog */
	uint8_t		daemonize;	/* run as daemon */
	uint8_t		tp_deletion;	/* subordinate trust point deletion */
	uint8_t		keep_start;	/* keep START DNSKEYs */
	uint8_t		keep_removed;	/* keep REMOVED DNSKEYs */
	uint8_t		pending_count;	/* how many checks when pending */
	uint8_t		address_family; /* IP version */
	uint8_t		alert_missing;  /* alert when keys are MISSING */
};

/* Initialize options
 * @return: initialized options
 */
options_t* initialize_options(void);

/* Function to set a string-based option
 * @param options: the options
 * @param opt: name of the option
 * @param val: value of the option
 */
int set_cfg_option_str(options_t* options, const char* opt, const char* val);

/* Function to set a integer-based option
 * @param options: the options
 * @param opt: name of the option
 * @param val: value of the option
 */
int set_cfg_option_int(options_t* options, const char* opt, int val);

/* Load configuration settings
 * @param options: the options
 * @param file: the configfile
 * @return: status
 */
int load_cfg_settings(options_t* options, const char* file);

/* Function to free options
 * @param options: the options
 */
void free_options(options_t* options);

/* Print options
 * @param options: the options
 */
#ifdef TDEBUG
void debug_print_options(options_t* options);
#endif /* TDEBUG */

/* Load state
 * @param trustpoints: storage
 * @param file: file that stores state
 * @return: status
 */
int load_state(tp_set_t* trustpoints, char* file);

/* Load trust anchor options
 * @param trustpoints: storage
 * @param list: list of trust anchors
 * @return: status
 */
int load_trustanchor_options(tp_set_t* trustpoints, strlist_t* list);

/* Load trust anchor files
 * @param trustpoints: storage
 * @param list: list of files
 * @return: status
 */
int load_trustanchor_files(tp_set_t* trustpoints, strlist_t* list);

/* Load trusted keys files
 * @param trustpoints: storage
 * @param list: list of files
 * @return: status
 */
int load_trustedkeys_files(tp_set_t* trustpoints, strlist_t* list);

/* Update state
 * @param trustpoints: storage
 * @param file: file that stores state
 * @param options: options
 * @return: status
 */
int update_state(tp_set_t* trustpoints, options_t* options);

/* Update trust anchor files
 * @param trustpoints: storage
 * @param options: options
 * @return: status
 */
int update_trustanchor_files(tp_set_t* trustpoints, options_t* options);

/* Update trusted keys files
 * @param trustpoints: storage
 * @param list: list of files
 * @return: status
 */
int update_trustedkeys_files(tp_set_t* trustpoints, strlist_t* list);

#endif /* OPTIONS_H */
