// This file is part of the AspectC++ compiler 'ac++'.
// Copyright (C) 1999-2003  The 'ac++' developers (see aspectc.org)
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __Phase1_h__
#define __Phase1_h__

#include <string>
#include <vector>

#include "ACModel/Elements.h"

#include "IncludeGraph.h"
#include "CodeWeaver.h"
#include "PointCutSearcher.h"
#include "ACPreprocessor.h"

class ModelBuilder;
class CodeWeaver;

namespace Puma {
  class CProject;
  class Unit;
  class FileUnit;
  class TokenProvider;
  class ErrorStream;
  class Token;
}

// This class performs preliminary parsing of a translation unit. Its
// purpose is to find all aspect-oriented constructs in the code,
// store information about it in the translation unit's model, and
// remove or replace the aspect-oriented code. Thereby, the main
// parsing can be performed by an ordinary C++ parser without
// token and grammar extensions.

class Phase1 : public PointCutSearcher, private ACPreprocessor {

  // Helper classes to create a simple syntax tree while parsing
  struct Node {
    enum { ELEMENT_POINTCUT, ELEMENT_CLASS_ALIAS,
      SCOPE_NAMESPACE, SCOPE_CLASS, SCOPE_STRUCT, SCOPE_ASPECT } _kind;
    bool is_scope() const { return _kind >= SCOPE_NAMESPACE; }
    bool is_aspect() const { return _kind == SCOPE_ASPECT; }
    bool is_class() const { return _kind == SCOPE_CLASS; }
    bool is_struct() const { return _kind == SCOPE_STRUCT; }
    bool is_namespace() const { return _kind == SCOPE_NAMESPACE; }
    bool is_class_or_struct() const { return is_class() || is_struct(); }
    std::string _name;
    ACM_Name *_jpm_link;
    bool operator < (const Node &compared) const { return _name < compared._name; }
  };
  struct Element : Node {
    Node *_referred; // for class aliases
  };
  struct Scope : Node {
    Scope *_parent_scope;
    bool _is_slice;
    std::set<Scope> _sub_scopes;
    std::set<Element> _elements;
    std::list<Scope*> _search_scopes; // base classes for classes/aspects
                                      // namespaces for namespaces
    void dump (int indent = 0) const {
      for (int i = indent; i >= 0; i--) { cout << "  "; }
      if (_is_slice)
        cout << "slice ";
      switch (_kind) {
      case SCOPE_NAMESPACE: cout << "namespace"; break;
      case SCOPE_CLASS:     cout << "class"; break;
      case SCOPE_ASPECT:    cout << "aspect"; break;
      default:              cout << "unknown"; break;
      }
      cout << " " << _name << ": ";
      for (std::list<Scope*>::const_iterator i = _search_scopes.begin ();
          i != _search_scopes.end (); ++i)
        cout << " " << (*i)->_name;
      for (std::set<Element>::const_iterator i = _elements.begin ();
          i != _elements.end (); ++i)
        cout << " elem " << (*i)._name;
      cout << endl;
      for (std::set<Scope>::const_iterator i = _sub_scopes.begin ();
          i != _sub_scopes.end (); ++i)
        (*i).dump (indent + 1);
    }
  };

  Scope _root_scope;
  Scope *_curr_scope;
  ModelBuilder &_jpm;
  CodeWeaver &_code_weaver;
  bool _in_template;

  void skip_block (int open, int close, bool inclusive = true);
  void skip_round_block (bool inclusive = true);
  void skip_curly_block (bool inclusive = true);
  void skip_template_params ();

  // internal registration functions
  ACM_Name *register_scope (Scope &scope);
  void register_base_classes (Scope &cls);

  // internal parse functions
  void parse_scope (Scope &scope);
  void parse_qual_name (std::vector<std::string> &names,
      bool &root_qualified, bool &contains_template_id);
  void parse_base_clause (Scope &class_scope);
  void parse_base_spec (Scope &class_scope);
  void parse_aspectof_function (const string &prot);
  void parse_using_directive (Scope &scope);
  void parse_pointcut_def (Scope &scope);
  void parse_advice_def (Scope &scope, int &advice_no, int &intro_no, int &order_no,
      int &anon_slice_no, const std::string &prot);
  void parse_advice_body (AdviceCodeContext &context, int open, int close);

  // internal name lookup functions
  Node *lookup_name (Scope &scope, bool root_qualified, std::vector<std::string> &names);
  Node *lookup_name_in_scope (Scope &scope, std::vector<std::string> &names, int depth = 0);
  Scope *lookup_scope (Scope &scope, bool root_qualified, std::vector<std::string> &names);
  Scope *find_slice (Scope &scope, Puma::Unit *rec);
  ACM_Pointcut *lookup_pct_func (bool root_qualified, std::vector<std::string> &qual_name);

  // internal helper functions
  PointCutExpr *create_pct_expr_tree(const string &pct, ACM_Name *scope, Puma::Token *pos);
  void handle_slice_member (Scope &scope, Puma::Unit *rec,
      Puma::Token *from, Puma::Token *to);
  bool is_visible_scope (Scope &scope, const std::string &name);

public:
  Phase1 (ModelBuilder &jpm, Puma::Unit &unit, Puma::CProject &project, CodeWeaver &code_weaver) :
	ACPreprocessor(unit, project), _curr_scope (0), _jpm (jpm),
    _code_weaver (code_weaver), _in_template (false) {}
  bool run (IncludeGraph &include_graph);
};

#endif // __Phase1_h__
