------------------------------------------------------------------------------
--                                                                          --
--                          GNATCHECK COMPONENTS                            --
--                                                                          --
--                   G N A T C H E C K . C O M P I L E R                    --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                     Copyright (C) 2005-2008, AdaCore                     --
--                                                                          --
-- GNATCHECK  is  free  software;  you can redistribute it and/or modify it --
-- under terms of the  GNU  General Public License as published by the Free --
-- Software Foundation;  either version 2, or ( at your option)  any  later --
-- version.  GNATCHECK  is  distributed in the hope that it will be useful, --
-- but  WITHOUT  ANY  WARRANTY;   without  even  the  implied  warranty  of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General --
-- Public License for more details.  You should have received a copy of the --
-- GNU  General Public License distributed with GNAT; see file  COPYING. If --
-- not,  write to the  Free Software Foundation,  51 Franklin Street, Fifth --
-- Floor, Boston, MA 02110-1301, USA.                                       --
--                                                                          --
-- GNATCHECK is maintained by AdaCore (http://www.adacore.com).             --
--                                                                          --
------------------------------------------------------------------------------

--  This package defines "rules" for getting the information for the GNATCHECK
--  report from the check performed by the compiler.

with GNAT.OS_Lib;      use GNAT.OS_Lib;

package Gnatcheck.Compiler is

   --------------------------------------------------------
   -- Using in GNATCHECK checks prformed by the compiler --
   --------------------------------------------------------

   --  The compiler already performs a lot of checks that may be of interest
   --  for code sertification and, therefore, it would be nice to perform
   --  these checks as a part of GNATCHECK run and to incorporate the results
   --  of these checks into the GNATCHECK report

   --  There are three kinds of the compiler-performed checks:
   --
   --  - checks initiated by the prarma Restriction_Warnings
   --  - style checks;
   --  - checks resulted in general (non-style) warnings;

   --  GNATCHECK uses three parametrized rules to activate these checks:
   --  RESTRICTIONS, STYLE_CHECKS and WARNINGS, the parameters of these rules
   --  specify which restrictions, style rules and other conditions covered by
   --  compiler (non-style) warnings should be checked

   type Compiler_Message_Kinds is
     (Not_A_Compiler_Nessage,
      General_Warning,
      Style,
      Restriction);
   --  Describes possible kinds of compiler messages analysed by gnatcheck

   Use_gnaty_Option : Boolean := False;
   Use_gnatw_Option : Boolean := False;
   --  Boolean flags indicating if the corresponding GNAT option should be
   --  set in the compiler call used to generate the tree

   Check_Restrictions : Boolean := False;
   --  Boolean flag indicating if the configuration file with
   --  Restriction_Warnings pragmas generated by gnatcheck chould be used in
   --  the compiler call used to generate the tree

   Suppess_Compiler_Check : Boolean;
   --  Flag indicating if we have to suppress all the compiler checks when
   --  creating a tree file.

   procedure Set_Compiler_Checks;
   --  Sets the values of Use_gnaty_Option, Use_gnatw_Option and
   --  Check_Restrictions on the base of setting made by the warnings, style
   --  checks and restrictions gnatcheck options

   Analyze_Compiler_Output : Boolean := False;
   --  Boolean flag indicating if gnatcheck shouldl analyze the compiler output

   Restriction_Config_File : String := "gnatcheck_restrictions";
   --  The name of the temporaty file to place the restriction pargmas into
   --  There can be no name conflict with any name created in temporary
   --  directory!

   procedure Process_Restriction_Param
     (Parameter : String;
      Enable    : Boolean);
   --  Processes a parameter of a restriction (the restriction "rule") passed
   --  as an actual for Parameter. Only a single parameter should be passed,
   --  not a set of parameters separated by commas from the rule option.
   --  Depending on the value of Enable, the corresponding restriction is set
   --  ON or OFF.

   procedure Process_Style_Check_Param (Param : String);
   --  Processes a parameter of a style check (the style_check "rule") passed
   --  as an actual for Param. Only a single parameter should be passed, not a
   --  set of parameters separated by comma(s) from the rule option.

   procedure Process_Warning_Param (Param : String);
   --  Processes a parameter of a warning (the warning "rule") passed as an
   --  actual for Param. Only a single parameter should be passed, not a set of
   --  parameters separated by comma(s) from the rule option. Depending on the
   --  value of Enable, the corresponding warning(s) is (are) set ON or OFF

   procedure Create_Restriction_Pragmas_File;
   --  Creates in the temporary directory the configuration file containing
   --  the needed restriction pragmas

   function Get_Style_Option return String;
   --  Returns the '-gnatyxxx' option to be used in the compiler call

   function Get_Warning_Option return String;
   --  Returns the '-gnatwxxx' option to be used in the compiler call, the
   --  option represents individual warning settings for the options that are
   --  not using the dot notation. The returned string first contains disabled
   --  warnings (upper case letters), and then - enabled warnings, because
   --  some options for disabling warnings may also implicitly  disable some
   --  other warnings. For example '-gnatwU' disables warnings activated by
   --  '-gnatwf'.

   --  The following routines implement the iterator for getting the warning
   --  options that are using the dot notation, we put them separately for
   --  readability reasons.
   procedure Reset_Dot_Warning_Opt_Iterator;
   function Done return Boolean;
   function Next_Dot_Warning_Option return String;
   --  This function returns the string of the form "-gnatw.X", where 'X' is
   --  a valid character for the warning option

   procedure Print_Active_Restrictions (Ident_Level : Natural := 0);
   --  Prints out the Restriction Identifiers of the checks that are set active
   --  for the given gnatcheck call (with the corresponding parameter value, if
   --  any). Restriction identifiers are printed in a column, Ident_Level is
   --  used to control the indentation/

   Compiler_Out_FD        : File_Descriptor := Invalid_FD;
   Compiler_Out_File_Name : Temp_File_Name;
   --  Temporary file used to redirect the compiler output into.

   function Compiler_Out_File_Name_String return String;
   --  Returns Compiler_Out_File_Name if gnatcheck has to analyze the compiler
   --  output. Otherwise returns null string

   procedure Analyze_Compiler_Warnings (Compiler_Out : Temp_File_Name);
   --  Parses the file with compiler output and places the corresponding
   --  diagnoses into the gnatcheck report file

end Gnatcheck.Compiler;
