#include <signal.h>

#include <volume.h>
#include <asdutil.h>

#include <gc.h>
#include <namespace.h>
#include <source.h>
#include <asd.h>
#include <source-default.h>
#include <thread-manager.h>
#include <conjunction.h>

static void _source_free(gpointer p);

Source* source_new(gchar *shortname, gchar *name, gboolean ns_reg)
{
  Source *s;

  if (ns_reg)
    if (!namespace_register(shortname))
      return NULL;

  g_assert(s = g_new(Source, 1));

  s->shortname = g_strndup(shortname, ASD_SHORTNAME_LENGTH);
  s->name = g_strndup(name, ASD_NAME_LENGTH);
  s->type = "ABSTRACT";

  s->sample_type = default_sample_type;
  volume_max(&s->volume);

  s->mode = SOURCE_DISABLED;
  s->flags = SOURCE_PUSH_WAIT;

  s->thread_running = FALSE;

  s->throughput = throughput_new();

  s->origin_id = origin_register_id();

  s->output_block_queues = NULL;
  s->output_block_queues_mutex = recursive_mutex_new();

  s->direct_sink = NULL;

  notify_init(&s->notify);

  s->private_data = NULL;
  s->open         = source_default_open_impl;
  s->close        = source_default_close_impl;
  s->reopen       = source_default_reopen_impl;
  s->read         = source_default_read_impl;
  s->read_block   = source_default_read_block_impl;
  s->free_private = source_default_free_private_impl;
  s->save_private = source_default_save_private_impl;

  namespace_set(s->shortname, s, NAMESPACE_SOURCE);

  return gc_add(s, _source_free, shortname);
}

static void _source_free(gpointer p)
{
  Source *s;
  g_assert(s = (Source*) p);

  namespace_unregister(s->shortname);

  if (s->free_private)
    s->free_private(s);

  notify_done(&s->notify);

  g_assert(s->output_block_queues == NULL);
  recursive_mutex_free(s->output_block_queues_mutex);

  g_free(s->shortname);
  g_free(s->name);

  gc_ref_dec(s->throughput);
  origin_unregister_id(s->origin_id);
}

void source_add_output_block_queue(Source *s, BlockQueue *b)
{
  g_assert(s && b);

  recursive_mutex_lock(s->output_block_queues_mutex);
  g_assert(!g_slist_find(s->output_block_queues, b));
  block_queue_set_pop_notify(b, &s->notify);
  block_queue_set_push_wait(b, s->flags & SOURCE_PUSH_WAIT);
  block_queue_set_push_enabled(b, s->mode == SOURCE_RUNNING);
  s->output_block_queues = g_slist_prepend(s->output_block_queues, gc_ref_inc(b));
  recursive_mutex_unlock(s->output_block_queues_mutex);

  notify_signal(&s->notify);
}

void source_remove_output_block_queue(Source *s, BlockQueue *b)
{
  g_assert(s && b);

  recursive_mutex_lock(s->output_block_queues_mutex);
  g_assert(g_slist_find(s->output_block_queues, b));
  s->output_block_queues = g_slist_remove(s->output_block_queues, b);
  block_queue_set_pop_notify(b, NULL);
  block_queue_set_push_enabled(b, FALSE);
  gc_ref_dec(b);
  recursive_mutex_unlock(s->output_block_queues_mutex);

  notify_signal(&s->notify);
}

void source_output_block_queue_death_proc(gpointer p, BlockQueue*b)
{
  source_remove_output_block_queue((Source*) p, b);
}

gboolean source_output_write_block(Source *s, Block *b)
{
  GSList *l;
  gboolean ok = FALSE;
  
  g_assert(s && b);

  pthread_cleanup_push(MAKE_CLEANUP_HANDLER(recursive_mutex_unlock), s->output_block_queues_mutex);
  recursive_mutex_lock(s->output_block_queues_mutex);  

  l = s->output_block_queues;
  while (l)
    {
      if (block_queue_push((BlockQueue*) l->data, b))
	ok = TRUE;
      l = l->next;
    }

  pthread_cleanup_pop(1);
  return ok;
}

/* void source_block_queues_empty(Source *s) */
/* { */
/*   GSList *l; */
/*   g_assert(s); */

/*   pthread_cleanup_push(MAKE_CLEANUP_HANDLER(recursive_mutex_unlock), s->block_queues_mutex); */
/*   recursive_mutex_lock(s->output_block_queues_mutex);   */

/*   l = s->block_queues; */
/*   while (l) */
/*     { */
/*       block_queue_empty((BlockQueue*) l->data); */
/*       l = l->next; */
/*     } */
  
/*   pthread_cleanup_pop(1); */
/* } */

void source_output_block_queues_enable(Source *s, gboolean b)
{
  GSList *l;
  g_assert(s);

  pthread_cleanup_push(MAKE_CLEANUP_HANDLER(recursive_mutex_unlock), s->output_block_queues_mutex);
  recursive_mutex_lock(s->output_block_queues_mutex);  

  l = s->output_block_queues;
  while (l)
    {
      block_queue_set_push_enabled((BlockQueue*) l->data, b);
      l = l->next;
    }

  pthread_cleanup_pop(1);
}

void source_output_block_queues_wait_until_empty(Source *s)
{
  GSList *l;
  g_assert(s);

  pthread_cleanup_push(MAKE_CLEANUP_HANDLER(recursive_mutex_unlock), s->output_block_queues_mutex);
  recursive_mutex_lock(s->output_block_queues_mutex);  

  l = s->output_block_queues;
  while (l)
    {
      block_queue_wait_until_empty((BlockQueue*) l->data);
      l = l->next;
    }
  
  pthread_cleanup_pop(1);
}


static void _source_process_cleanup(gpointer p)
{
  Source *s;
  g_assert(s = (Source*) p);

  if (s->close)
    s->close(s);
}

static void _source_process(Source *s)
{
  gboolean quit;
  g_assert(s);

  if (s->open)
    if (!s->open(s))
      return;

  g_assert(s->read_block);

  pthread_cleanup_push(_source_process_cleanup, s);

  for (;;)
    {
      for (;;)
	{
	  Block *b;
	  gboolean ok;

	  quit = (b = s->read_block(s)) ? FALSE : TRUE;
	  if (quit) break;

          b = block_volume(b, &s->volume);

          origin_set(&b->origin, s->origin_id, TRUE);
          latency_set(&b->latency, s->origin_id);

	  pthread_cleanup_push(gc_ref_dec, b);
	  ok = source_output_write_block(s, b);
	  pthread_cleanup_pop(1);
	  if (!ok) break;
	}
      
      if (quit) break;
	
      notify_wait(&s->notify);
    }

  pthread_cleanup_pop(1);

  //  g_message("Immediate stop: %s", s->flags & SOURCE_IMMEDIATE_STOP ? "Yes" : "No");

  if (!(s->flags & SOURCE_IMMEDIATE_STOP))
    source_output_block_queues_wait_until_empty(s);

}

static void _source_thread_cleanup(gpointer p)
{
  Source *s;
  g_assert(s = (Source*) p);

  if (s->flags && SOURCE_AUTOCLEAN)
    conjunction_remove_item(s);

  thread_unregister(pthread_self());
  s->thread_running = FALSE;

  gc_ref_dec(s);
}

static void* _source_thread(void *arg)
{
  sigset_t ss;
  Source *s;

  g_assert(s = (Source*) arg);

  s->thread_running = TRUE;
  thread_register(pthread_self());
  pthread_cleanup_push(_source_thread_cleanup, s);

  pthread_sigmask(SIG_BLOCK, NULL, &ss);
  sigaddset(&ss, SIGINT);
  sigaddset(&ss, SIGQUIT);
  //  sigaddset(&ss, SIGHUP);
  pthread_sigmask(SIG_BLOCK, &ss, NULL);

  pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED, NULL);
  pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);

  _source_process(s);

  pthread_cleanup_pop(1);

  pthread_detach(pthread_self());

  return NULL;
}

void source_start(Source *s)
{
  g_assert(s);
  g_assert(!s->thread_running);
  gc_ref_inc(s);
  s->thread_running = TRUE;
  g_assert(!pthread_create(&s->thread, NULL, _source_thread, (void*) s));
}

void source_stop(Source *s)
{  
  g_assert(s);
  if (s->thread_running)
    {
      pthread_cancel(s->thread);
      pthread_join(s->thread, NULL);
    }
}

void source_kill(Source *s)
{
  g_assert(s);
  source_stop(s);
  conjunction_remove_item(s);
}

void source_set_mode(Source *s, SourceMode m)
{
  g_assert(s);
  
  s->mode = m;
  
  source_output_block_queues_enable(s, m == SOURCE_RUNNING);
}

gulong source_get_latency(Source *s)
{
  g_assert(s);
  
  return latency_global_get(s->origin_id);
}

gboolean source_direct(Source *s, gboolean enable, Sink* sink)
{
  g_assert(s);

  if (enable)
    g_assert(sink);

  if (!(s->flags & SOURCE_DIRECT_SUPPORTED))
    return FALSE;

  if ((enable && s->mode != SOURCE_RUNNING) || (!enable && s->mode != SOURCE_DIRECT))
    return FALSE;

  s->direct_sink = sink;

  source_set_mode(s, enable ? SOURCE_DIRECT : SOURCE_RUNNING);
  
  return TRUE;
}

void source_set_volume(Source *s, Volume *v)
{
  g_assert(s && v);
  s->volume = *v;
}
