#include <pthread.h>

#include <asdutil.h>

#include <block-queue.h>
#include <gc.h>

static void _block_queue_free(gpointer p);

BlockQueue* block_queue_new(guint l, guint hold)
{
  BlockQueue *b = g_new(BlockQueue, 1);
  
  b->mutex = g_mutex_new();
  b->size = (l <= 0) ? 4 : l ;
  b->blocks = g_new0(Block*, b->size);
  
  b->push_index = b->pop_index = 0;
  b->empty = TRUE;

  b->hold = hold <= 1 ? 1 : hold;
  b->hold = MIN(b->hold, b->size);
  b->streaming = FALSE;

  g_message("Queue length/hold set to %i/%i.", b->size, b->hold);

  b->push_wait = b->pop_wait = FALSE;
  b->push_enabled = b->pop_enabled = FALSE;

  notify_init(&b->pop_notify);
  notify_init(&b->push_notify);

  b->user_pop_notify = b->user_push_notify = NULL;

  return gc_add(b, _block_queue_free, "block-queue");
}

static void _notify_pop(BlockQueue* b)
{
  notify_signal(&b->pop_notify);
  
  if (b->user_pop_notify) 
    notify_signal(b->user_pop_notify);
}

static void _notify_push(BlockQueue* b)
{
  notify_signal(&b->push_notify);
  
  if (b->user_push_notify) 
    notify_signal(b->user_push_notify);
}

static void _block_queue_free(gpointer p)
{
  BlockQueue *b;
  g_assert(b = (BlockQueue*) p);
  
  block_queue_empty(b);
  g_free(b->blocks);

  notify_done(&b->pop_notify);
  notify_done(&b->push_notify);

  g_mutex_free(b->mutex);

  g_free(b);
}

static guint _block_queue_length(BlockQueue *b)
{
  g_assert(b);

  if (b->empty)
    return 0;
  else if (b->push_index > b->pop_index)
    return b->push_index - b->pop_index;
  else
    return (guint) ((gint)b->size - ((gint)b->pop_index - (gint)b->push_index));
}

guint block_queue_length(BlockQueue *b)
{
  guint r;
  g_assert(b);

  g_mutex_lock(b->mutex);
  r = _block_queue_length(b);
  g_mutex_unlock(b->mutex);

  return r;
}

static gboolean _block_queue_is_empty_with_hold(BlockQueue *b)
{
  g_assert(b);
  return _block_queue_length(b) < (b->streaming ? 1 : b->hold);
}

static gboolean _block_queue_is_full(BlockQueue *b)
{
  g_assert(b);
  return !b->empty && (b->push_index == b->pop_index);
}

Block *block_queue_pop(BlockQueue *b)
{
  Block *e = NULL;
  g_assert(b);

  g_mutex_lock(b->mutex);

  while (b->push_enabled && b->pop_wait && _block_queue_is_empty_with_hold(b))
    {
      g_mutex_unlock(b->mutex);
      notify_wait(&b->push_notify);
      g_mutex_lock(b->mutex);
    }

  if (!_block_queue_is_empty_with_hold(b))
    {
      g_assert(e = b->blocks[b->pop_index]);
      b->blocks[b->pop_index++] = NULL;
      
      while (b->pop_index >= b->size)
	b->pop_index -= b->size;
      
      if (b->pop_index == b->push_index)
	b->empty = TRUE;

      b->streaming = TRUE;

      _notify_pop(b);
    }
  else
    b->streaming = FALSE;
  
  g_mutex_unlock(b->mutex);

  return e;
}

gboolean block_queue_push(BlockQueue *b, Block *e)
{
  gboolean ok = FALSE;
  g_assert(b && e);

  if (!b->pop_enabled)
    return FALSE;

  g_mutex_lock(b->mutex);

  while (b->push_wait && _block_queue_is_full(b))
    {
      g_mutex_unlock(b->mutex);
      notify_wait(&b->pop_notify);
      g_mutex_lock(b->mutex);
    }

  if (!_block_queue_is_full(b))
    {
      b->blocks[b->push_index++] = gc_ref_inc(e);
      
      while (b->push_index >= b->size)
	b->push_index -= b->size;
      
      b->empty = FALSE;
      ok = TRUE;

      _notify_push(b);
    }

  g_mutex_unlock(b->mutex);

  return ok;
}

gboolean block_queue_is_empty(BlockQueue *b)
{
  g_assert(b);

  return b->empty;
}

void block_queue_set_hold(BlockQueue *b, guint hold)
{
  b->hold = hold;
  g_mutex_lock(b->mutex);
  _notify_pop(b);
  g_mutex_unlock(b->mutex);
}

void block_queue_set_push_wait(BlockQueue *b, gboolean wait)
{
  b->push_wait = wait;
  g_mutex_lock(b->mutex);
  _notify_push(b);
  g_mutex_unlock(b->mutex);
}

void block_queue_set_pop_wait(BlockQueue *b, gboolean wait)
{
  b->pop_wait = wait;
  g_mutex_lock(b->mutex);
  _notify_pop(b);
  g_mutex_unlock(b->mutex);
}

void block_queue_set_pop_enabled(BlockQueue *b, gboolean e)
{
  b->pop_enabled = e;
  g_mutex_lock(b->mutex);
  _notify_push(b);
  _notify_pop(b);
  g_mutex_unlock(b->mutex);
}

void block_queue_set_push_enabled(BlockQueue *b, gboolean e)
{
  b->push_enabled = e;
  g_mutex_lock(b->mutex);
  _notify_push(b);
  _notify_pop(b);
  g_mutex_unlock(b->mutex);
}

void block_queue_empty(BlockQueue *b)
{
  guint i;
  g_mutex_lock(b->mutex);
  
  if (!b->empty)
    {
      for (i = 0; i < b->size; i++)
	{
	  if (b->blocks[i])
	    gc_ref_dec(b->blocks[i]);
	  
	  b->blocks[i] = NULL;
	}	  
      
      b->push_index = b->pop_index = 0;
      b->empty = TRUE;
      b->streaming = FALSE;
    }

  _notify_push(b);
  _notify_pop(b);

  g_mutex_unlock(b->mutex);
}

void block_queue_resize(BlockQueue *b, guint s) // FIXME
{
  Block** new_blocks;
  g_assert(b && s);
  
  g_mutex_lock(b->mutex);
  new_blocks = g_new0(Block*, s);
  
  if (b->empty)
    b->push_index = b->pop_index = 0;
  else
    {
      guint i, j, k, l;
      l = _block_queue_length(b);

      k = 0;
      for (i = 0; i < MIN(l, s); i++)
	{
	  j = i+b->pop_index;
	  while (j >= b->size)
	    j -= b->size;

	  new_blocks[k++] = b->blocks[j];
	}

      b->pop_index = 0;

      b->push_index = i;
      while (b->push_index >= s)
	b->push_index -= s;
    }

  g_free(b->blocks);
  b->blocks = new_blocks;
  b->size = s;

  _notify_push(b);
  _notify_pop(b);

  g_mutex_unlock(b->mutex);
}

void block_queue_set_push_notify(BlockQueue *b, Notify *notify)
{
  g_assert(b);
  g_mutex_lock(b->mutex);
  b->user_push_notify = notify;
  g_mutex_unlock(b->mutex);
}

void block_queue_set_pop_notify(BlockQueue *b, Notify *notify)
{
  g_assert(b);
  g_mutex_lock(b->mutex);
  b->user_pop_notify = notify;
  g_mutex_unlock(b->mutex);
}

void block_queue_wait_until_empty(BlockQueue *b)
{
  g_assert(b);

  g_mutex_lock(b->mutex);
  while (b->pop_enabled && !_block_queue_is_empty_with_hold(b))
    {
      g_mutex_unlock(b->mutex);
      notify_wait(&b->pop_notify);
      g_mutex_lock(b->mutex);
    }
  g_mutex_unlock(b->mutex);
}
