#include <signal.h>
#include <unistd.h>
#include <sys/time.h>
//#include <string.h>

#include <sample-type.h>
#include <auth.h>
#include <esound-auth.h>

#include <asd.h>
#include <conjunction.h>
#include <source.h>
#include <sink.h>
#include <thread-manager.h>

#include "protocols/protocol-asd-unix.h"
#include "protocols/protocol-asd-inet.h"
#include "protocols/protocol-esound-unix.h"
#include "protocols/protocol-esound-inet.h"
#include "protocols/protocol-http-inet.h"

#include "sources/source-fifo.h"
#include "sinks/sink-fifo.h"
#include "sources/source-dsp.h"
#include "sinks/sink-dsp.h"

SampleType default_sample_type = { 2, 16, 1, 0, 44100 };
guint default_block_size = 2048*2*2;
pid_t main_pid;

gchar* asd_version = "asd/0.2pre Lennart Poettering $Date: 2001/06/07 11:03:12 $";

static volatile gboolean asd_quit = FALSE;

static void _setup()
{
  Source *source, *source2;
  Sink *sink, *sink2, *sink3;
  
  g_assert(source = source_dsp_new(ASD_DEFAULT_CAPTURE_SOURCE, "/dev/dsp"));
  g_assert(source2 = source_fifo_new("fifo0", "/tmp/asd-play"));

  g_assert(sink = sink_dsp_new(ASD_DEFAULT_PLAY_SINK, "/dev/dsp"));
  g_assert(sink2 = sink_fifo_new("fifo1", "/tmp/asd-capture"));
  g_assert(sink3 = sink_fifo_new("fifo2", "/tmp/asd-monitor"));
  
  link_source_sink(source, sink2, 0, 0);
  link_source_sink(source2, sink, 0, 0);

  link_sink_sink(sink, sink3, 0, 0);

  source_start(source);
  source_start(source2);
  sink_start(sink);
  sink_start(sink2);
  sink_start(sink3);

  gc_ref_dec(source);
  gc_ref_dec(source2);
  gc_ref_dec(sink);
  gc_ref_dec(sink2);
  gc_ref_dec(sink3);
}

static void sighandler(int sig)
{
  g_message("Signal '%s' (%i) recieved.", sys_siglist[sig], sig);
  asd_quit = TRUE;
  signal(sig, sighandler);
}

static void _init()
{
  main_pid = getpid();
  signal(SIGPIPE, SIG_IGN);    siginterrupt(SIGPIPE, 1);
  signal(SIGHUP, SIG_IGN);     siginterrupt(SIGHUP, 1);

  signal(SIGINT, sighandler);  siginterrupt(SIGINT, 1);
  signal(SIGQUIT, sighandler); siginterrupt(SIGQUIT, 1);
  signal(SIGTERM, sighandler); siginterrupt(SIGTERM, 1);

  g_thread_init(NULL);

  block_init();
  auth_init(TRUE);
  esound_auth_init(TRUE);

  protocol_asd_inet_init(0);
  protocol_asd_unix_init(NULL);
  protocol_esound_inet_init(0);
  protocol_esound_unix_init();
  protocol_http_inet_init(0);

  _setup();
}

static void _done()
{
  thread_killall();

  conjunction_reset();

  protocol_asd_unix_done();
  protocol_asd_inet_done();
  protocol_esound_unix_done();
  protocol_esound_inet_done();
  protocol_http_inet_done();

  auth_done();
  esound_auth_done();
  
  block_done();
  gc_cleanup();
}

static void _loop()
{
  g_message("Server startup complete.");

  while(!asd_quit)
    {
      fd_set fds;

      FD_ZERO(&fds);
      if (protocol_asd_inet_enabled())
	FD_SET(protocol_asd_inet_get_fd(), &fds);
      if (protocol_asd_unix_enabled())
        FD_SET(protocol_asd_unix_get_fd(), &fds);
      if (protocol_esound_inet_enabled())
        FD_SET(protocol_esound_inet_get_fd(), &fds);
      if (protocol_esound_unix_enabled())
        FD_SET(protocol_esound_unix_get_fd(), &fds);
      if (protocol_http_inet_enabled())
        FD_SET(protocol_http_inet_get_fd(), &fds);

      if (select(FD_SETSIZE, &fds, NULL, NULL, NULL) >= 1)
        {
          if (FD_ISSET(protocol_asd_inet_get_fd(), &fds))
            protocol_asd_inet_accept();
          if (FD_ISSET(protocol_asd_unix_get_fd(), &fds))
            protocol_asd_unix_accept();
          if (FD_ISSET(protocol_esound_inet_get_fd(), &fds))
            protocol_esound_inet_accept();
          if (FD_ISSET(protocol_esound_unix_get_fd(), &fds))
            protocol_esound_unix_accept();
          if (FD_ISSET(protocol_http_inet_get_fd(), &fds))
            protocol_http_inet_accept();
	}
    }
  g_message("Server shutting down.");
}

gint main(int argc, char *argv[])
{
  _init();
  _loop();
  _done();
  return 0;
}
