#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <getopt.h>
#include <asdlib.h>

gchar speaker[256] = "";

void help()
{
  g_print("Usage: %s ... [COMMAND]\n\n", g_get_prgname());
  g_print("Options:\n");
  g_print(" -h, --help                Show this help\n"
          " -s, --server=SERVERSPEC   Set server to SERVERSPEC\n"
          " -u, --usage               Show terse usage info\n\n");

  g_print("COMMAND maybe:\n"
          " lock                   lock the server for foreign access.\n"
          " unlock                 unlock the server.\n"
          " getvolume DEV          get the volume of sink/source DEV.\n"
          " setvolume DEV VOLSPEC  set the volume of sink/source DEV to VOLSPEC.\n"
          " source SOURCE          query information about SOURCE\n"
          " sink SINK              query information about SINK\n"
          " ls                     shows all sinks and sources\n"
          " info                   show general server info\n"
          " version                show version string\n\n");
  
  g_print("SERVERSPEC has the following format: PROTOCOL:SERVER\n"
          "PROTOCOL maybe 'unix:' for unix-socket, and 'inet:' for tcp-socket.\n"
          "SERVER specifies a unix-socket filename for unix-protocol, maybe ommitted for default.\n"
          "SERVER specifies a hostname for inet-protocol, maybe ommited for 'localhost'.\n"
          "You may add a port number to SERVER for inet-protocol by adding ':PORT' to it.\n\n");
  
  g_print("Example:\n"
          " unix:            connect via unix socket.\n"
          " unix:/tmp/gurke  connect via unix socket /tmp/gurke.\n"
          " inet:tomate:4711 connect via inet socket to host tomate on port 4711.\n"
          " inet:banane      connect via inet socket to host banane.\n"
          " inet:            connect via inet socket to localhost.\n\n");

  g_print("%s tries to evaluate $ASDSPEAKER, when no --server is specified.\n", g_get_prgname());

  exit(0);
}

void  usage()
{
  g_print("Usage: %s [-h] [-s SERVERSPEC] [-u] COMMAND\n", g_get_prgname());
  g_print("Commands are: lock, unlock, getvolume, setvolume, source, sink, ls, info, version\n");
  exit(0);
}

guint parse_args(int argc, char *argv[])
{
  static struct option long_options[] =
  {
    {"server",   required_argument, NULL, 's'},
    {"help",     no_argument,       NULL, 'h'},
    {"usage",    no_argument,       NULL, 'u'},
    {NULL, 0, NULL, 0}
  };

  gchar c;
  int long_index;

  while ((c = getopt_long(argc, argv, "s:hu", long_options, &long_index)) >= 0)
    {
      switch (c)
        {
        case 's' :
          g_assert(optarg);
          strncpy(speaker, optarg, sizeof(speaker));
          break;
        case 'h' :
          help();
        case '?' :
        case 'u' :
          usage();
        default:
          g_error("Not yet implemented!");
        }
    }
  
  return optind;
}

void _proc(ProtocolAsdListResponse* response, gpointer userdata)
{
  g_assert(response);
  
  g_print(" [%s] %s (%s)\n", response->shortname, response->name, response->type);
}

int main(int argc, char *argv[])
{
  AsdConnection *c;
  guint i;
  gchar *cmd = NULL;
  
  g_set_prgname(g_basename(argv[0]));

  i = parse_args(argc, argv);

  if (argc-i < 1)
    {
      if (strlen(g_get_prgname()) > 3) 
        cmd = g_get_prgname()+3;
    }
  else 
    cmd = argv[i++];

  if (!cmd)
    usage();

  if (!(c = asd_connection_new(speaker[0] ? speaker : NULL)))
    {
      asd_perror("Could not create connection");
      return 1;
    }

  if (strcmp(cmd, "lock") == 0)
    {
      if (!asd_lock(c, TRUE))
        asd_perror("Could not lock server");
    }
  else if (strcmp(cmd, "unlock") == 0)
    {
      if (!asd_lock(c, FALSE))
        asd_perror("Could not unlock server");
    }
  else if (strcmp(cmd, "getvolume") == 0)
    {
      if (argc-i < 1)
        g_printerr("You need to specify a sink/source.\n");
      else
        {
          Volume v;
          if (!asd_volume_get(c, argv[i], &v))
            asd_perror("Could not get device volume");
          else
            {
              gchar t[256];
              volume_to_string(&v, t, sizeof(t));
              g_print("Volume of device '%s' is %s\n", argv[i], t);
            }
        }
    }
  else if (strcmp(cmd, "setvolume") == 0)
    {
      if (argc-i < 2)
        g_printerr("You need to specifiy a sink/source and a volume.\n");
      else
        {
          Volume v;
          volume_parse(&v, argv[i+1]);
          if (!asd_volume_set(c, argv[i], v))
            asd_perror("Could not set device volume");
        }
    }
  else if (strcmp(cmd, "source") == 0)
    {
      if (argc-i < 1)
        g_printerr("You need to specifiy a source.\n");
      else
        {
          ProtocolAsdInfoSourceResponse info;
          if (!asd_info_source(c, argv[i], &info))
            asd_perror("Could not read source info");
          else
            {
              gchar t[256], v[256];

              sample_type_to_string(&info.sample_type, t, sizeof(t));
              volume_to_string(&info.volume, v, sizeof(v));
              
              g_print("      Source: %s\n"
                      "        Name: %s\n"
                      "        Type: %s\n"
                      " Sample-type: %s\n"
                      "      Volume: %s\n"
                      "        Mode: %s\n"
                      "       Flags:%s%s%s%s\n"
                      "  Throughput: %i kb/s\n"
                      "Avg. latency: %i msec\n"
                      "Byte counter: %i kb\n",
                      info.shortname,
                      info.name,
                      info.type,
                      t,
                      v,
                      info.mode == SOURCE_DISABLED ? "disabled" : info.mode == SOURCE_RUNNING ? "running" : info.mode == SOURCE_PAUSED ? "paused" : "direct",
                      info.flags & SOURCE_AUTOCLEAN ? " AUTOCLEAN" : "",
                      info.flags & SOURCE_PUSH_WAIT ? " PUSH_WAIT" : "",
                      info.flags & SOURCE_IMMEDIATE_STOP ? " IMMEDIATE_STOP" : "",
                      info.flags & SOURCE_DIRECT_SUPPORTED ? " DIRECT_SUPPORTED" : "",
                      info.throughput/1024,
                      info.latency,
                      info.byte_counter/1024); 
            }
        }
    }
  else if (strcmp(cmd, "sink") == 0)
    {
      if (argc-i < 1)
        g_printerr("You need to specifiy a sink.\n");
      else
        {
          ProtocolAsdInfoSinkResponse info;
          if (!asd_info_sink(c, argv[i], &info))
            asd_perror("Could not read sink info");
          else
            {
              gchar t[256], v[256];
              
              sample_type_to_string(&info.sample_type, t, sizeof(t));
              volume_to_string(&info.volume, v, sizeof(v));
              
              g_print("        Sink: %s\n"
                      "        Name: %s\n"
                      "        Type: %s\n"
                      " Sample-type: %s\n"
                      "      Volume: %s\n"
                      "        Mode: %s\n"
                      "       Flags:%s%s\n"
                      "  Throughput: %i kb/s\n"
                      "Byte counter: %i kb\n",
                      info.shortname,
                      info.name,
                      info.type,
                      t,
                      v,
                      info.mode == SINK_DISABLED ? "disabled" : info.mode == SINK_RUNNING ? "running" : info.mode == SINK_PAUSED ? "paused" : "direct",
                      info.flags & SINK_AUTOCLEAN ? " AUTOCLEAN" : "",
                      info.flags & SINK_DIRECT_SUPPORTED ? " DIRECT_SUPPORTED" : "",
                      info.throughput/1024,
                      info.byte_counter/1024);  
            }
        }
    }
  else if (strcmp(cmd, "ls") == 0)
    {
      g_print("Sources:\n");
      if (!asd_list_sources(c, _proc, NULL))
        asd_perror("Could not list sources");
      else
        {
          g_print("Sinks:\n");
          if (!asd_list_sinks(c, _proc, NULL))
            asd_perror("Could not list sinks");
        }
    }
  else if (strcmp(cmd, "info") == 0)
    {
      ProtocolAsdServerInfoResponse info;

      if (!asd_server_info(c, &info))
        asd_perror("Could not query info");
      else
        {
          gchar st[256];
          
          sample_type_to_string(&info.sample_type, st, sizeof(st));
          
          g_print("              Server: %s\n"
                  "             Version: %s\n"
                  "Internal sample type: %s\n"
                  "          Block size: %u bytes\n"
                  "    Allocated blocks: %u\n"
                  "   ASD Server locked: %s\n"
                  "EsounD Server locked: %s\n"
                  "     Average latency: %u msec\n",
                  info.string,
                  info.version,
                  st,
                  info.block_size,
                  info.alloc_blocks,
                  info.asd_auth_locked ? "Yes" : "No",
                  info.esound_auth_locked ? "Yes" : "No",
                  info.average_latency);
        }
    }
  else if (strcmp(cmd, "version") == 0)
    {
      gchar version[256];
      if (!asd_server_version(c, version, sizeof(version)))
        asd_perror("Could not query version");
      else
        g_print("%s\n", version);
    }
  else
    usage();
  
  asd_connection_free(c);

  return 0;
}
