/*
   ARPACK++ v1.0 8/1/1997
   c++ interface to ARPACK code.

   MODULE DCompGSh.cc.
   Example program that illustrates how to solve a complex dense
   generalized eigenvalue problem in shift and invert mode using
   the ARluCompGenEig class.

   1) Problem description:

      In this example we try to solve A*x = B*x*lambda in shift and
      invert mode, where A and B are derived from a finite element
      discretization of a 1-dimensional convection-diffusion operator
                         (d^2u/dx^2) + rho*(du/dx)
      on the interval [0,1], with zero boundary conditions, using
      piecewise linear elements.

   2) Data structure used to represent matrices A and B:

      Although A and B are very sparse in this example, they are 
      stored here as dense matrices (by columns).

   3) Library called by this example:

      The LAPACK package is called by ARluCompGenEig to solve
      some linear systems involving (A-sigma*B).

   4) Included header files:

      File             Contents
      -----------      ---------------------------------------------
      dcmatrxb.h       CompMatrixE, a function that generates 
                       matrix A.
      dcmatrxc.h       CompMatrixF, a function that generates 
                       matrix B.
      ardnsmat.h       The ARdsNonSymMatrix class definition.
      ardgcomp.h       The ARluCompGenEig class definition.
      lcompsol.h       The Solution function.
      arcomp.h         The "arcomplex" (complex) type definition.

   5) ARPACK Authors:

      Richard Lehoucq
      Kristyn Maschhoff
      Danny Sorensen
      Chao Yang
      Dept. of Computational & Applied Mathematics
      Rice University
      Houston, Texas
*/

#include "arcomp.h"
#include "dcmatrxb.h"
#include "dcmatrxc.h"
#include "ardnsmat.h"
#include "ardgcomp.h"
#include "lcompsol.h"


main()
{

  // Defining variables;

  int               n;            // Dimension of the problem.
  arcomplex<double> rho;          // Parameter used to define A.
  arcomplex<double> *valA, *valB; // pointers to arrays that store
                                  // the elements of A and B.

  // Creating complex matrices A and B.

  n   =  100;
  rho = arcomplex<double>(10.0, 0.0);
  CompMatrixB(n, rho, valA);
  ARdsNonSymMatrix<arcomplex<double> > A(n, valA);

  CompMatrixC(n, valB);
  ARdsNonSymMatrix<arcomplex<double> > B(n, valB);

  // Defining what we need: the four eigenvectors nearest to sigma.

  ARluCompGenEig<double> dprob(4L, A, B, arcomplex<double>(10.0,0.0));

  // Finding eigenvalues and eigenvectors.

  dprob.FindEigenvectors();

  // Printing solution.

  Solution(A, B, dprob);

} // main.

