;;;***************************************************************
;;;An ACL2 Library of Floating Point Arithmetic

;;;David M. Russinoff
;;;Advanced Micro Devices, Inc.
;;;June, 2001
;;;***************************************************************

(in-package "ACL2")

(local (include-book "../support/top"))
(local (include-book "../support/cat"))

(include-book "basic")


;;;**********************************************************************
;;;                             BVECP
;;;**********************************************************************

(defun bvecp (x k)
  (and (integerp x)
       (>= x 0)
       (< x (expt 2 k))))

(in-theory (disable bvecp))

(defthm bvecp-forward
  (implies (bvecp x k)
           (and (integerp x)
                (<= 0 x)
                (< x (expt 2 k))))
  :rule-classes :forward-chaining)

(defthm natp-bvecp
    (implies (bvecp x n)
	     (natp x)))

(defthm bvecp<=
    (implies (and (natp n)
		  (bvecp x n))
	     (<= x (1- (expt 2 n))))
  :rule-classes ())

(defthm bvecp-fl
    (implies (and (not (zp n))
		  (bvecp x n))
	     (bvecp (fl (* 1/2 x)) (1- n))))

(in-theory (disable bvecp-fl))

(defthm bvecp-plus
    (implies (and (bvecp x m)
		  (bvecp y n)
		  (natp m)
		  (natp n))
	     (bvecp (* x y) (+ m n)))
  :rule-classes ())

(defthm bvecp+1
    (implies (and (natp n)
		  (bvecp x n))
	     (bvecp x (+ 1 n))))

(in-theory (disable bvecp+1))

(defthm bvecp-shift
    (implies (and (bvecp x n)
		  (natp n)
		  (natp k)
		  (>= n k))
	     (bvecp (fl (/ x (expt 2 k))) (- n k))))

(in-theory (disable bvecp-shift))

;;This lemma may be enabled to induce case-splitting on bit vectors of
;;length 1:

(defthm bvecp-1-rewrite
    (iff (bvecp x 1)
	 (member x '(0 1))))

(in-theory (disable bvecp-1-rewrite))



;;;**********************************************************************
;;;                         BITS
;;;**********************************************************************

#|
(defun bits (x i j)
  (fl (/ (rem x (expt 2 (1+ i))) (expt 2 j))))
|#

;new
;thms about bits are further below
(defun bits (x i j)
  (if (or (not (integerp i))
          (not (integerp j)))
      0
  (fl (/ (mod x (expt 2 (1+ i))) (expt 2 j)))))

(in-theory (disable bits))

(defthm bits-nonnegative-integerp-type
  (and (<= 0 (bits x i j))
       (integerp (bits x i j)))
  :rule-classes (:type-prescription))

;this rule is no better than bits-nonnegative-integer-type and might be worse
(in-theory (disable (:type-prescription bits)))

;new
(defthm bits-natp
  (natp (bits x i j)))

(defthm bits-bvecp
  (implies (and (<= (+ 1 i (- j)) n)
                (case-split (integerp n)))
           (bvecp (bits x i j) n)))

(defthm bits-mod
  (implies (and (case-split (integerp x))
                (case-split (integerp i))
                (case-split (<= 0 i)))
           (equal (bits x i 0)
                  (mod x (expt 2 (1+ i))))))

(in-theory (disable bits-mod))

(defthm mod-bits-equal
  (implies (and (natp x)
                (natp y)
                (natp i)
                (natp j)
                (= (mod x (expt 2 (1+ i))) (mod y (expt 2 (1+ i)))))
           (= (bits x i j) (bits y i j)))
  :rule-classes ())

(defthm bits-0
  (equal (bits 0 i j) 0))

(defthm bits-with-indices-in-the-wrong-order
  (implies (< i j)
           (equal (bits x i j)
                  0)))


;;;**********************************************************************
;;;                             BITN
;;;**********************************************************************

#| old defn
(defun bitn (x n)
  (if (logbitp n x) 1 0))
|#

;new definition!!
(defun bitn (x n)
  (bits x n n))
(in-theory (disable bitn))

(defthm bitn-nonnegative-integer-type
  (and (<= 0 (bitn x n))
       (integerp (bitn x n)))
  :rule-classes (:type-prescription))

;this rule is no better than bitn-nonnegative-integer-type and might be worse
(in-theory (disable (:type-prescription bitn)))

(defthm bitn-natp
  (natp (bitn x n)))

(defthm bitn-def
  (implies (case-split (integerp k))
           (equal (bitn x k)
                  (mod (fl (/ x (expt 2 k))) 2))))

(in-theory (disable bitn-def))

(defthm bitn-bvecp
  (implies (and (<= 1 k)
                (case-split (integerp k)))
           (bvecp (bitn x n) k)))

;call this bitn-does-nothing?
(defthm bitn-bvecp-1
    (implies (bvecp x 1)
	     (equal (bitn x 0) x)))

;other case?
(defthm bitn-bitn-0
    (equal (bitn (bitn x n) 0)
	   (bitn x n)))

(defthm bitn-0
  (equal (bitn 0 k) 0))

;new
(defthm bitn-0-1
    (or (equal (bitn x n) 0)
        (equal (bitn x n) 1))
  :rule-classes ())

;new
(DEFTHM BITN-REC-0
  (IMPLIES (NATP X)
           (EQUAL (BITN X 0) (MOD X 2))))

(in-theory (disable bitn-rec-0))

(defthm bitn-rec-pos-def
    (implies (and (natp x)
		  (natp k)
		  (> k 0))
	     (equal (bitn x k)
		    (bitn (fl (/ x 2)) (1- k))))
  :rule-classes ((:definition :controller-alist ((bitn t t)))))

(in-theory (disable bitn-rec-pos-def))

(defthm bitn-mod-bitn
    (implies (and (> n k)
                  (natp x)
		  (natp n)
		  (natp k)
		  )
	     (equal (bitn (mod x (expt 2 n)) k)
		    (bitn x k))))

(in-theory (disable bitn-mod-bitn))

;was called just bitn-bvecp
(defthm bitn-bvecp-0-thm
    (implies (bvecp x n)
	     (equal (bitn x n) 0)))

(defthm bitn-bvecp-0
    (implies (and (bvecp x n)
		  (natp m)
		  (natp n))
	     (equal (bitn x (+ m n)) 0)))

(in-theory (disable bitn-bvecp-0))

(defthm bitn-force-1
    (implies (and (bvecp x (1+ n))
		  (<= (expt 2 n) x)
                  (natp n))
	     (equal (bitn x n) 1)))

(in-theory (disable bitn-force-1))

(defthm bitn-force-2
    (implies (and (bvecp x n)
		  (<= (- (expt 2 n) (expt 2 k)) x)
		  (< k n)
                  (natp n)
		  (natp k)
		  )
	     (equal (bitn x k) 1)))

(in-theory (disable bitn-force-2))

(defthm bitn-expt
    (implies (natp n) ;drop?
	     (equal (bitn (expt 2 n) n) 1)))

(in-theory (disable bitn-expt))

(defthm bit+expt
    (implies (and (natp x)
		  (natp n))
	     (not (equal (bitn (+ x (expt 2 n)) n)
			 (bitn x n))))
  :rule-classes ())

(defthm bit+expt-2
    (implies (and (natp x)
		  (natp n)
		  (natp m)
		  (> m n))
	     (equal (bitn (+ x (expt 2 m)) n)
		    (bitn x n))))

(in-theory (disable bit+expt-2))

(defthm bitn+mult
    (implies (and (natp x)
		  (natp k)
		  (natp n)
		  (natp m)
		  (> m n))
	     (equal (bitn (+ x (* k (expt 2 m))) n)
		    (bitn x n))))

(in-theory (disable bitn+mult))

(defthm mod+bitn
    (implies (and (natp a)
		  (natp n))
	     (= (mod a (expt 2 (1+ n)))
		(+ (* (bitn a n) (expt 2 n))
		   (mod a (expt 2 n)))))
  :rule-classes ())

(defthm mod-bitn-0
    (implies (and (natp a)
		  (natp n))
	     (iff (= (mod a (expt 2 (1+ n))) 0)
		  (and (= (mod a (expt 2 n)) 0)
		       (= (bitn a n) 0))))
  :rule-classes ())

(defthm bitn-shift
    (implies (and (natp x)
		  (natp n)
		  (natp k))
	     (= (bitn (* x (expt 2 k)) (+ n k))
		(bitn x n)))
  :rule-classes ())

(defthm bitn-shift-gen
    (implies (and (natp x)
		  (natp (* x (expt 2 k)))
		  (natp n)
		  (natp (+ n k))
		  (integerp k))
	     (= (bitn (* x (expt 2 k)) (+ n k))
		(bitn x n)))
  :rule-classes ())

;this is the most interesting case. perhaps add the other cases for k<0 and k>i-j
(defthm bitn-bits
  (implies (and (<= k (- i j))
                (case-split (<= 0 k))
                (case-split (integerp i))
                (case-split (integerp j))
                (case-split (integerp k))
                )
           (equal (bitn (bits x i j) k)
                  (bitn x (+ j k))))
  :hints (("Goal" :in-theory (enable bitn bits))))
(in-theory (disable bitn-bits)) ;why?



; bits thms


(defthm bits-n-n-rewrite
  (equal (bits x n n)
         (bitn x n)))



(defthm bvecp-bits-0
  (implies (bvecp x j)
           (equal (bits x i j) 0)))

(in-theory (disable bvecp-bits-0))

(defun sumbits (x n)
  (if (zp n)
      0
    (+ (* (expt 2 (1- n)) (bitn x (1- n)))
       (sumbits x (1- n)))))

(defthm sumbits-bits
    (implies (and (natp x)
		  (natp n)
		  (> n 0))
	     (equal (sumbits x n)
		    (bits x (1- n) 0))))

(in-theory (disable sumbits-bits))

(defthm sumbits-thm
    (implies (and (bvecp x n)
		  (natp n)
		  (> n 0))
	     (equal (sumbits x n)
		    x)))

(in-theory (disable sumbits-thm))

(defthm bits-shift-1
    (implies (and (natp x)
		  (natp i)
		  (natp j)
		  (natp k))
	     (equal (bits (fl (/ x (expt 2 k)))
			  i
			  j)
		    (bits x (+ i k) (+ j k)))))

(in-theory (disable bits-shift-1))

;consider enabling!
(defthm bits-bits-1
  (implies (and (<= k (- i j))
                (case-split (<= 0 l))
                (case-split (integerp i))
                (case-split (integerp j))
                (case-split (integerp k))
                (case-split (integerp l))
                )
           (= (bits (bits x i j) k l)
              (bits x (+ k j) (+ l j))))
  :hints (("Goal" :in-theory (enable bits))))
(in-theory (disable bits-bits-1)) ;why?

;consider enabling!
(defthm bits-bits-2
  (implies (and (> k (- i j))
                (case-split (<= 0 l))
;                (case-split (integerp i))
                (case-split (integerp j))
                (case-split (integerp k))
                (case-split (integerp l))
                )
           (equal (bits (bits x i j) k l)
                  (bits x i (+ l j))))
  :hints (("Goal" :in-theory (enable bits))))
(in-theory (disable bits-bits-2)) ;why?

(defthm bits-bits
  (implies (and (case-split (<= 0 l))
                (case-split (integerp i))
                (case-split (integerp j))
                (case-split (integerp k))
                (case-split (integerp l))
                )
           (equal (bits (bits x i j) k l)
                  (if (<= k (- i j))
                      (bits x (+ k j) (+ l j))
                    (bits x i (+ l j)))))
:hints (("Goal" :in-theory (enable bits-bits-1 bits-bits-2))))
(in-theory (disable bits-bits)) ;why?

(defthm bits-shift-5
    (implies (and (natp x)
		  (natp k)
		  (natp i))
	     (equal (* (expt 2 k) (bits x i 0))
		    (bits (* (expt 2 k) x) (+ i k) 0)))
  :rule-classes ())

(defthm bits-0-mod-0
    (implies (and (natp x)
		  (natp m)
		  (natp n))
	     (iff (= (mod x (expt 2 (+ m n 1))) 0)
		  (and (= (bits x (+ m n) n) 0)
		       (= (mod x (expt 2 n)) 0))))
  :rule-classes ())

(defthm bits-0-bitn-0
    (implies (and (natp x)
		  (natp n)
		  (not (= n 0)))
	     (iff (= (bits x n 0) 0)
		  (and (= (bitn x n) 0)
		       (= (bits x (1- n) 0) 0))))
  :rule-classes ())

(defthm bits-plus-bits
    (implies (and (natp x)
		  (natp r)
		  (natp n)
		  (natp m)
		  (> n r)
		  (> m n))
	     (= (bits x (1- m) r)
		(+ (bits x (1- n) r)
		   (* (expt 2 (- n r)) (bits x (1- m) n)))))
  :rule-classes ())

(defthm bits-plus-bitn
    (implies (and (natp x)
		  (natp m)
		  (natp n)
		  (> n m))
	     (= (bits x n m)
		(+ (* (bitn x n) (expt 2 (- n m)))
		   (bits x (1- n) m))))
  :rule-classes ())

(defthm bitn-plus-bits
    (implies (and (natp x)
		  (natp n)
		  (natp m)
		  (> n m))
	     (= (bits x n m)
		(+ (bitn x m)
		   (* 2 (bits x n (1+ m))))))
  :rule-classes ())

(defthm bits-plus-mult
    (implies (and (natp m)
		  (natp n)
		  (>= n m)
		  (natp k)
		  (<= k m)
		  (natp y)
		  (bvecp x k))
	     (= (bits (+ x (* y (expt 2 k))) n m)
		(bits y (- n k) (- m k))))
  :rule-classes ())

(defthm bits-plus-mult-2
    (implies (and (natp m)
		  (natp n)
		  (>= n m)
		  (natp k)
		  (> k n)
		  (natp y)
		  (natp x))
	     (= (bits (+ x (* y (expt 2 k))) n m)
		(bits x n m)))
  :rule-classes ())

(defthm bits-sum-0
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (>= i j)
		  (> j 0)
		  (= (bits (+ x y) (1- j) 0) 0))
	     (equal (bits (+ x y) i j)
		    (bits (+ (bits x i j) (bits y i j) (logior (bitn x (1- j)) (bitn y (1- j))))
			  (- i j) 0)))
  :rule-classes ())

(defthm bits-sum
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (>= i j)
		  (> j 0))
	     (equal (bits (+ x y) i j)
		    (bits (+ (bits x i j)
			     (bits y i j)
			     (bitn (+ (bits x (1- j) 0)
				      (bits y (1- j) 0))
				   j))
			  (- i j) 0)))
  :rule-classes ())


;dangerous?
;consider disabling
(defthm bits-reduce
  (implies (and (< x (expt 2 (+ 1 i)))
                (case-split (integerp x))
                (case-split (<= 0 x))
                (case-split (integerp i))
                (case-split (<= 0 i))
                )
           (equal (bits x i 0) x)))

;dangerous?
;consider disabling
(defthm bits-tail
  (implies (and (natp n)
                (bvecp x (1+ n)))
           (equal (bits x n 0)
                  x)))



;;;**********************************************************************
;;;                          CAT
;;;**********************************************************************

#|(defun CAT (x y n)
  (+ (* (expt 2 n) x) y))
|#

;now always returns a nat
(defun cat (x y n)
  (+ (* (expt 2 (nfix n)) (nfix x))
     (nfix y)))

(in-theory (disable cat))

(defthm cat-nonnegative-integer-type
  (and (integerp (CAT X Y N))
       (<= 0 (CAT X Y N)))
  :rule-classes (:type-prescription)
  )

;this rule is no better than cat-nonnegative-integer-type and might be worse
(in-theory (disable (:type-prescription cat)))

;just a rewrite rule
(defthm cat-natp
  (natp (cat x y n)))

(defthm cat-bvecp
  (implies (and (>= p n) ;handle other case?
                (bvecp x (- p n))
                (case-split (natp n))
                (case-split (natp p))
                (case-split (bvecp y n))
                )
           (bvecp (cat x y n) p)))


(defthm cat-bvecp-rewrite
  (implies (and (>= p n) ;handle the other case?
                (case-split (integerp x))
                (case-split (<= 0 x))
                (case-split (bvecp y n))
                (case-split (natp n))
                (case-split (natp p))
                )
           (equal (bvecp (cat x y n) p)
                  (bvecp x (- p n)))))

;consider enabling this?
(in-theory (disable cat-bvecp-rewrite))

(defthm cat-associative
  (implies (and (case-split (<= 0 m)) ;new now that cat fixes its args
                (case-split (<= 0 n)) ;new now that cat fixes its args
                (case-split (integerp m))
                (case-split (integerp n))
                )
           (equal (cat (cat x y m) z n)
                  (cat x (cat y z n) (+ m n)))))

(in-theory (disable cat-associative)) ;why?

;this rule has caused some problems in the past (in particular, size info is lost)
(defthm cat-0-rewrite
  (implies (and (case-split (integerp x))
                (case-split (<= 0 x)))
           (equal (cat 0 x n) x)))

(defthm bitn-cat-1
  (implies (and (< i n)
                (case-split (natp n))
                (case-split (integerp i))
                (case-split (natp x))
                (case-split (natp y)))
           (equal (bitn (cat x y n) i) (bitn y i))))

(defthm bitn-cat-2
    (implies (and (>= i n)
                  (case-split (bvecp y n))
                  (case-split (natp x))
                  (case-split (natp n))
                  (case-split (integerp i)))
             (equal (bitn (cat x y n) i)
                    (bitn x (- i n)))))

;includes both bitn-cat-1 and bitn-cat-2
(defthm bitn-cat
  (implies (and (case-split (bvecp y n))
                (case-split (natp x))
                (case-split (natp n))
                (case-split (integerp i))
                )
           (equal (bitn (cat x y n) i)
                  (if (< i n)
                      (bitn y i)
                    (bitn x (- i n))))))

;bitn-cat should be all we need for simplifying (bitn (cat...))
(in-theory (disable bitn-cat-1 bitn-cat-2))



(defthm bits-cat-1
  (implies (and (< i n)
                (case-split (natp y))
                (case-split (integerp i))
                (case-split (integerp j))
                (case-split (natp n))
                )
           (equal (bits (cat x y n) i j)
                  (bits y i j)))
  :hints (("Goal" :in-theory (set-difference-theories
                              (enable cat)
                              '(expt-2-integerp))
           :use ((:instance mod-bits (x (cat x y n)))
                 (:instance expt-2-integerp (i (- n (1+ i)))) ;elim?
                 (:instance mod+-thm (n (expt 2 (1+ i))) (m y) (a (* x (expt 2 (- n (1+ i))))))))))

(defthm bits-cat-2
  (implies (and (>= j n)
                (case-split (natp x))
                (case-split (bvecp y n))
                (case-split (natp n))
                (case-split (integerp i))
                (case-split (integerp j))
                )
           (equal (bits (cat x y n) i j)
                  (bits x (- i n) (- j n))))
  :hints (("Goal" :in-theory (enable cat)
           :use ((:instance fl-unique (x (/ (cat x y n) (expt 2 n))) (n x))
                 (:instance bit-bits-a (x (cat x y n)) (k n))))))

(defthm bits-cat-3
    (implies (and (>= i n)
		  (< j n)
                  (case-split (bvecp y n))
                  (case-split (natp x))
                  (case-split (natp n))
                  (case-split (natp i))
                  (case-split (natp j))
                  )
	     (equal (bits (cat x y n) i j)
		    (cat (bits x (- i n) 0)
			 (bits y (1- n) j)
			 (- n j))))
  :hints (("Goal" :use ((:instance bits-plus-bits (x (cat x y n)) (m (1+ i)) (r j))))
	  ("Subgoal 1" :in-theory (enable cat))))

;includes bits-cat-1, bitn-cat-2, and bits-cat-3
;we expect the indices to be constants, so this won't cause case-splits
(defthm bits-cat
  (implies (and (case-split (bvecp y n))
                (case-split (natp x))
                (case-split (natp n))
                (case-split (natp i))
                (case-split (natp j))
                )
           (equal (bits (cat x y n) i j)
                  (if (< i n)
                      (bits y i j)
                    (if (>= j n)
                        (bits x (- i n) (- j n))
                      (cat (bits x (- i n) 0)
                           (bits y (1- n) j)
                           (- n j)))))))

;bits-cat should be all we need for simplifying (bits (cat...))
(in-theory (disable bits-cat-1 bits-cat-2 bits-cat-3))

(defthm cat-bits-bits
    (implies (and (>= i j)
		  (= j (1+ k))
		  (>= k l)
		  (= n (1+ (- k l)))
                  (natp x)
		  (natp i)
		  (natp j)
		  (natp k)
		  (natp l)
		  )
	     (equal (cat (bits x i j) (bits x k l) n)
		    (bits x i l))))

(defthm cat-bitn-bits
    (implies (and (= j (1+ k))
		  (>= k l)
		  (= n (1+ (- k l)))
                  (natp x)
		  (natp j)
		  (natp k)
		  (natp l)
		  )
	     (equal (cat (bitn x j) (bits x k l) n)
		    (bits x j l))))

(defthm cat-bits-bitn
    (implies (and (>= i j)
		  (= j (1+ k))
                  (natp x)
		  (natp i)
		  (natp j)
		  (natp k)
		  )
	     (equal (cat (bits x i j) (bitn x k) 1)
		    (bits x i k))))

;;;**********************************************************************
;;;                       MULCAT
;;;**********************************************************************

(defun mulcat (l n x)
  (if (and (integerp n) (> n 0))
      (cat (mulcat l (1- n) x)
	   x
	   l)
    0))
(in-theory (disable mulcat)) ;consider

(defthm mulcat-nonnegative-integer-type
  (and (integerp (mulcat l n x))
       (<= 0 (mulcat l n x)))
  :rule-classes (:type-prescription)
  )

;this rule is no better than mulcat-nonnegative-integer-type and might be worse:
(in-theory (disable (:type-prescription mulcat)))

(defthm mulcat-natp
  (natp (mulcat l n x)))

(defthm mulcat-bvecp
  (implies (and (>= p (* l n))
                (case-split (integerp p))
                (case-split (natp l))
                (case-split (bvecp x l))
                )
           (bvecp (mulcat l n x) p)))

(defthm mulcat-1
  (implies (and (case-split (integerp x))
                (case-split (<= 0 x)))
           (equal (mulcat l 1 x) x)))

(defthm mulcat-0
  (equal (mulcat l n 0) 0))

(defthm mulcat-n-1
  (implies (case-split (<= 0 n))
           (equal (mulcat 1 n 1)
                  (1- (expt 2 n)))))

; note: only applies when mulcat's first param is 1
(defthm bitn-mulcat-1
  (implies (and (natp x)
                (integerp n) ;(natp n)
                (natp n2)
                (< n2 n)
                (case-split (bvecp x 1))
                )
           (equal (BITN (MULCAT 1 n x) n2)
                  x)))

;;;**********************************************************************
;;;                       SETBITS
;;;**********************************************************************

#| old definition:
(defun setbits (x i j y)
  (cat (cat (ash x (- (1+ i)))
	    y
	    (1+ (- i j)))
       (bits x (1- j) 0)
       j))
|#

;setbits has a new parameter, w, indicating the size of the expression returned
;the new outer call to bits only effects things if the indices i,j are bad;
;it forces setbits to always return something of the advertised size w
(defun setbits (x w i j y)
  (bits (cat (bits x (1- w) (1+ i))
             (cat (bits y (- i j) 0)
                  (bits x (1- j) 0)
                  j)
             (1+ i))
        (1- w)
        0))

(in-theory (disable setbits))

(defthm setbits-nonnegative-integer-type
  (and (integerp (setbits x w i j y))
       (<= 0 (setbits x w i j y)))
  :hints (("Goal" :in-theory (enable setbits)))
  :rule-classes (:type-prescription)
  )

;this rule is no better than setbits-nonnegative-integer-type and might be worse:
(in-theory (disable (:type-prescription setbits)))

(defthm setbits-natp
  (natp (setbits x w i j y)))

(defthm setbits-bvecp
  (implies (and (<= w k)
                (case-split (integerp k))
                )
           (bvecp (setbits x w i j y) k)))

;all hyps (except the syntaxp hyp) are about indices and should always be true
(defthm setbits-rewrite
  (implies (and (syntaxp (not (and (quotep j) (equal (cadr j) 0)))) ;setbits-rewrite-2 handles this
                (case-split (natp i))
                (case-split (natp j))
                (case-split (natp w))
                (case-split (< i w))
                (case-split (<= j i))
                )
  (equal (setbits x w i j y)
         (cat (bits x (1- w) (1+ i))
              (cat (bits y (- i j) 0)
                   (bits x (1- j) 0)
                   j)
              (+ 1 i)))))

;when j is 0, there is no lower part of x.
;all hyps are about indices and should always be true
(defthm setbits-rewrite-when-j-is-0
  (implies (and (case-split (natp i))
                (case-split (natp w))
                (case-split (<= 1 w))
                (case-split (< i w)))
           (equal (setbits x w i 0 y) ;note the 0
                  (cat (bits x (1- w) (1+ i))
                       (bits y i 0)
                       (+ 1 i)))))

#|
(defmacro setbitn (x n y)
  `(setbits ,x ,n ,n ,y))
|#

(defun setbitn (x w n y)
  (setbits x w n n y))

(in-theory (disable setbitn))

(defthm setbitn-nonnegative-integer-type
  (and (integerp (setbitn x w n y))
       (<= 0 (setbitn x w n y)))
  :rule-classes (:type-prescription)
  )

;this rule is no better than setbits-nonnegative-integer-type and might be worse:
(in-theory (disable (:type-prescription setbitn)))

(defthm setbitn-natp
  (natp (setbitn x w n y)))

(defthm setbitn-bvecp
  (implies (and (<= w k)
                (case-split (integerp k)))
           (bvecp (setbitn x w n y) k)))

;This rewrites setbitn to setbits (and thus to cat) when the index of the setbitn call is a constant.
(defthm setbitn-rewrite
  (implies (syntaxp (quotep n))
           (equal (setbitn x w n y)
                  (setbits x w n n y))))




#|  commented out the setbits thms, because now we expect setbits to always open to a cat expression (and we
changed setbits, so these would take some work and rephrasing to re-prove).
(defthm bitn-setbits-1
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (natp k)
		  (<= j i)
		  (< k j))
	     (equal (bitn (setbits x i j y) k)
		    (bitn x k))))

(defthm bitn-setbits-2
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (natp k)
		  (<= k i)
		  (<= j k))
	     (equal (bitn (setbits x i j y) k)
		    (bitn y (- k j)))))

(defthm bitn-setbits-3
    (implies (and (natp x)
		  (bvecp y (1+ (- i j)))
		  (natp i)
		  (natp j)
		  (natp k)
		  (< i k)
		  (<= j i))
	     (equal (bitn (setbits x i j y) k)
		    (bitn x k))))

(defthm bits-setbits-1
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (natp k)
		  (natp l)
		  (<= j i)
		  (< k j)
		  (<= l k))
	     (equal (bits (setbits x i j y) k l)
		    (bits x k l))))

(defthm bits-setbits-2
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (natp k)
		  (natp l)
		  (<= k i)
		  (<= l k)
		  (<= j l))
	     (equal (bits (setbits x i j y) k l)
		    (bits y (- k j) (- l j)))))

(defthm bits-setbits-3
    (implies (and (natp x)
		  (bvecp y (1+ (- i j)))
		  (natp i)
		  (natp j)
		  (natp k)
		  (natp l)
		  (<= l k)
		  (< i l)
		  (<= j i))
	     (equal (bits (setbits x i j y) k l)
		    (bits x k l))))

(defthm bits-setbits-4
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (natp k)
		  (natp l)
		  (<= k i)
		  (<= j k)
		  (< l j)
		  (bvecp y (1+ (- i j))))
	     (equal (bits (setbits x i j y) k l)
		    (cat (bits y (- k j) 0)
			 (bits x (1- j) l)
			 (- j l)))))

(defthm bits-setbits-5
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (natp k)
		  (natp l)
		  (< i k)
		  (<= l i)
		  (<= j l)
		  (bvecp y (1+ (- i j))))
	     (equal (bits (setbits x i j y) k l)
		    (cat (bits x k (1+ i))
			 (bits y (- i j) (- l j))
			 (1+ (- i l))))))

(defthm bits-setbits-6
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (natp k)
		  (natp l)
		  (< i k)
		  (<= j i)
		  (< l j)
		  (bvecp y (1+ (- i j))))
	     (equal (bits (setbits x i j y) k l)
		    (cat (cat (bits x k (1+ i))
			      y
			      (1+ (- i j)))
			 (bits x (1- j) l)
			 (- j l)))))
|#

;taking bits from the lower third
(defthm bits-setbits-1
  (implies (and (< k j)
                (case-split (< 0 w))
                (case-split (< i w))
                (case-split (<= 0 l))
                (case-split (<= j i)) ;drop?
                (case-split (integerp w))
                (case-split (integerp i))
                (case-split (integerp j))
                )
           (equal (bits (setbits x w i j y) k l)
                  (bits x k l))))

;taking bits from the middle third
(defthm bits-setbits-2
    (implies (and (<= k i)
		  (<= j l)
		  (case-split (integerp i))
                  (case-split (<= 0 j))
		  (case-split (integerp j))
		  (case-split (integerp k))
		  (case-split (integerp l))
                  (case-split (<= 0 l))
                  (case-split (integerp w))
                  (case-split (< 0 w))
                  (case-split (< i w))
		  )
	     (equal (bits (setbits x w i j y) k l)
		    (bits y (- k j) (- l j)))))

;taking bits from the upper third
(defthm bits-setbits-3
    (implies (and (< i l)
		  (case-split (natp i))
		  (case-split (natp j))
		  (case-split (natp k)) ;gen
		  (case-split (natp l)) ;gen
		  (case-split (<= j i))
                  (case-split (integerp w))
                  (case-split (< 0 w))
                  (case-split (< i w))
                  (case-split (< k w)) ;handle this?
                  )
	     (equal (bits (setbits x w i j y) k l)
		    (bits x k l))))

;setbits should expand to cat
(in-theory (disable bits-setbits-1 bits-setbits-2 bits-setbits-3))

;in most cases, setbitn will rewrite to setbits and thus to cat
(defthm bitn-setbitn
  (implies (and (case-split (bvecp y 1))
                (case-split (< 0 w))
                (case-split (< n w))
                (case-split (< n2 w))
                (case-split (<= 0 n2))
                (case-split (integerp w))
                (case-split (integerp n))
                (<= 0 n)
                (case-split (integerp n2))
                )
           (equal (bitn (setbitn x w n y) n2)
                  (if (equal n n2)
                      y
                    (bitn x n2))))
  :hints (("Goal" :cases ((< n n2) (= n n2))
           :in-theory (enable setbitn bitn bits-reduce)))
  )

 
;;;**********************************************************************
;;;                       SHFT
;;;**********************************************************************

#|
(defun shft (x s l)
  (mod (fl (* (expt 2 s) x)) (expt 2 l)))
|#

(defun shft (x s l)
  (mod (fl (* (expt 2 s) x)) (expt 2 (nfix l))))
(in-theory (disable shft))

(defthm shft-nonnegative-integer-type
  (and (integerp (shft x s l))
       (<= 0 (shft x s l)))
  :rule-classes (:type-prescription))

;(:type-prescription shft) is no better than shft-nonnegative-integer-type and might be worse:
(in-theory (disable (:type-prescription shft)))

(defthm shft-natp
  (natp (shft x s n)))

(defthm shft-bvecp
  (implies (and (<= n k)
                (case-split (integerp k)))
           (bvecp (shft x s n) k)))

;(in-theory (disable bvecp-shft))


;;;**********************************************************************
;;;                       COMP1
;;;**********************************************************************

#| old version
(defun comp1 (x n)
  (1- (- (expt 2 n) x)))
|#

;new version
(defun COMP1 (x n)
   (if (natp n)
       (+ -1 (expt 2 n) (- (bits x (+ -1 n) 0)))
     0))

(in-theory (disable comp1))

(defthm comp1-nonnegative-integer-type
  (and (integerp (comp1 x n))
       (<= 0 (comp1 x n)))
  :rule-classes ((:type-prescription :typed-term (comp1 x n))))

;this rule is no better than comp1-nonnegative-integer-type and might be worse:
(in-theory (disable (:type-prescription comp1))) 

(defthm comp1-natp
   (natp (comp1 x n)))

(defthm comp1-bvecp
  (implies (and (<= n k)
                (case-split (integerp k)))
           (bvecp (comp1 x n) k)))

(defthm COMP1-COMP1
  (implies (and (case-split (natp n)); (integerp n)
                (case-split (bvecp x n)) ;added
                )
           (= (comp1 (comp1 x n) n)
              x)))

(defthm comp1-2+1
  (implies (and (case-split (natp x))
                (case-split (natp n))
                )
           (equal (+ 1 (* 2 (comp1 x n)))
                  (comp1 (* 2 x) (1+ n)))))


(defthm comp1-fl-rewrite
    (implies (and (not (zp n))
		  (bvecp x n))
	     (equal (fl (* 1/2 (comp1 x n)))
		    (comp1 (fl (* 1/2 x)) (1- n)))))

(in-theory (disable comp1-fl-rewrite))

(defthm comp1-mod-2
    (implies (and (not (zp n))
		  (bvecp x n))
	     (equal (mod (comp1 x n) 2)
		    (comp1 (mod x 2) 1))))

(in-theory (disable comp1-mod-2))

(defthm mod-comp1-2
    (implies (and (natp n)
		  (not (= n 0))
		  (bvecp x n))
	     (not (= (mod (comp1 x n) 2)
		     (mod x 2))))
  :rule-classes ())

(defthm bitn-comp1-not-equal
  (implies (and (< k n)
                (natp n)
                (bvecp x n)
                (natp k))
           (not (= (bitn (comp1 x n) k) (bitn x k))))
  :rule-classes nil
  :hints (("goal" :use (bitn-comp1-thm))))

(defthm bits-comp1
    (implies (and (natp m) 
		  (natp i) 
		  (natp j)
		  (> m i)
		  (>= i j)
		  (bvecp x m))
	     (equal (bits (comp1 x m) i j)
		    (comp1 (bits x i j) (1+ (- i j))))))

;consider enabling this:
(in-theory (disable bits-comp1))

(defthm mod-comp1-rewrite
    (implies (and (natp n)
		  (natp m)
		  (bvecp x m)
		  (not (= n 0))
		  (>= m n))
	     (equal (mod (comp1 x m) (expt 2 n))
		    (comp1 (mod x (expt 2 n)) n))))

(in-theory (disable mod-comp1-rewrite))

(defthm bitn-comp1
    (implies (and (natp m) 
		  (natp n) 
		  (> m n)
		  (bvecp x m))
	     (equal (bitn (comp1 x m) n)
		    (comp1 (bitn x n) 1))))

;consider enabling this:
(in-theory (disable bitn-comp1))


;;;**********************************************************************
;;;                       LOGAND, LOGIOR, and LOGXOR
;;;**********************************************************************

(in-theory (disable logand logior logxor))

(defthm logand-rewrite
    (implies (and (natp x)
		  (natp y))
	     (equal (logand x y)
		    (+ (* 2 (logand (fl (/ x 2)) (fl (/ y 2))))
		       (logand (mod x 2) (mod y 2)))))
  :rule-classes ((:definition :controller-alist ((binary-logand t t)))))

(in-theory (disable logand-rewrite))

(defthm logior-rewrite
    (implies (and (natp i)
		  (natp j))
	     (equal (logior i j)
		    (+ (* 2 (logior (fl (/ i 2)) (fl (/ j 2))))
		       (logior (mod i 2) (mod j 2)))))
  :rule-classes ((:definition :controller-alist ((binary-logior t t)))))

(in-theory (disable logior-rewrite))

(defthm logxor-def-rewrite
    (implies (and (natp x)
		  (natp y))
	     (equal (logxor x y)
		    (+ (* 2 (logxor (fl (/ x 2)) (fl (/ y 2))))
		       (logxor (mod x 2) (mod y 2)))))
  :rule-classes ((:definition :controller-alist ((binary-logxor t t)))))

(in-theory (disable logxor-def-rewrite))

(defthm logand-natp
    (implies (and (natp i)
		  (natp j))
	     (natp (logand i j)))
  :rule-classes (:type-prescription :rewrite))

(defthm logand-bvecp
    (implies (and (natp n)
		  (bvecp x n)
		  (natp y))
	     (bvecp (logand x y) n)))

(defthm logior-natp
    (implies (and (natp i)
		  (natp j))
	     (natp (logior i j)))
  :rule-classes (:type-prescription :rewrite))

(defthm logior-bvecp
    (implies (and (natp n)
		  (bvecp x n)
		  (bvecp y n))
	     (bvecp (logior x y) n)))

(defthm logxor-natp
    (implies (and (natp i)
		  (natp j))
	     (natp (logxor i j)))
  :rule-classes (:type-prescription :rewrite))

(defthm logxor-bvecp
    (implies (and (natp n)
		  (bvecp x n)
		  (bvecp y n))
	     (bvecp (logxor x y) n)))

(defun logop-2-induct (x y)
  (if (or (zp x) (zp y))
      ()
    (logop-2-induct (fl (/ x 2)) (fl (/ y 2)))))

(defun logop-2-n-induct (x y n)
  (if (zp n)
      (cons x y)
    (logop-2-n-induct (fl (/ x 2)) (fl (/ y 2)) (1- n))))

(defun logop-3-induct (x y z)
  (if (and (natp x) (natp y) (natp z))
      (if (and (zp x) (zp y) (zp z))
	  t
	(logop-3-induct (fl (/ x 2)) (fl (/ y 2)) (fl (/ z 2))))
    t))

(defthm logand-mod-2
    (implies (and (natp x)
		  (natp y))
	     (equal (mod (logand x y) 2)
		    (logand (mod x 2) (mod y 2)))))

(in-theory (disable logand-mod-2))

(defthm logand-fl-2-rewrite
    (implies (and (natp x)
		  (natp y))
	     (equal (fl (* 1/2 (logand x y)))
		    (logand (fl (* 1/2 x)) (fl (* 1/2 y))))))

(in-theory (disable logand-fl-2-rewrite))

(defthm logior-mod-2
    (implies (and (natp i)
		  (natp j))
	     (equal (mod (logior i j) 2)
		    (logior (mod i 2) (mod j 2)))))

(in-theory (disable logior-mod-2))

(defthm logior-fl-2-rewrite
    (implies (and (natp i)
		  (natp j))
	     (equal (fl (* 1/2 (logior i j)))
		    (logior (fl (* 1/2 i)) (fl (* 1/2 j))))))

(in-theory (disable logior-fl-2-rewrite))

(defthm logxor-mod-2
    (implies (and (natp i)
		  (natp j))
	     (equal (mod (logxor i j) 2)
		    (logxor (mod i 2) (mod j 2)))))

(in-theory (disable logxor-mod-2))

(defthm logxor-fl-2-rewrite
    (implies (and (natp i)
		  (natp j))
	     (equal (fl (* 1/2 (logxor i j)))
		    (logxor (fl (* 1/2 i)) (fl (* 1/2 j))))))

(in-theory (disable logxor-fl-2-rewrite))

(defthm logand-x-0
    (equal (logand x 0) 0))

(defthm logand-0-y
    (equal (logand 0 y) 0))

(defthm logior-x-0
    (implies (natp x)
	     (equal (logior x 0) x)))

(defthm logior-0-y
    (implies (natp y)
	     (equal (logior 0 y) y)))

(defthm logxor-x-0
    (implies (integerp x)
	     (equal (logxor x 0) x)))

(defthm logxor-0-y
    (implies (integerp y)
	     (equal (logxor 0 y) y)))

(defthm logand-self
  (implies (case-split (integerp i))
           (equal (logand i i) i)))

(defthm logior-self
    (implies (natp x)
	     (equal (logior x x) x)))

(defthm logxor-self
    (implies (natp x)
	     (equal (logxor x x) 0)))

(defthm logior-not-0
    (implies (and (natp x)
		  (natp y)
		  (= (logior x y) 0))
	     (and (= x 0) (= y 0)))
  :rule-classes ())

(defthm logand-ones
    (implies (and (natp n)
		  (bvecp x n))
	     (equal (logand x (1- (expt 2 n)))
		    x)))

(in-theory (disable logand-ones))

(defthm logand-x-1
    (implies (bvecp x 1)
	     (equal (logand x 1) x)))

(defthm logand-1-x
    (implies (bvecp x 1)
	     (equal (logand 1 x) x)))

(defthm logior-ones
    (implies (and (natp n)
		  (bvecp x n))
	     (equal (logior x (1- (expt 2 n)))
		    (1- (expt 2 n))))
  :rule-classes ())

(defthm logior-x-1
    (implies (bvecp x 1)
	     (equal (logior x 1) 1)))

(defthm logior-1-x
    (implies (bvecp x 1)
	     (equal (logior 1 x) 1)))

(defthm logxor-ones
    (implies (and (natp n)
		  (bvecp x n))
	     (equal (logxor x (1- (expt 2 n)))
		    (comp1 x n)))
  :rule-classes ())

(defthm logand-commutative
    (equal (logand j i) (logand i j)))

(defthm logior-commutative
    (implies (and (integerp x)
		  (integerp y))
	     (equal (logior x y) (logior y x))))

(defthm logxor-commutative
    (implies (and (integerp x)
		  (integerp y))
	     (equal (logxor x y) (logxor y x))))

(defthm logand-associative
    (equal (logand (logand i j) k)
           (logand i (logand j k))))

(in-theory (disable logand-associative)) ;why?

(defthm logior-associative
    (implies (and (natp x)
		  (natp y)
		  (natp z))
	     (equal (logior (logior x y) z)
		    (logior x (logior y z)))))

(in-theory (disable logior-associative))

(defthm logxor-associative
    (implies (and (natp x)
		  (natp y)
		  (natp z))
	     (equal (logxor (logxor x y) z)
		    (logxor x (logxor y z)))))

(in-theory (disable logxor-associative))

(defthm comp1-logxor
    (implies (and (natp n)
		  (bvecp x n)
		  (bvecp y n))
	     (equal (comp1 (logxor x y) n)
		    (logxor (comp1 x n) y))))

(in-theory (disable comp1-logxor))

(defthm logior-logand
    (implies (and (natp x)
		  (natp y)
		  (natp z))
	     (equal (logior x (logand y z))
		    (logand (logior x y) (logior x z)))))

(in-theory (disable logior-logand))

(defthm logand-logior
    (implies (and (natp x)
		  (natp y)
		  (natp z))
	     (equal (logand x (logior y z))
		    (logior (logand x y) (logand x z)))))

(in-theory (disable logand-logior))

(defthm logior-logand-2
    (implies (and (natp x)
		  (natp y)
		  (natp z))
	     (equal (logand  (logior y z) x)
		    (logior (logand y x) (logand z x)))))

(in-theory (disable logior-logand-2))

(defthm log3
    (implies (and (natp x) (natp y) (natp z))
	     (equal (logior (logand x y) (logior (logand x z) (logand y z)))
		    (logior (logand x y) (logand (logxor x y) z))))
  :rule-classes ())

(defthm logxor-rewrite-2
    (implies (and (bvecp x n)
		  (bvecp y n)
                  (natp n)
		  (not (= n 0)))
	     (equal (logxor x y)
		    (logior (logand x (comp1 y n))
			    (logand y (comp1 x n))))))

(in-theory (disable logxor-rewrite-2))

(defthm logior-expt
    (implies (and (natp n)
		  (natp x)
		  (bvecp y n))
	     (= (logior (* (expt 2 n) x) y)
		(+ (* (expt 2 n) x) y)))
  :rule-classes ())

(defthm logior-expt-2
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (= (logior (* (expt 2 n) x)
			(* (expt 2 n) y))
		(* (expt 2 n) (logior x y))))
  :rule-classes ())

(defthm mod-logior
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (equal (mod (logior x y) (expt 2 n))
		    (logior (mod x (expt 2 n)) (mod y (expt 2 n))))))

(in-theory (disable mod-logior))

(defthm logand-bnd
    (implies (and (natp x)
		  (natp y))
	     (<= (logand x y) x))
  :rule-classes :linear)

(defthm logand-expt
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (= (logand (* (expt 2 n) x) y)
		(* (expt 2 n) (logand x (fl (/ y (expt 2 n)))))))
  :rule-classes ())

(defthm mod-logand-expt
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (= (mod (logand x y) (expt 2 n))
		(logand (mod x (expt 2 n)) y)))
  :rule-classes ())

(defthm mod-logand-rewrite
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (equal (mod (logand x y) (expt 2 n))
		    (logand (mod x (expt 2 n)) (mod y (expt 2 n))))))

(in-theory (disable mod-logand-rewrite))

(defthm mod-logxor-rewrite
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (equal (mod (logxor x y) (expt 2 n))
		    (logxor (mod x (expt 2 n))
			    (mod y (expt 2 n))))))

(in-theory (disable mod-logxor-rewrite))

(defthm logand-mod-expt
    (implies (and (natp x)
		  (natp y)
		  (natp n)
		  (< x (expt 2 n)))
	     (= (logand x y)
		(logand x (mod y (expt 2 n)))))
  :rule-classes ())

(defthm bitn-logand
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (equal (bitn (logand x y) n)
		    (logand (bitn x n) (bitn y n)))))

(in-theory (disable bitn-logand)) ;why?

(defthm bits-logand
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (>= i j))
	     (equal (bits (logand x y) i j)
		    (logand (bits x i j) (bits y i j)))))

(in-theory (disable bits-logand)) ;why?

(defthm bitn-logior
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (equal (bitn (logior x y) n)
		    (logior (bitn x n) (bitn y n)))))

(in-theory (disable bitn-logior)) ;why?

(defthm bits-logior
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (>= i j))
	     (equal (bits (logior x y) i j)
		    (logior (bits x i j) (bits y i j)))))

(in-theory (disable bits-logior)) ;why?

(defthm bitn-logxor
  (implies (and (case-split (integerp x))
                (case-split (>= x 0))
                (case-split (integerp y))
                (case-split (>= y 0))
                (case-split (integerp n))
                (case-split (>= n 0)))
           (= (bitn (logxor x y) n)
              (logxor (bitn x n) (bitn y n)))))

(in-theory (disable bitn-logxor)) ;why?

(defthm bits-logxor
  (implies (and (case-split (natp x))
                (case-split (natp y))
                (case-split (natp i))
                (case-split (natp j)))
           (equal (bits (logxor x y) i j)
                  (logxor (bits x i j) (bits y i j)))))

(in-theory (disable bits-logxor)) ;why?

;what's the point of this?
(defthm bits-logxor-upper-slice
    (implies (and (equal n (+ 1 i))
                  (bvecp x n)
		  (bvecp y n)
		  (natp n)
		  (natp i)
		  (natp j)
                  )
	     (equal (bits (logxor x y) i j)
		    (logxor (bits x i j) (bits y i j)))))

;why isn't bits-logxor-upper-slice disabled?

(defthm logand-expt-2
    (implies (and (natp x)
		  (natp k))
	     (= (logand x (expt 2 k))
		(* (expt 2 k) (bitn x k))))
  :rule-classes ())

(defthm logior-expt-3
    (implies (and (natp x)
		  (natp k))
	     (= (logior x (expt 2 k))
		(+ x
		   (* (expt 2 k) 
		      (- 1 (bitn x k))))))
  :rule-classes ())

(defthm logand-expt-3
    (implies (and (natp x)
		  (natp n)
		  (natp k)
		  (< k n))
	     (= (logand x (- (expt 2 n) (expt 2 k)))
		(* (expt 2 k) (bits x (1- n) k))))
  :rule-classes ())

(defthm logand-expt-4
    (implies (and (natp n)
		  (natp k)
		  (natp l)
		  (< l k)
		  (<= k n))
	     (= (logand (- (1- (expt 2 n)) (expt 2 l)) (- (expt 2 n) (expt 2 k)))
		(- (expt 2 n) (expt 2 k))))
  :rule-classes ())

(defthm bitn-logxor-0
    (implies (and (natp a)
		  (natp b))
	     (= (bitn (+ a b) 0)
		(bitn (logxor a b) 0)))
  :rule-classes ())



;(in-theory (disable logand1 logior1 logxor1)) ;consider