/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

/*
 * Created on Oct 24, 2005
 */
package org.jboss.test.remoting.transport.multiplex.utility;

import org.jboss.logging.Logger;
import org.jboss.remoting.transport.multiplex.utility.ShrinkableByteArrayOutputStream;


/**
 * A ShrinkableByteArrayOutputStreamTest.

 * @author <a href="mailto:r.sigal@computer.org">Ron Sigal</a>
 * @version $Revision: 585 $
 * <p>
 * Copyright (c) 2005
 * </p>
 */

public class ShrinkableByteArrayOutputStreamTestCase extends SuccessCountingTestCase
{
   protected static final Logger log = Logger.getLogger(ShrinkableByteArrayOutputStreamTestCase.class);
   
   public void testSmallArraysByInspection()
   {
      log.info("entering: testSmallArraysByInspection()");
      
      try
      {
         // Test basic operation of small arrays, by way of inspection of variables.
         ShrinkableByteArrayOutputStreamSubClass sbaos = new ShrinkableByteArrayOutputStreamSubClass();
         sbaos.write(new byte[] {3, 5, 7});
         assertTrue(sbaos.available() == 3);
         assertTrue(sbaos.start() == 0);
         assertTrue(sbaos.bytesReturned() == 0);
         assertTrue(sbaos.getUsed() == 0);
         assertTrue(sbaos.getNextUsed() == 0);
         assertTrue(sbaos.getCount() == 3);
         byte[] bytes = sbaos.toByteArray(3);
         assertTrue(sbaos.available() == 0);
         assertTrue(sbaos.start() == 0);
         assertTrue(sbaos.bytesReturned() == 3);
         assertTrue(sbaos.getUsed() == 0);
         assertTrue(sbaos.getNextUsed() == 3);
         assertTrue(sbaos.getCount() == 3);
         assertTrue(bytes[0] == 3 && bytes[1] == 5 && bytes[2] == 7);
         bytes = sbaos.toByteArray(3);
         assertTrue(sbaos.available() == 0);
         assertTrue(sbaos.start() == 3);
         assertTrue(sbaos.bytesReturned() == 0);
         assertTrue(sbaos.getUsed() == 3);
         assertTrue(sbaos.getNextUsed() == 3);
         assertTrue(sbaos.getCount() == 3);
         
         sbaos.write(new byte[] {9, 11});
         assertTrue(sbaos.available() == 2);
         assertTrue(sbaos.start() == 3);
         assertTrue(sbaos.bytesReturned() == 0);
         assertTrue(sbaos.getUsed() == 3);
         assertTrue(sbaos.getNextUsed() == 3);
         assertTrue(sbaos.getCount() == 5);
         bytes = sbaos.toByteArray(1);
         assertTrue(sbaos.available() == 1);
         assertTrue(sbaos.start() == 3);
         assertTrue(sbaos.bytesReturned() == 1);
         assertTrue(sbaos.getUsed() == 3);
         assertTrue(sbaos.getNextUsed() == 4);
         assertTrue(sbaos.getCount() == 5);
         assertTrue(bytes[3] == 9);
         
         sbaos.write(new byte[] {13, 15});
         assertTrue(sbaos.available() == 3);
         assertTrue(sbaos.start() == 3);
         assertTrue(sbaos.bytesReturned() == 1);
         assertTrue(sbaos.getUsed() == 3);
         assertTrue(sbaos.getNextUsed() == 4);
         assertTrue(sbaos.getCount() == 7);
         bytes = sbaos.toByteArray(2);
         assertTrue(sbaos.available() == 1);
         assertTrue(sbaos.start() == 4);
         assertTrue(sbaos.bytesReturned() == 2);
         assertTrue(sbaos.getUsed() == 4);
         assertTrue(sbaos.getNextUsed() == 6);
         assertTrue(sbaos.getCount() == 7);
         assertTrue(bytes[4] == 11 && bytes[5] == 13);
         bytes = sbaos.toByteArray(2);
         assertTrue(sbaos.available() == 0);
         assertTrue(sbaos.start() == 6);
         assertTrue(sbaos.bytesReturned() == 1);
         assertTrue(sbaos.getUsed() == 6);
         assertTrue(sbaos.getNextUsed() == 7);
         assertTrue(sbaos.getCount() == 7);
         assertTrue(bytes[6] == 15);
         bytes = sbaos.toByteArray(2);
         assertTrue(sbaos.available() == 0);
         assertTrue(sbaos.start() == 7);
         assertTrue(sbaos.bytesReturned() == 0);
         assertTrue(sbaos.getUsed() == 7);
         assertTrue(sbaos.getNextUsed() == 7);
         assertTrue(sbaos.getCount() == 7);
      }
      catch (Exception e)
      {
         e.printStackTrace();
         fail();
      }
      
      log.info("testSmallArraysByInspection() PASSES");
      OKCounter++;
   }
    
   
   public void testSmallArraysByUsage()
   {
      log.info("entering: testSmallArraysByUsage()");
      
      try
      {
         // Test basic operation of small arrays, by way of expected usage.
         ShrinkableByteArrayOutputStreamSubClass sbaos = new ShrinkableByteArrayOutputStreamSubClass();
         byte[] returnedBytes;
         byte[] newBytes;
         sbaos.write(new byte[] {3, 5, 7});
         returnedBytes = sbaos.toByteArray(3);
         newBytes = new byte[sbaos.bytesReturned()];
         System.arraycopy(returnedBytes, sbaos.start(), newBytes, 0, sbaos.bytesReturned());
         assertTrue(newBytes.length == 3 && newBytes[0] == 3 && newBytes[1] == 5 && newBytes[2] == 7);
         
         sbaos.write(new byte[] {9, 11});
         returnedBytes = sbaos.toByteArray(1);
         newBytes = new byte[sbaos.bytesReturned()];
         System.arraycopy(returnedBytes, sbaos.start(), newBytes, 0, sbaos.bytesReturned());
         assertTrue(newBytes.length == 1 && newBytes[0] == 9);
         
         sbaos.write(new byte[] {13, 15});
         returnedBytes = sbaos.toByteArray(2);
         newBytes = new byte[sbaos.bytesReturned()];
         System.arraycopy(returnedBytes, sbaos.start(), newBytes, 0, sbaos.bytesReturned());
         assertTrue(newBytes.length == 2 && newBytes[0] == 11 && newBytes[1] == 13);
         returnedBytes = sbaos.toByteArray(2);
         newBytes = new byte[sbaos.bytesReturned()];
         System.arraycopy(returnedBytes, sbaos.start(), newBytes, 0, sbaos.bytesReturned());
         assertTrue(newBytes.length == 1 && newBytes[0] == 15);
      }
      catch (Exception e)
      {
         e.printStackTrace();
         fail();
      }
      
      log.info("testSmallArraysByUsage() PASSES");
      OKCounter++;
   }
   
   
   public void testLargeArraysShrinking()
   {
      log.info("entering: testLargeArraysShrinking()");
      
      try
      {
         ShrinkableByteArrayOutputStreamSubClass sbaos = new ShrinkableByteArrayOutputStreamSubClass();
         byte[] bytes = new byte[512];
         sbaos.write(bytes);
         assertTrue(sbaos.getLength() == 1024);
         sbaos.write(bytes);
         assertTrue(sbaos.getLength() == 1024);
         bytes = new byte[1];
         sbaos.write(bytes);
         assertTrue(sbaos.getLength() == 2048);
         sbaos.toByteArray(1025);
         bytes = new byte[64];
         
         for (int i = 0; i < 7; i++)
         {
            sbaos.write(bytes);
            sbaos.toByteArray(64);
         }
         
         sbaos.write(bytes);
         sbaos.toByteArray(62);
         assertTrue(sbaos.getLength() == 2048);
         sbaos.toByteArray(1);
         assertTrue(sbaos.getLength() == 2048);
         sbaos.toByteArray(1);
         assertTrue(sbaos.getLength() == 1024);
      } 
      catch (Exception e)
      {
         e.printStackTrace();
         fail();
      }
      
      log.info("testLargeArraysShrinking() PASSES");
      OKCounter++;
   }

   
   
   
   public void testLargeArraysNotGrowing()
   {
      log.info("entering: testLargeArraysNotGrowing()");
      
      try
      {
         ShrinkableByteArrayOutputStreamSubClass sbaos = new ShrinkableByteArrayOutputStreamSubClass();
         byte[] bytes = new byte[1024];			// 0, 0, 1024
         sbaos.write(bytes);
         assertTrue(sbaos.getLength() == 1024);
         sbaos.toByteArray(513);				// 0, 513, 1024
         sbaos.toByteArray(0);					// 513, 513, 1024
         sbaos.write(3);						// 0, 0, 512
         assertTrue(sbaos.getLength() == 1024);
         assertTrue(sbaos.toByteArray()[511] == 3);
         sbaos.toByteArray(512);				// 0, 512, 512
         sbaos.toByteArray(0);					// 512, 512, 512
         assertTrue(sbaos.available() == 0);
         
         bytes = new byte[512];
         sbaos.write(bytes);					// 512, 512, 1024
         assertTrue(sbaos.getLength() == 1024);
         sbaos.write(5);						// 512, 512, 1025
         assertTrue(sbaos.getLength() == 2048);
         assertTrue(sbaos.toByteArray()[1024] == 5);
      } 
      catch (Exception e)
      {
         e.printStackTrace();
         fail();
      }
      
      log.info("testLargeArraysNotGrowing() PASSES");
      OKCounter++;
   }
   
   
   public void testLargeArraysContent()
   {
      log.info("entering: testLargeArraysContent()");
      ShrinkableByteArrayOutputStreamSubClass sbaos = new ShrinkableByteArrayOutputStreamSubClass();
      int size = 64 * 1024;
      boolean pass = true;
      
      for (int i = 0; i < size; i++)
         sbaos.write(i);
      
      byte[] bytes = sbaos.toByteArray(size);
      
      for (int i = 0; i < size; i++)
      {
         if ((0xff & bytes[i]) != i % 256)
         {
            log.info("discrepancy [" + i + " (" + (i % 256) + ")]: " + (0xff & bytes[i]));
            pass = false;
         }
      }
      
      assertTrue(pass);
      log.info("testLargeArraysContent() PASSES");
      OKCounter++;
   }
   
   
   class ShrinkableByteArrayOutputStreamSubClass extends ShrinkableByteArrayOutputStream
   {
      public int getCount()
      {
         return super.getCount();
      }
      
      public int getLength()
      {
         return super.getLength();
      }
      
      public int getNextUsed()
      {
         return super.getNextUsed();
      }
      
      public int getUsed()
      {
         return super.getUsed();
      }

   }
}

