/* GraphListLoader.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.io.list;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.logging.Logger;

import org.grinvin.io.IOFormatException;
import org.grinvin.io.LoaderSaverHelper;
import org.grinvin.io.SectionLoader;
import org.grinvin.list.graphs.GraphList;
import org.grinvin.list.graphs.GraphListElement;
import org.grinvin.list.graphs.GraphListElementManager;

import org.jdom.Document;
import org.jdom.Element;

/**
 * Helper methods for loading lists of graphs and lists of invariants from
 * disk.
 */
public final class GraphListLoader {
    
    //
    private static final String PUBLIC_ID = "-//GrInvIn IO//GraphList 1.1//EN";
    
    // should not be instantiated
    private GraphListLoader () { }
    
    //
    private static final Logger LOGGER = Logger.getLogger("org.grinvin.io");
    
    //
    private static void loadGraphListElements(GraphList list, Element element, SectionLoader sloader, String graphURIAttribute) throws IOFormatException {
        
        for (Object obj : element.getChildren("graphlistelement")) {
            try {
                Element child = (Element)obj;
                String uriString = child.getAttributeValue(graphURIAttribute);
                if (uriString != null) {
                    URI uri = new URI(uriString);
                    GraphListElement graphlistelement 
                        = GraphListElementManager.getInstance().getGraphListElement(uri, sloader);
                    list.add(graphlistelement);
                } else {
                    throw new IOFormatException("invalid graph URI");
                }
            } catch (URISyntaxException ex) {
                throw new IOFormatException("invalid graph URI");
            }
        }
        
    }

    /**
     * Load a graph list from an XML file.
     * @param list List to which the graphs are added. It may be necessary to clear the list before calling
     * this method.
     * @param ilist Optional list of invariants. If non-null, any invariants stored in the XML-file will be
     * added to this list. It may be necessary to clear the list before calling
     * this method.
     * @param file XML file from which the list(s) will be loaded.
     * @param directory directory containing sssion graphs, or null when
     * session graphs are not supported
     * @see GraphListSaver#save(GraphList,InvariantList,File)
     */
    public static void load(GraphList list,
            InputStream input, SectionLoader sloader) throws IOException {
        Document document = LoaderSaverHelper.loadDocument(input);
        if (!PUBLIC_ID.equals(document.getDocType().getPublicID()))
            throw new IOFormatException("Unkown document type: " + document.getDocType().getPublicID() + " expects " + PUBLIC_ID);
        loadGraphListElements(list, document.getRootElement(), sloader, "uri");
    }

    //
    public static void load_1_0(GraphList list,
            InputStream input, SectionLoader sloader) throws IOException {
        loadGraphListElements(list, LoaderSaverHelper.loadElementNonValidating(input, "graphlistelements"), sloader, "graphURI");
    }
    
}
