/* EmbeddingModelSupport.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.graphs;

import javax.swing.event.EventListenerList;

/**
 * Helper class for implementations of {@link EmbeddingModel}. 
 * Models can delegate listener management and notification to
 * an object of this class.
 */
public class EmbeddingModelSupport {
  
    //
    private EventListenerList listenerList = new EventListenerList ();
    
    // [kc] for performance reasons
    private static final Class<EmbeddingListener>
            EMBEDDING_LISTENER_CLASS = EmbeddingListener.class;
    
    // implements EmbeddingModel
    public void addEmbeddingListener (EmbeddingListener l) {
        listenerList.add (EMBEDDING_LISTENER_CLASS, l);
    }
    
    // implements EmbeddingModel
    public void removeEmbeddingListener (EmbeddingListener l) {
        listenerList.remove (EMBEDDING_LISTENER_CLASS, l);
    }
    
    /**
     * Notify all listeners of the fact that coordinates of a vertex
     * were changed.
     */
    public void fireCoordinatesChanged (Vertex vertex) {
        Object[] listeners = listenerList.getListenerList ();
        for (int i = listeners.length-2; i>=0; i-=2)
            if (listeners[i]==EMBEDDING_LISTENER_CLASS)
                ((EmbeddingListener)listeners[i+1]).vertexCoordinatesChanged(vertex);
    }
    
    /**
     * Notify all listeners of the fact that the dimension of the embedding
     * was changed.
     */
    public void fireDimensionChanged (int oldDimension) {
        Object[] listeners = listenerList.getListenerList ();
        for (int i = listeners.length-2; i>=0; i-=2)
            if (listeners[i]==EMBEDDING_LISTENER_CLASS)
                ((EmbeddingListener)listeners[i+1]).dimensionChanged(oldDimension);
    }
    
    /**
     * Notify  all listeners of a bulk change to this embedding.
     */
    public void fireEmbeddingChanged () {
        Object[] listeners = listenerList.getListenerList ();
        for (int i = listeners.length-2; i>=0; i-=2)
            if (listeners[i]==EMBEDDING_LISTENER_CLASS)
                ((EmbeddingListener)listeners[i+1]).embeddingChanged ();
    }
    
}
