/*
 *      Copyright (C) 1998 Claus-Justus Heine.

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *
 *      This program computes all 16 bit CRC sums of all single byte
 *      data streams for the backpack CRC polynomial
 *
 *      x^16 + x^12 + x^5 + x^0 = 0x1021 ^ 0x10000
 *
 *      Note that backpack's point of view of incoming data bytes
 *
 *      a = (a_0, a_1, ..., a_7)
 *
 *      is
 *
 *      a(x) = a_0 x^0 + ... + a_7 x^7
 *
 *      The CRC sum of an incoming data stream (a^0, ..., a^n) where
 *
 *      a^i = (a^i_0, ..., a^i_7)
 * 
 *      then is the image of the polynomial
 *
 *      f(x) = \sum_{i=0}^{n}x^{8 (n-i)} (\sum_{j=0}^7 a^i_j x^j)
 *
 *      under the natural epimorphism \pi:GF(2)[x] -> GF(2)[x]/(p(x))
 *      where p(x) is the generating polynomial
 *
 *      To effectively compute the CRC of (a^0,...,a^n) we write the
 *      corresponding polynomial like follows:
 *
 *      f(x) = a^n(x) + x^8 (a^{n-1}(x) + x^8 ( ... + x^8 (a^0(x) + crc)))...))  
 *
 *      where "crc" is the initial value of the CRC checksum.  
 *
 *      The way backpack's CRC algorithm is implemented means that
 *      multiplication by x is a LEFT shift.
 *
 *      addition off polynomials is a simple exclusive OR.
 *
 *      Multiplication by x in the factor module GF(2)[x]/(p(x)) is
 *      LEFT shift followed XORing with 0x1021 (the generating
 *      polynomial without the leading power) if the bit shifted out
 *      was a one.  (in GF(2)[x]/(p(x)) we have x^16 = x^12 + x^5 +
 *      x^0)
 *
 */

char src[] = "$Source: /usr/src/oldsrc/cvs/ftape-stacked/ftape/parport/bpck-fdc-makecrc.c,v $";
char rev[] = "$Revision: 1.4 $";
char dat[] = "$Date: 1998/08/17 13:14:47 $";

#include <sys/types.h>
#include <stdio.h>

int reverse = 1; /* (a_0, ..., a_7) <-> a_0 x^0 + ... + a_7 x^7 */

void make_crc16(u_int16_t pol)
/*
 * Generate a table for a byte-wise 32-bit CRC calculation on the polynomial:
 *
 * x^16 + x^12 + x^5 + x^0
 *
 *  The table is simply the CRC of all possible eight bit values.
 *  This is all the information needed to generate CRC's on data a
 *  byte at a time for all combinations of CRC register values and
 *  incoming bytes.  The table is written to stdout as 256 long
 *  hexadecimal values in C language format.
 */

{
	u_int16_t c;      /* crc shift register */
	u_int16_t e;      /* polynomial exclusive-or pattern */
	int i;            /* counter for all possible eight bit values */
	int k;            /* byte being shifted into crc apparatus */

  /* terms of polynomial defining this crc (except x^32): */
	int p[16]; /*  = {0,1,2,4,5,7,8,10,11,12,16,22,23,26}; */
	int nc;           /* number of non-zero coeff. */

	c = pol;
	nc = 0;
	k = 0;
	while (c) {
		if (reverse) {
			if (c & 0x0001) {
				p[nc++] =  k;			
			}
			k++;
			c >>= 1;
		} else {
			if (c & 0x8000) {
				p[nc++] =  k;			
			}
			k++;
			c <<= 1;
		}
	} 

	printf("/* automatically generated by makecrc\n"\
	       " *\n"\
	       " * Polynomial: x^%d", p[0]);
	for (i = 1; i < nc; i++) {
		printf(" + x^%d", p[i]);
	}
	printf("\n");

	/* Make exclusive-or pattern from polynomial (0xedb88320) */
	e = 0;
	if (reverse) {
		for (i = 0; i < nc; i++) {
			e |= 1L << p[i];
		}
	} else {
		for (i = 0; i < nc; i++) {
			e |= 1L << (15 - p[i]);
		}
	}

	printf(" * binary representation of the polynomial: 0x%04x\n */\n", e);
	printf("#ifndef _FT_CRC_H_\n");
	printf("#define _FT_CRC_H_\n");
	printf("\nstatic const __u16 crc_tab[] = {");
	printf("\n\t0x0000,");
	/* crc_16_tab[0] = 0x0000; */
	for (i = 1; i < 256; i++) {
		if (reverse) {
			c = i << 8;
			/* The idea to initialize the register with
			 * the byte instead of zero was stolen from
			 * Haruhiko Okumura's ar002
			 */
			for (k = 0; k < 8; k++) {
				c = c & 0x8000 ? (c << 1) ^ e : c << 1;
			}
		} else {
			c = i;
			/* The idea to initialize the register with
			 * the byte instead of zero was stolen from
			 * Haruhiko Okumura's ar002
			 */
			for (k = 0; k < 8; k++) {
				c = c & 1 ? (c >> 1) ^ e : c >> 1;
			}
		}

		/* crc_16_tab[i] = c; */		
		if (i % 4 == 0) {
			printf("\n\t");
		} else {
			printf(" ");
		}
		printf("0x%04x,", c);
	}
	printf("\n};\n");
	printf("#endif\n");
}

int main(int argc, char *argv[])
{
	make_crc16(0x1021);	
	exit(0);
}
