"""
  prc.py - read/write prc/pdb files in pure Python
  $Id: prc.py,v 1.2 1998/08/29 01:09:00 rob Exp $

  Copyright 1998 Rob Tillotson <rob@io.com>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License, version 2,
  as published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU Library General Public License
  along with this program; if not, write the Free Software Foundation,
  Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.


  This module serves two purposes: one, it allows access to Pilot
  database files on the desktop in pure Python without requiring
  pilot-link (hence, it may be useful for import/export utilities),
  and two, it caches the contents of the file in memory so it can
  be freely modified using an identical API to databases over a
  DLP connection.
  
"""

#
# DBInfo structure:
#
#   int more
#   unsigned int flags
#   unsigned int miscflags
#   unsigned long type
#   unsigned long creator
#   unsigned int version
#   unsigned long modnum
#   time_t createDate, modifydate, backupdate
#   unsigned int index
#   char name[34]
#
#
# DB Header:
#   32 name
#   2  flags
#   2  version
#   4  creation time
#   4  modification time
#   4  backup time
#   4  modification number
#   4  appinfo offset
#   4  sortinfo offset
#   4  type
#   4  creator
#   4  unique id seed (garbage?)
#   4  next record list id (normally 0)
#   2  num of records for this header
#   (maybe 2 more bytes)
#
# Resource entry header: (if low bit of attr = 1)
#   4  type
#   2  id
#   4  offset
#
# record entry header: (if low bit of attr = 0)
#   4  offset
#   1  attributes
#   3  unique id
#
# then 2 bytes of 0
#
# then appinfo then sortinfo
#

import sys, os, stat, struct

PI_HDR_SIZE = 78
PI_RESOURCE_ENT_SIZE = 10
PI_RECORD_ENT_SIZE = 8

PILOT_TIME_DELTA = 2082844800

flagResource = 0x0001
flagReadOnly = 0x0002
flagAppInfoDirty = 0x0004
flagBackup = 0x0008
flagOpen = 0x8000
# 2.x
flagNewer = 0x0010
flagReset = 0x0020
#
flagExcludeFromSync = 0x0080

attrDeleted = 0x80
attrDirty = 0x40
attrBusy = 0x20
attrSecret = 0x10
attrArchived = 0x08

default_info = {
    'name': '',
    'type': 'DATA',
    'creator': '    ',
    'createDate': 0,
    'modifyDate': 0,
    'backupDate': 0,
    'modnum': 0,
    'version': 0,
    'flagReset': 0,
    'flagResource': 0,
    'flagNewer': 0,
    'flagExcludeFromSync': 0,
    'flagAppInfoDirty': 0,
    'flagReadOnly': 0,
    'flagBackup': 0,
    'flagOpen': 0,
    'more': 0,
    'index': 0
    }

def null_terminated(s):
    for x in range(0, len(s)):
	if s[x] == '\000': return s[:x]
    return s


class File:
    def __init__(self, name=None, mode='rb'):
	self.filename = None
	self.records = []
	self.appinfo = ''
	self.sortinfo = ''
	self.info = {}
	self.info.update(default_info)
	self.next = 0
	
	if type(name) == type(''):
	    self.open(name, mode)
	elif name is not None:
	    self.f = name
	else:
	    self.f = None


    def open(self, name, mode='rb'):
	# pdb files cannot be appended to or overwritten
	if mode not in ['r', 'rb', 'w', 'wb']:
	    raise ValueError, 'invalid file mode "%s"' % mode
	# make sure to use binary mode only
	if 'b' not in mode: mode = mode + 'b'
	self.f = open(name, mode)
	self.filename = name

    def load(self):
	file_length = os.stat(self.filename)[6]
	hstr = self.f.read(PI_HDR_SIZE)
	if not hstr or len(hstr) < PI_HDR_SIZE:
	    raise IOError, 'prc/pdb header too short'

	(name, flags, ver, ctime, mtime, btime, mnum, appinfo, sortinfo,
	 type, creator, uid, nextrec, numrec) \
	 = struct.unpack('>32shhLLLlll4s4sllh', hstr)

	if nextrec or appinfo < 0 or sortinfo < 0 or numrec < 0:
	    raise IOError, 'invalid prc/pdb header'

	self.info = {
	    'name': null_terminated(name),
	    'type': type,
	    'creator': creator,
	    'createDate': ctime - PILOT_TIME_DELTA,
	    'modifyDate': mtime - PILOT_TIME_DELTA,
	    'backupDate': btime - PILOT_TIME_DELTA,
	    'modnum': mnum,
	    'version': ver,
	    'flagReset': flags & flagReset,
	    'flagResource': flags & flagResource,
	    'flagNewer': flags & flagNewer,
	    'flagExcludeFromSync': flags & flagExcludeFromSync,
	    'flagAppInfoDirty': flags & flagAppInfoDirty,
	    'flagReadOnly': flags & flagReadOnly,
	    'flagBackup': flags & flagBackup,
	    'flagOpen': flags & flagOpen,
	    'more': 0,
	    'index': 0
	 }

	# read entries
	rsrc = flags & flagResource
	if rsrc: s = PI_RESOURCE_ENT_SIZE
	else: s = PI_RECORD_ENT_SIZE

	entries = []

	for x in range(0,numrec):
	    hstr = self.f.read(s)
	    if not hstr or len(hstr) < s:
		raise IOError, 'bad prc/pdb header'

	    if rsrc:
		(typ, id, offset) = struct.unpack('>lhl', hstr)
		entries.append((offset, typ, id))
	    else:
		(offset, attr, uid0, uid1, uid2) = struct.unpack('>lbbbb', hstr)
		uid = uid0 << 16 | uid1 << 8 | uid2
		entries.append((offset, attr, uid))

	# read records, from the end of the file back
	# (since we have to go backwards calculating offsets anyway)
	offset = file_length
	entries.reverse()
	for of, q, id in entries:
	    size = offset - of
	    if size < 0: raise IOError, 'bad pdb/prc record entry (size < 0)'
	    offset = of
	    self.f.seek(of)
	    d = self.f.read(size)
	    if len(d) != size: raise IOError, 'failed to read record'
	    self.records.append( [id, q, d] )

	self.records.reverse()

	if sortinfo:
	    sortinfo_size = offset - sortinfo
	    offset = sortinfo
	else:
	    sortinfo_size = 0

	if appinfo:
	    appinfo_size = offset - appinfo
	    offset = appinfo
	else:
	    appinfo_size = 0

	if appinfo_size < 0 or sortinfo_size < 0:
	    raise IOError, 'bad pdb/prc header (appinfo or sortinfo size < 0)'

	if appinfo_size:
	    self.f.seek(appinfo)
	    self.appinfo = self.f.read(appinfo_size)
	    if len(self.appinfo) != appinfo_size:
		raise IOError, 'failed to read appinfo block'

	if sortinfo_size:
	    self.f.seek(sortinfo)
	    self.sortinfo = self.f.read(sortinfo_size)
	    if len(self.sortinfo) != sortinfo_size:
		raise IOError, 'failed to read sortinfo block'

	    
    # pi-file API
    def getRecords(self):
	return len(self.records)

    def getAppBlock(self):
	return self.appblock

    def setAppBlock(self, blk):
	self.appblock = blk

    def getSortBlock(self):
	return self.sortblock

    def setSortBlock(self, blk):
	self.sortblock = blk

    def checkID(self, id):
	return id in map(operator.getitem, self.records, [0]*len(self.records))

    def getRecord(self, i):
	try:
	    id, attr, d = self.records[i]
	except:
	    return ''
	return (d, i, id, attr & 0xf0, attr & 0x0f)

    def getRecordByID(self, id):
	ids = map(operator.getitem, self.records, [0]*len(self.records))
	try:
	    i = ids.index(id)
	except:
	    return ''
	return self.getRecord(i)
    
    def getResource(self, i):
	try:
	    id, typ, d = self.records[i]
	except:
	    return ''
	return (d, typ, id)

    def addRecord(self, attr, id, cat, data):
	self.records.append( [id, attr & 0xf0 | cat & 0x0f, data] )
	return id

    def addResource(self, typ, id, data):
	self.records.append( [id, typ, data] )
	return ''

    def getDBInfo(self):
	return self.info

    def setDBInfo(self, info):
	self.info = {}
	self.info.update(info)

    def install(self, dlp):
	raise RuntimeError, 'DLP operations not supported'

    def retrieve(self, dlp):
	raise RuntimeError, 'DLP operations not supported'

    # DLP database api
    def setRecord(self, attr, id, cat, data):
	# yes, it is indeed true that this always puts the replacement
	# record at the END of the database.  That's what the Pilot does...
	ids = getRecordIDs(self)
	if id and id in ids:
	    self.deleteRecord(id)
	self.addRecord(attr, id, cat, data)
	return id
    
    def setResource(self, typ, id, data):
	ids = getRecordIDs(self)
	if id and id in ids:
	    self.deleteResource(typ, id)
	self.addResource(typ, id, data)
	return id

    def getNextRecord(self, cat):
	while self.next < len(self.records):
	    r = self.records[self.next]
	    i = self.next
	    self.next = self.next + 1
	    if (r[1] & 0x0f) == cat:
		return (r[2], i, r[0], r[1] & 0xf0, r[1] & 0x0f)
	return ''

    def getNextModRecord(self, cat=-1):
	while self.next < len(self.records):
	    r = self.records[self.next]
	    i = self.next
	    self.next = self.next + 1
	    if (r[1] & attrModified) and (cat < 0 or (r[1] & 0x0f) == cat):
		return (r[2], i, r[0], r[1] & 0xf0, r[1] & 0x0f)
	return ''

    def getResourceByID(self, id):
	ids = map(operator.getitem, self.records, [0]*len(self.records))
	try:
	    i = ids.index(id)
	except:
	    return ''
	return self.getResource(i)

    def deleteRecord(self, id):
	ids = self.getRecordIDs()
	try:
	    i = ids.index(id)
	except:
	    pass
	del self.records[i]

    def deleteRecords(self):
	self.records = []

    def deleteResource(self, type, id):
	pass

    def deleteResources(self):
	pass

    def getRecordIDs(self, sort=0):
	ids = map(operator.getitem, self.records, [0]*len(self.records))
	if sort: ids.sort()
	return ids

    def moveCategory(self, frm, to):
	for r in self.records:
	    if (r[1] & 0x0f) == frm:
		r[1] = r[1] & 0xf0 | to & 0x0f

    def deleteCategory(self, cat):
	pass

    def purge(self):
	self.records = filter(lambda x: not (x[1] & attrDeleted), self.records)

    def resetNext(self):
	self.next = 0

    def resetFlags(self):
	# reset dirty flags
	if not self.info.get('flagResource', 0):
	    for x in self.records:
		x[1] = x[1] & ~attrDirty
	# reset sync time ... not possible?
	
    def getPref(self, creator, id=0, backup=1):
	pass

    def setPref(self, creator, id, backup, version, data):
	pass

	
