"""
  PDA.Palm.Tk.DBFileDialog - File select dialog optimized for PalmOS DBs
  $Id: DBFileDialog.py,v 1.4 1998/08/29 01:09:02 rob Exp $

  Copyright 1998 Rob Tillotson <rob@io.com>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License, version 2,
  as published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU Library General Public License
  along with this program; if not, write the Free Software Foundation,
  Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.


  The standard file selector boxes provided with Tk (and other GUI systems,
  for that matter) are nice, but it would be nice if something a little more
  PalmOS-oriented were available for applications based on this library.
  This file selector dialog is an attempt to provide that.

  It resembles other Unixish file selector boxes in most respects -- two
  scrollable listboxes, etc.  It does several things they (usually) do not:

    - the 'Info' button pops up a dialog showing the PalmOS database
      header of the (first) currently selected file

    - only PalmOS databases are shown in the right-side listbox.  Whether
      a file is a PalmOS database is determined by scanning the header,
      not by name.  (I may change this in the future, as it is somewhat
      slow and prone to error; on the other hand, it eliminates the
      dependence on file extensions.  Comments would be appreciated.)

    - the name of each database is shown in the right-side listbox in
      addition to (or instead of) the filename.

    - the right-side listbox uses the 'extended' selection mode, which
      allows multiple selection in the Windows style (shift-click to
      select a range, control-click to select multiple ranges)

  The utility function "askDBFile" simply pops up the dialog and returns
  filenames that were selected.

  Use:
    DBFileDialog(parent, ...) - creates a DBFileDialog

      Keyword arguments:
        path (string)       - the starting path
	multiple (boolean)  - if true, allow multiple file selection
	filenames (boolean) - if true, show filenames
	dbnames (boolean)   - if true, show dbnames

      Instance variables:
        filenames           - holds the selection *after* the dialog is done

"""


from Tkinter import *
import tkSimpleDialog

from PDA.Palm import pdbfile
from PDA.Palm.Tk.DBInfoDialog import DBInfoDialog

import sys, os, string


class DBFileDialog(tkSimpleDialog.Dialog):
    def __init__(self, parent, path='.', multiple=1, filenames=1, dbnames=1):
	self.path = os.path.normpath(os.path.join(os.getcwd(), path))
	self.filenames = []
	self.opt_multiple = multiple
	self.opt_filenames = filenames
	self.opt_dbnames = dbnames
	tkSimpleDialog.Dialog.__init__(self, parent, 'DB File Browser')

    def body(self, p):
	top = Frame(p)
	middle = Frame(p)

	# Top frame: path and buttons
	w = Label(top, text='Path:')
	w.grid(row=0, column=0, sticky=W)
	
	self.path_label = Label(top, text=self.path)
	self.path_label.grid(row=0, column=1, sticky=W)

	w = Button(top, text="Up", command=self.parent_dir)
	w.grid(row=0, column=2, sticky=E)

	w = Button(top, text="Info", command=self.pdb_info)
	w.grid(row=0, column=3, sticky=E)
	
	top.columnconfigure(1, weight=1)

	# Middle frame... listboxes
	self.listbox_dir = Listbox(middle, selectmode=BROWSE)
	self.listbox_dir.grid(row=0, column=0, sticky='news')
	self.listbox_dir.bind('<Double-1>', self.doubleclick_dir)
	self.vscroll_dir = Scrollbar(middle, orient='vertical',
				     command=self.listbox_dir.yview)
	self.vscroll_dir.grid(row=0, column=1, sticky='ns')
	self.hscroll_dir = Scrollbar(middle, orient='horizontal',
				     command=self.listbox_dir.xview)
	self.hscroll_dir.grid(row=1, column=0, sticky='ew')
	self.listbox_dir.configure(xscrollcommand=self.hscroll_dir.set,
				  yscrollcommand=self.vscroll_dir.set)
				  
	self.listbox = Listbox(middle,
			       selectmode=self.opt_multiple and EXTENDED or BROWSE)

	self.listbox.grid(row=0, column=2, sticky='news')
	self.listbox.bind('<Double-1>', self.doubleclick)
	
	middle.columnconfigure(2, weight=1)
	middle.columnconfigure(0, weight=1)

	self.vscroll = Scrollbar(middle, orient='vertical',
				 command = self.listbox.yview)
	self.vscroll.grid(row=0, column=3, sticky='ns')

	self.hscroll = Scrollbar(middle, orient='horizontal',
				 command = self.listbox.xview)
	self.hscroll.grid(row=1, column=2, sticky='ew')
	self.listbox.configure(xscrollcommand = self.hscroll.set,
			       yscrollcommand = self.vscroll.set)
	
	self.fill_listboxes()

	top.grid(row=0, sticky='ew')
	middle.grid(row=1, sticky='news')
	p.columnconfigure(0, weight=1)
	
	return self.listbox

    def curselection(self):
	l = self.listbox.curselection()
	try:
	    l = map(string.atoi, l)
	except ValueError:
	    pass
	return l

    def chdir(self, d):
	if self.path != '/' or d != '..':
	    self.path = os.path.normpath(os.path.join(self.path, d))
	    self.path_label.configure(text=self.path)
	    self.fill_listboxes()
	
    def doubleclick_dir(self, event):
	l = self.listbox_dir.curselection()
	try:
	    l = map(string.atoi, l)
	except ValueError:
	    pass

	if l:
	    d = self.subdir_list[l[0]]
	    self.chdir(d)
    
    def doubleclick(self, event):
	self.ok()

    def apply(self):
	l = self.curselection()
	if l:
	    self.filenames = map(lambda x, f=self.file_list, p=self.path:
				 os.path.join(p,f[x][0]), l)
	else:
	    self.filenames = []
	
    def pdb_info(self, event=None):
	l = self.curselection()
	if l:
	    d = self.file_list[l[0]]
	    DBInfoDialog(self.parent, d[1])
	
    def parent_dir(self, event=None):
	"""Go up a directory when the Up button is pressed."""
	self.chdir('..')

    def fill_listboxes(self):
	"""Fill the listbox with information about the current directory."""
	l = self.listbox_dir
	l.delete(0, END)

	self.subdir_list = pdbfile.listsubdirs(self.path)
	self.subdir_list.sort()
	for n in self.subdir_list:
	    l.insert(END, '%s' % n)

	l = self.listbox
	l.delete(0, END)

	if self.opt_filenames and self.opt_dbnames:
	    fmt = '%(filename)s (%(dbname)s)'
	elif self.opt_dbnames:
	    fmt = '%(dbname)s'
	else:
	    fmt = '%(filename)s'
	    
	self.file_list = pdbfile.listdir(self.path)
	self.file_list.sort()
	for n, i in self.file_list:
	    l.insert(END, fmt % {'filename':n, 'dbname': i.get('name')})


# utility functions
def askDBFile(parent, *a, **kw):
    return apply(DBFileDialog, (parent,)+a, kw).filenames
