"""
  PDA.Palm.Config - Configuration support
  $Id: Config.py,v 1.3 1998/09/01 11:02:50 rob Exp $

  Copyright 1998 Rob Tillotson <rob@io.com>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License, version 2,
  as published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU Library General Public License
  along with this program; if not, write the Free Software Foundation,
  Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.


"""

__version__ = '$Id: Config.py,v 1.3 1998/09/01 11:02:50 rob Exp $'

import sys, os, string, re, pwd, stat, shutil

from PDA.Palm import protect_name, unprotect_name

system_conf = '/etc/palm.conf'
user_conf = '~/.palm.conf'
palm_user_conf = 'palm.conf'
data_dir = '/var/palm'

rx_section = '^\[(.*)\]'
rx_variable = '^([^ \t=]+)\s*=\s*(.*)$'
rx_comment = '^#'

def _mkdir(d):
    if os.path.exists(d):
	if not os.path.isdir(d):
	    raise RuntimeError, '%s exists, but is not a directory' % d
    else:
	os.mkdir(d)

def _isdir(d):
    if not os.path.exists(d):
	raise RuntimeError, '%s does not exist' % d
    if not os.path.isdir(d):
	raise RuntimeError, '%s is not a directory' % d
    
class Registry:
    """A 'sectioned dictionary' containing configuration information."""
    def __init__(self):
	self.data = {}
	self.override_vars = 1

    def __str__(self):
	s = ''
	for k in self.data.keys():
	    s = s + '[%s]\n' % k
	    for l in self.data[k].keys():
		s = s + '%s = %s\n' % (l, self.data[k][l])
	    s = s + '\n'
	return s

    def save_file(self, fn):
	f = open(fn, 'w')
	f.write(str(self))
	f.close()
	
    def load_file(self, fn, merge=1):
	if not merge: self.data = {}
	f = open(fn, 'r')

	sec = None
	for l in map(string.strip, f.readlines()):
	    if not l or re.match(rx_comment, l):
		continue
	    
	    m = re.match(rx_section, l)
	    if m:
		sec = m.group(1)
		if not self.data.has_key(sec): self.data[sec] = {}
		continue

	    m = re.match(rx_variable, l)
	    if m:
		n, v = m.group(1,2)
		if self.override_vars or not self.data[sec].has_key(n):
		    self.data[sec][n] = v
		else:
		    if type(self.data[sec][n]) != type([]):
			self.data[sec][n] = [self.data[sec][n], v]
		    else:
			self.data[sec][n].append(v)

    def has_section(self, k):
	return self.data.has_key(k)

    def has_key(self, sec, var):
	return self.has_section(sec) and self.data[sec].has_key(var)

    def del_key(self, sec, var):
	if self.has_key(sec, var):
	    del self.data[sec][var]

    def del_section(self, sec):
	if self.data.has_key(sec):
	    del self.data[sec]
	    
    def get(self, sec, var, default=None):
	try:
	    return self.data[sec][var]
	except KeyError:
	    return default

    def set(self, sec, var, value):
	if not self.data.has_key(sec):
	    self.data[sec] = {}
	self.data[sec][var] = value
	
class PalmRegistry(Registry):
    def __init__(self):
	Registry.__init__(self)
	self.user = ''
	self.uid = 0
	
    def load(self):
	"""Load standard configuration files."""
	try: Registry.load_file(self, system_conf)
	except IOError: pass
	try: Registry.load_file(self, os.path.expanduser(user_conf))
	except IOError: pass

    def load_user(self, name, id=None):
	"""Load per-Palm-user configuration."""
	self.user = name
	self.uid = id
	d = self.user_directory()
	if d:
	    try: Registry.load_file(self, os.path.join(d, palm_user_conf))
	    except IOError: pass

    def port(self):
	if self.has_key('defaults','default-port'):
	    return self.get('defaults','default-port')
	else:
	    p = self.get('defaults','port','/dev/pilot')
	    if type(p) == type([]): return p[0]
	    else: return p
	    
    def data_directory(self):
	return self.get('defaults','data-directory',data_dir)

    def user_directory(self, u=None, id=None):
	if u is None: u = self.user
	if id is None: id = self.uid
	p = self.data_directory()
	if self.uid:
	    d = os.path.join(p, protect_name('%s.%s' % (u, id)))
	    try:
		os.stat(d)
		return d
	    except os.error:
		pass

	d = os.path.join(p, protect_name(u))
	try:
	    os.stat(d)
	    return d
	except os.error:
	    return None
	
    
    def expand_str(self, s):
	#
	#  %U - unix user name
	#  %h - home directory
	#  %m - machine name
	#  %F - full name
	#  %u - palm user name
	#  %p - palm path
	#  %P - palm user path
	#
	l = []
	for x in re.split('(%.)', s):
	    if x == '%%': l.append('%')
	    elif x == '%U': l.append(pwd.getpwuid(os.getuid())[0])
	    elif x == '%h': l.append(os.environ.get('HOME',''))
	    elif x == '%m': l.append(os.uname()[1])
	    elif x == '%F': l.append(string.split(pwd.getpwuid(os.getuid())[4],',')[0])
	    elif x == '%u': l.append(self.user)
	    elif x == '%p': l.append(self.data_directory())
	    elif x == '%P': l.append(self.user_directory())
	    else: l.append(x)
	return string.join(l, '')
    
    # user support
    def user_directory_name(self, u, id=None):
	if id is not None:
	    return os.path.join(self.data_directory(),
				protect_name('%s.%s' % (u, id)))
	else:
	    return os.path.join(self.data_directory(), protect_name(str(u)))
	
    def has_user(self, u=None, id=None):
	return self.user_directory(u, id) is not None

    def add_user(self, u, id=None):
	n = self.user_directory_name(u, id)
	_mkdir(n)
	_mkdir(os.path.join(n, 'backup'))
	_mkdir(os.path.join(n, 'install'))

    def delete_user(self, u, id=None):
	n = self.user_directory(u, id)
	if n:
	    shutil.rmtree(n, 1)
	    
    def rename_user(self, u, new_u, id=None, new_id=None):
	o = self.user_directory(u, id)
	n = self.user_directory_name(new_u, new_id)
	os.rename(o, n)

    def list_users(self):
	l = []
	p = self.data_directory()
	d = os.getcwd()
	os.chdir(p)
	for dir in filter(os.path.isdir, os.listdir('.')):
	    os.chdir(os.path.join(p, dir))
	    # This is kind of hackish, but it is the best way I can think of
	    # to do this short of putting some sort of magic cookie in each
	    # user directory.
	    if os.path.isdir('backup') and os.path.isdir('install'):
		l.append(unprotect_name(dir))
	os.chdir(d)
	l.sort()
	return l

###
def load(user = None, uid=None):
    """Load the configuration assuming the passed user, or default.
    Should only be used for offline programs; programs which connect
    to the PalmPilot should use the SyncManager instead.
    """
    r = PalmRegistry()
    r.load()
    if not user:
	user = r.get('defaults','user',None)
	uid = r.get('defaults','uid',None)
    if user:
	r.load_user(user, uid)
    return r, user, uid
