/*
libutil -- misc nntp-related stuff

Written by Arnt Gulbrandsen <agulbra@troll.no> and copyright 1995
Troll Tech AS, Postboks 6133 Etterstad, 0602 Oslo, Norway, fax +47
22646949.
Modified by Cornelius Krasel <krasel@wpxx02.toxi.uni-wuerzburg.de>
and Randolf Skerka <rskerka@metronet.de>.
Copyright of the modifications 1997.
Modified by Kent Robotti <robotti@erols.com>. Copyright of the
modifications 1998.
Modified by Markus Enzenberger <enz@cip.physik.uni-muenchen.de>.
Copyright of the modifications 1998.
Modified by Cornelius Krasel <krasel@wpxx02.toxi.uni-wuerzburg.de>.
Copyright of the modifications 1998.

See file COPYING for restrictions on the use of this software.
*/

#include <fcntl.h>
#include <sys/uio.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <stdlib.h>
#include <netdb.h>
#include <setjmp.h>
#include <signal.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <time.h>
#include <unistd.h>
#include <dirent.h>

#include "leafnode.h"

char last_command[1025];
char lineout[1025];

FILE *nntpin;
FILE *nntpout;

static jmp_buf timeout;

int authenticated;

static void timer(int sig) {
    longjmp(timeout, 1);
    exit(sig);
}

/*
05/26/97 - T. Sweeney - Send a string out, keeping a copy in reserve.
*/

void putaline( void ) {
    syslog( LOG_DEBUG, ">%s\n", lineout );
    strcpy(last_command, lineout);
    fprintf(nntpout, "%s", lineout);
    fflush(nntpout);
}

/* originally from Tim Sweeney
 *
 * Returns TRUE if authentication succeeds, FALSE if it does not.
 *
 *  Precondition: username != 0.
 */

int authenticate( void ) {
    int reply;

    fprintf( nntpout, "authinfo user %s\r\n", username );
    fflush( nntpout );

    reply = nntpreply();
    if (reply == 281) {
	return TRUE;
    } else if ( reply != 381 ) {
	syslog( LOG_INFO, "username rejected: %03d", reply);
	return FALSE;
    }

    if (password == NULL) {
	syslog( LOG_INFO, "password needed for authentication" );
	return FALSE;
    }
    fprintf( nntpout, "authinfo pass %s\r\n", password );
    fflush( nntpout );

    reply = nntpreply();

    if ( reply != 281) {
	syslog( LOG_INFO, "password failed: %03d", reply);
	return FALSE;
    }
    return TRUE;
}


/*
 * decode an NNTP reply number
 * reads a line from the server and returns an integer
 *
 * 498 is used to mean "protocol error", like smail
 *
 * the text returned is discarded
 *
 * from Tim Sweeney: retry in case of authinfo failure.
 */

int nntpreply(void) {
    char *response;
    int r = 0;
    int c = 1;

    while (c) {
	response=getaline(nntpin);
	if (!response) {
	    syslog( LOG_ERR, "NNTP server went away" );
	    return 498;
	}
	if (strlen(response)>2
	    && isdigit(response[0])
	    && isdigit(response[1])
	    && isdigit(response[2])
	    && ( (response[3]==' ')
		 || (response[3]=='\0')
		 || (response[3]=='-') ) ) {
	    int rl;
	    rl = atoi(response);
	    if (r>0 && r!=rl)
		r = 498;    /* protocol error */
	    else
		r = rl;
	    c = (response[3]=='-');
	} else {
	    c = 0;
	    r = 498;	/* protocol error */
	}
    }

    if (r == 480 && username && !authenticated ) { /* need to authenticate */
	authenticated = TRUE;
	if ( authenticate() ) {
	    fprintf(nntpout, "%s", last_command);
	    fflush(nntpout);
	    r = nntpreply();
	}
    }
    return r;
}


extern int errno;
extern int h_errno;
extern struct state _res;

#define incopy(a)       *((struct in_addr *)a)

/*
 * connect to upstream nntp server
 *
 * returns non-zero if a connection could be established
 */
int nntpconnect( const char * upstream ) {
    struct hostent *hp;
    struct servent *sp;
    struct servent sp_def;
    struct sockaddr_in s_in;
    int sock;
    register int i;

    memset((void *)&s_in, 0, sizeof(s_in));
    if (nntpport == 0) {
	sp = getservbyname("nntp", "tcp");
	if (sp == NULL) {
	    syslog( LOG_ERR, "unable to find service NNTP" );
	    return FALSE;
	}
    } else {
	sp=&sp_def;
	sp->s_port=htons(nntpport);
    }

    /* Fetch the ip addresses of the given host. */
    hp = gethostbyname( upstream );
    if (hp) {

	/* Try to make connection to each of the addresses in turn. */
	for (i = 0; (int *)(hp->h_addr_list)[i]; i++) {
	    s_in.sin_family = hp->h_addrtype;
	    s_in.sin_port = sp->s_port;
	    s_in.sin_addr = incopy(hp->h_addr_list[i]);

	    sock = socket(AF_INET, SOCK_STREAM, 0);
	    if (sock < 0)
		break;

	    if (setjmp(timeout) != 0) {
		(void) close(sock);
		continue;
	    }

	    (void) signal(SIGALRM, timer);
	    (void) alarm((unsigned)10); /* 10 seconds to open conn */
	    if (connect(sock, (struct sockaddr *)&s_in, sizeof(s_in)) < 0)
		break;
	    (void) alarm((unsigned)0);

	    nntpout = fdopen(sock, "w");
	    if (nntpout == NULL)
		break;

	    nntpin  = fdopen(dup(sock), "r");
	    if (nntpin == NULL)
		break;

	    switch(nntpreply()) {
	    case 200:
	    case 201:
		syslog( LOG_INFO, "connected to %s",
			inet_ntoa( s_in.sin_addr ) );
		return TRUE;
	    }
	    shutdown(fileno(nntpout), 0);
	}/* end of IP-addresses for loop */
    }
    return FALSE;
}/* end of connect function */


/*
 * read size bytes into buf from file fp,
 * with four hours timeout
 * return a pointer to buf, or NULL in case of error
 */
static char *sfgets(char *buf, size_t size, FILE *fp) {
    register char *p;

    if (setjmp(timeout)) {
	return(NULL);
    }

    (void) signal(SIGALRM, timer);
    (void) alarm(14400U);    /* four hours read timeout */
    p = fgets(buf, size, fp);
    if (errno == EINTR)
	errno = ETIMEDOUT;
    (void) alarm(0U);
    return p;
}


/*
 * Lars Wirzenius: read a line into memory, with no max length
 * return a pointer to the line, or NULL in case of error
 *
 * strip \r at EOL
 */
char *getaline(FILE *f) {
    static char *buf;       /* buffer for line */
    static size_t size;     /* size of buffer */
    size_t len;             /* # of chars stored into buf before '\0' */
    char * p;

    len = 0;
    if (!size)
	size = 256;
    if (!buf)
	buf = critmalloc( size, "reading line" );

    while ((p=sfgets(buf+len, size-len, f)) != NULL) {
	len += strlen(buf+len);
        if (len > 0 && buf[len-1] == '\n')
	    break;          /* the whole line has been read */

	size += size+100;
	buf = critrealloc(buf, size, "reading line" );
    }

    if ( len == 0 )
	return NULL;

    if (len && (buf[len-1] == '\n')) { /* go back on top of the newline */
	--len;
	if (len && (buf[len-1] == '\r')) /* also delete CR */
	    --len;
    }

    buf[len] = '\0';        /* unconditionally terminate string,
                               possibly overwriting newline */

    if ( debug )
        syslog( LOG_DEBUG, "<%s\n", buf );
    return buf;
}
