!**************************************************************************
!*
!* Boot-ROM-Code to load an operating system across a TCP/IP network.
!*
!* Module:  memory.S
!* Purpose: Memory management functions for DOS simulator
!* Entries: dos48, dos49, dos4A, freeall, checkmcb
!*
!**************************************************************************
!*
!* Copyright (C) 1995-1998 Gero Kuhlmann <gero@gkminix.han.de>
!*
!*  This program is free software; you can redistribute it and/or modify
!*  it under the terms of the GNU General Public License as published by
!*  the Free Software Foundation; either version 2 of the License, or
!*  any later version.
!*
!*  This program is distributed in the hope that it will be useful,
!*  but WITHOUT ANY WARRANTY; without even the implied warranty of
!*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!*  GNU General Public License for more details.
!*
!*  You should have received a copy of the GNU General Public License
!*  along with this program; if not, write to the Free Software
!*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
!*


!
!**************************************************************************
!
! Include assembler macros:
!
#include <macros.inc>
#include <memory.inc>
#include "./dospriv.inc"


!
!**************************************************************************
!
! BSS segment
!
	.bss

	extrn	curr_psp		! current PSP segment

	.comm	first_mcb,2		! pointer to first MCB


!
!**************************************************************************
!
! Start code segment.
!
	.text

	public	dos48
	public	dos49
	public	dos4A		! define entry poimts
	public	freeall
	public	checkmcb


!
!**************************************************************************
!
! Allocate memory block. This routine is very easy in that it allocates
! the first possible block it can find. This is not very economical, but
! sufficient for the DOS simulator.
! Note that this routine might also get called from outside a DOS interrupt,
! so it should not use any reference to the DOS stack structure!
! Input:  BX  -  number of paragraphs
! Output: AX  -  segment address or error code, if carry flag set
!         BX  -  maximum available number of paragraphs
!         Carry flag  -  set if error
! Registers changed: AX, BX
!
dos48:

	push	es
	push	dx
	xor	dx,dx			! size of largest available block
	mov	es,first_mcb		! start looking at first MCB
	call	checkmcb		! check the first MCB
	jc	dos489

dos481:	seg	es
	mov	ax,[mcb_owner]
	or	ax,ax			! is block available?
	jz	dos483
dos484:	call	nextmcb			! no, jump to next MCB
	jnc	dos481
	or	ax,ax
	jnz	dos482			! serious error
	mov	bx,dx			! end of list reached, return largest
	mov	ax,#ERR_NOMEM		! available memory block size
dos482:	stc				! return with error
	jmp	dos489

dos483:	seg	es
	mov	ax,[mcb_size]
	cmp	bx,#$0FFFF		! 0FFFFh size means: determine amount of
	je	dos487			! free memory. never get that much.
	cmp	ax,bx			! check size of free memory block
	jae	dos485
dos487:	cmp	ax,dx			! not sufficient, save in DX if greater
	jbe	dos484			! than any preceding block
	mov	dx,ax
	jmp	dos484			! continue with next memory block

dos485:	mov	dx,ax			! block is large enough, save its old
	seg	es			! size and then set the new one
	mov	[mcb_size],bx
	mov	ax,curr_psp
	or	ax,ax			! is a process running already
	jnz	dos486
	mov	ax,#$0FFFF		! dummy PSP address
dos486:	seg	es
	mov	[mcb_owner],ax		! save owner of memory block
	sub	dx,bx			! compute size of remaining free block
	jbe	dos488			! nothing left

	mov	ax,es
	add	ax,bx			! compute address of new MCB
	inc	ax
	dec	dx			! adjust size of remaining memory block
	mov	bl,#MCB_MEM_ID
	seg	es
	xchg	bl,[mcb_id]		! remember old ID and set new ID
	push	ds
	mov	ds,ax
	mov	byte ptr [mcb_id],bl
	mov	word ptr [mcb_size],dx	! setup new free memory block
	mov	word ptr [mcb_owner],#0
	pop	ds
	seg	es
	mov	bx,[mcb_size]		! restore BX register
	call	cleanup			! cleanup free memory blocks

dos488:	mov	ax,es			! return address of new memory block
	inc	ax			! to caller
	clc
dos489:	pop	dx
	pop	es
	ret


!
!**************************************************************************
!
! Free a memory block.
! Note that this routine might also get called from outside a DOS interrupt,
! so it should not use any reference to the DOS stack structure!
! Input:  ES  -  segment of memory block to free
! Output: AX  -  error code if carry flag set
!         Carry flag  -  set if error
! Registers changed: AX
!
dos49:

	push	es
	push	dx
	push	ax
	mov	dx,es
	dec	dx			! convert memory block address into
	call	findmcb			! MCB address and find MCB
	jc	dos497
	seg	es
	mov	word ptr [mcb_owner],#0	! free memory block
	call	cleanup			! integrate the new free memory block
dos497:	pop	dx
	jc	dos499			! on error dont restore AX register
	mov	ax,dx
dos499:	pop	dx
	pop	es
	ret


!
!**************************************************************************
!
! Resize memory block.
! Note that this routine might also get called from outside a DOS interrupt,
! so it should not use any reference to the DOS stack structure!
! Input:  BX  -  new size of memory block in paragraphs
!         ES  -  segment of memory block
! Output: BX  -  maximum available memory space
!         AX  -  error code if carry flag set
!         Carry flag  -  set if error
! Registers changed: AX, BX
!
dos4A:

	push	es
	push	dx
	push	ax
	mov	dx,es
	dec	dx			! convert memory block address into
	call	findmcb			! MCB address and find MCB
	jc	dos4A7

	seg	es
	mov	dx,[mcb_size]		! used at label dos4A4
	cmp	bx,dx
	je	dos4A8			! nothing changes
	jb	dos4A4			! make it smaller

! Make the memory block larger. To do this, first check if the next
! block has enough memory space available. Then include it into the
! current block.

	mov	dx,es
	call	nextmcb			! check next MCB
	jnc	dos4A2
	or	ax,ax
	jnz	dos4A7
dos4A1:	mov	ax,#ERR_NOMEM		! when at the end of the memory,
	jmp	dos4A7			! there is no more available

dos4A2:	seg	es
	cmp	word ptr [mcb_owner],#0	! is the next memory block free?
	jne	dos4A1
	push	dx
	seg	es
	mov	ax,[mcb_size]		! load size of free memory block
	seg	es
	mov	dl,[mcb_id]		! load ID of free memory block
	pop	es
	seg	es
	add	ax,[mcb_size]		! compute size of both memory blocks
	inc	ax			! combined
	cmp	ax,bx			! is it enough
	jae	dos4A3
	mov	bx,ax			! return maximum size to caller
	jmp	dos4A1

dos4A3:	seg	es
	mov	[mcb_size],ax		! set size of new combined segment
	seg	es
	mov	[mcb_id],dl		! set ID of new segment
	cmp	ax,bx			! if the size is exactly what has
	je	dos4A8			! been requested, we dont need to split
	mov	dx,ax

! Make the memory block smaller.

dos4A4:	sub	dx,bx			! compute size of remaining memory block
	dec	dx
	jz	dos4A8			! new memory block is zero size
	seg	es
	mov	[mcb_size],bx		! set new size of memory block
	mov	ax,es
	add	ax,bx			! compute address of new MCB
	inc	ax
	mov	bl,#MCB_MEM_ID
	seg	es
	xchg	bl,[mcb_id]		! remember old ID and set new ID
	push	ds
	mov	ds,ax
	mov	byte ptr [mcb_id],bl
	mov	word ptr [mcb_size],dx	! setup new free memory block
	mov	word ptr [mcb_owner],#0
	pop	ds
	seg	es
	mov	bx,[mcb_size]		! restore BX register
	call	cleanup			! cleanup free memory blocks
	jnc	dos4A8

! Termination with or without error.

dos4A7:	stc				! return with error
	pop	dx			! dont restore AX
	jmp	dos4A9

dos4A8:	clc				! no error
	pop	ax
dos4A9:	pop	dx
	pop	es
	ret


!
!**************************************************************************
!
! Free all memory blocks of an owner. This routine is called internally
! by the process management code.
! Input:  DX  -  PSP segment of owner
! Output: AX  -  error code if carry flag set
!         CX  -  number of MCBs freed
!         Carry flag -  set if error
! Registers changed: AX, CX
!
freeall:

	push	es
	xor	cx,cx
	mov	es,first_mcb		! start at first MCB
	call	checkmcb		! check that first MCB
	jc	free9

free1:	seg	es
	mov	ax,[mcb_owner]
	cmp	ax,dx			! found memory block of owner
	jne	free2
	seg	es
	mov	word ptr [mcb_owner],#0	! free current memory block
	inc	cx
free2:	call	nextmcb			! continue with next MCB
	jnc	free1
	or	ax,ax			! reached end of MCB list
	jz	free8
	stc
	jmp	free9

free8:	call	cleanup			! cleanup free memory blocks
free9:	pop	es
	ret


!
!**************************************************************************
!
! Find an MCB in the list.
! Input:  DX  -  MCB to search for
! Output: ES  -  segment of MCB
!         AX  -  error code
!         Carry flag  -  set if error
! Registers changed: AX, ES
!
findmcb:

	mov	es,first_mcb		! start at first MCB
	call	checkmcb		! check that first MCB
	jc	findm3
findm1:	mov	ax,es
	cmp	ax,dx			! compare MCB addresses
	je	findm3			! found it
	call	nextmcb			! proceed with next MCB
	jnc	findm1
	or	ax,ax			! found an error?
	jnz	findm2
	mov	ax,#ERR_INVMEM		! just got to end of list
findm2:	stc
findm3:	ret


!
!**************************************************************************
!
! Cleanup MCB list by concatenating adjacent free MCBs.
! Input:  none
! Output: AX  -  error code
!         Carry flag  -  set if error
! Registers changed: AX
!
cleanup:

	push	bx
	push	dx
	push	es
	mov	es,first_mcb		! start at first MCB
	call	checkmcb		! check that first MCB
	jc	clean9

clean1:	seg	es
	mov	ax,[mcb_owner]
	or	ax,ax			! concatenation possible if current
	jz	clean2			! MCB is free
clean5:	call	nextmcb			! otherwise check next MCB if its
	jnc	clean1			! free
	jmp	clean4			! handle error

clean2:	mov	dx,es
clean6:	call	nextmcb			! get next MCB
	jnc	clean3
clean4:	or	ax,ax			! at end of list? this 'or' also
	jz	clean9			! clears the carry flag
	stc				! return with error
	jmp	clean9

clean3:	seg	es
	mov	ax,[mcb_owner]		! is next MCB free?
	or	ax,ax
	jnz	clean5			! no, continue with next one
	seg	es
	mov	bl,[mcb_id]
	seg	es
	mov	ax,[mcb_size]
	inc	ax			! get total size of next block
	mov	es,dx
	seg	es
	add	[mcb_size],ax		! add size to preceding MCB
	seg	es
	mov	[mcb_id],bl		! copy ID from old to new MCB
	jmp	clean6			! continue with the next MCB

clean9:	pop	es
	pop	dx
	pop	bx
	ret


!
!**************************************************************************
!
! Proceed to next MCB.
! Input:  ES  -  Segment of old MCB
! Output: ES  -  Segment of new MCB
!         AX  -  error code (0 if end of MCB list)
!         Carry Flag  -  set if error
! Registers changed: AX, ES
!
nextmcb:

	xor	ax,ax
	seg	es
	cmp	byte ptr [mcb_id],#MCB_END_ID
	je	nextm2
	mov	ax,es
	seg	es
	add	ax,[mcb_size]		! add size to current MCB pointer
	jc	nextm1			! should never happen
	inc	ax
	cmp	ax,#LOWMEM		! check for upper limit
	jae	nextm1
	cmp	ax,first_mcb		! check for lower limit
	jb	nextm1
	mov	es,ax
	xor	ax,ax
	jmp	checkmcb		! check for correct MCB

nextm1:	mov	ax,#ERR_INVMCB		! return with error
nextm2:	stc
	ret


!
!**************************************************************************
!
! Check if MCB is correct.
! Input:  ES  -  segment of MCB
! Output: AX  -  error code (unchanged if no error)
!         Carry flag  -  set if error
! Registers changed: AX
!
checkmcb:

	seg	es
	cmp	byte ptr [mcb_id],#MCB_MEM_ID
	je	check1
	seg	es
	cmp	byte ptr [mcb_id],#MCB_END_ID
	je	check1
	mov	ax,#ERR_INVMCB
	stc				! error
check1:	ret


!
!**************************************************************************
!
	end

