
/*****************************************************************************
 * Command Line Masqdialer Client v1.0                                       *
 * October 6, 1998                                                           *
 * Copyright (c) 1998 Kevin Lindsay                                          *
 *                                                                           *
 * For use with the C Masqdialer server by Charles P. Wright                 *
 *                                                                           *
 * Report any Bugs to: klindsay@mkintraweb.com                               *
 *****************************************************************************/

/* 
   Copyright (C) 1998 Kevin Lindsay
  
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation.
*/


#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <errno.h>
#include <sys/time.h>
#include <unistd.h>
#include <ctype.h>

/* Globals */
#define VERSION "Command Line Masqdialer Client v1.0 - Copyright (c) 1998 Kevin Lindsay\n"
#define BUFSIZE 4096

char *progname;
int VERBOSE=0;
int port=0;

/* Usage */
void
usage()
{
   printf("\n%s\n"
	  "usage: %s -d -v -h <hostname>[:<port>] [-p <port>] -c <connectin>\n"
	  "usage: %s <[-lsk]> -h <hostname>[:<port>] [-p <port>]\n"
	  "usage: %s -V\n\n"	  
	  "OPTIONS\n\n"
	  "   -h <hostname>[:<port>] - Specify hostname with optional port.\n"
	  "   -p <port>              - Specify port. Overrides -h port option.\n"
	  "   -c <connection>        - Specify connection name.\n\n"
	  "   -d                     - Dial specified connection.\n"
	  "   -l                     - List available connections.\n"
	  "   -s                     - Request status of current connection.\n"
	  "   -k                     - Kill current connectin.\n\n"
	  "   -v                     - Verbose Mode.\n"
	  "   -V                     - Display Version.\n\n"
	  "",VERSION,progname,progname,progname);
   exit(0);
}

/* Put String */
char *
put_str(char *str)
{
   char *tmpstr;
   
   tmpstr = (char *)malloc(strlen(str)+1);
   tmpstr[0] = '\0';
   strcat(tmpstr,str);
   
   return(tmpstr);
}

/* Get Connection */
char *
get_connection(char *str)
{
   char *constr;
   
   constr = (char *)malloc(strlen(str)+1);
   constr[0] = '\0';
   strcat(constr,str);
   
   return(constr);
}

/* Get Host information - Parse PORT if necessary */
char *
get_host(char *str)
{
   char *ptr1;
   char *hoststr;
      
   if ((ptr1 = strchr(str,':')) == NULL) {
      hoststr = (char *)malloc(strlen(str)+1);
      hoststr[0] = '\0';      
      strcat(hoststr,str);
   } else {
      hoststr = (char *)malloc((ptr1-str)+1);
      hoststr[0] = '\0';
      strncat(hoststr,str,ptr1-str);
      if (!port) {
	 if (isdigit(ptr1[1])) {
	    ptr1++;
	    port = atoi(ptr1);
	 }
      }
   }
   
   return(hoststr);
}

/* Print Dial */
int
print_dial(char *fullbuf) {
   int res=0;
   char *ptr1;
   char *ptr2;
   
   if ((ptr1 = strstr(fullbuf,"BEGIN")) == NULL) {
      fprintf(stderr,"%s: print_list(): Error processing list!\n",progname);
      return(1);
   }
   ptr1+=6;
   
   if ((ptr2 = strstr(ptr1,"END")) == NULL) {
      fprintf(stderr,"%s: print_list(): Error processing list!\n",progname);
      return(1);
   }
   
   ptr2[0] = '\0';
   printf("\nConnection Information\n");
   printf("----------------------\n");
   printf("%s\n",ptr1);
   
   return(res);
}
/* Print List */
int
print_list(char *fullbuf) {
   int res=0;
   char *ptr1;
   char *ptr2;
   
   if ((ptr1 = strstr(fullbuf,"BEGIN")) == NULL) {
      fprintf(stderr,"%s: print_list(): Error processing list!\n",progname);
      return(1);
   }
   ptr1+=6;
   
   if ((ptr2 = strstr(ptr1,"END")) == NULL) {
      fprintf(stderr,"%s: print_list(): Error processing list!\n",progname);
      return(1);
   }
   
   ptr2[0] = '\0';
   printf("\nDialup Connections Available\n");
   printf("----------------------------\n");
   printf("%s\n",ptr1);
   
   return(res);
}

/* Print Kill */
int
print_kill(char *fullbuf) {
   int res=0;
   char *ptr1;
   char *ptr2;
   
   if ((ptr1 = strstr(fullbuf,"READY")) == NULL) {
      fprintf(stderr,"%s: print_list(): Error processing list!\n",progname);
      return(1);
   }
   ptr1+=6;
   
   if ((ptr2 = strstr(ptr1,"READY")) == NULL) {
      fprintf(stderr,"%s: print_list(): Error processing list!\n",progname);
      return(1);
   }
   ptr2--;
   
   ptr2[0] = '\0';

   if (!strcmp(ptr1,"SUCCESS"))
       printf("\nConnection terminated!\n\n");
   else
       printf("\n%s\n\n",ptr1);
   
   return(res);
}

/* Print Connection Status */
int
print_stat(char *fullbuf) {
   int res=0;
   char *ptr1;
   char *ptr2;
   char *tmpstat;
   char *tmpspeed;
   char *tmpname;
   
   if ((ptr1 = strstr(fullbuf,"READY")) == NULL) {
      fprintf(stderr,"%s: print_list(): Error processing list!\n",progname);
      return(1);
   }
   ptr1+=6;
   if ((ptr2 = strstr(ptr1,"READY")) == NULL) {
      fprintf(stderr,"%s: print_list(): Error processing list!\n",progname);
      return(1);
   }
   ptr2[0] = '\0';
   printf("\nConnection Status\n");
   printf("-----------------\n\n");

   if (!strcmp(ptr1,"DOWN\n")) {      
      printf("No connections!\n\n");
      return(res);
   }
   
   while (ptr1[0] != '\0') {
      ptr2=strchr(ptr1,':');
      tmpstat=ptr1;
      ptr1=strchr(ptr1,':')+1;
      ptr2[0] = '\0';
      
      ptr2=strchr(ptr1,':');
      tmpname=ptr1;
      ptr1=strchr(ptr1,':')+1;      
      ptr2[0] = '\0';
      
      ptr2=strchr(ptr1,'\n');
      tmpspeed=ptr1;      
      ptr2[0]='\0';
      printf("Connection: %s\n",tmpname);
      printf("    Status: %s\n",tmpstat);
      printf("     Speed: %s\n\n",tmpspeed);
      ptr2[0] = '\n';
      ptr1=ptr2+1;
   }
   
   return(res);
}

/* Process the proper command */

int
do_command(char *hoststr, char *command, char *connection)
{
   int sock;
   struct sockaddr_in address;
   struct in_addr inaddr;
   struct hostent *host;
   int res=0;
   char buffer[BUFSIZE];
   char *writedata;
   int grow=0;   
   char *fullbuf;
   int bytes;
   
   if (inet_aton(hoststr,&inaddr)) 
       host = gethostbyaddr((char *) &inaddr, sizeof(inaddr),AF_INET);
   else
       host = gethostbyname(hoststr);
   
   if (!host) {
      fprintf(stderr,"get_info(): Could not find IP address for %s\n",hoststr);
      exit(1);      
   }

   if (VERBOSE)
       printf("\nConnecting to server: %s\n",hoststr);
   
   if ((sock = socket(AF_INET,SOCK_STREAM, 0)) < 0) {
      fprintf(stderr,"get_info(): socket: (%d) %s\n",errno,strerror(errno));
      exit(1);
   }
   
   address.sin_family = AF_INET;
   address.sin_port = htons(port);
      
   memcpy(&address.sin_addr,host->h_addr_list[0],sizeof(address.sin_addr));
      
   if (connect(sock, (struct sockaddr *) &address, sizeof(address))) {
      fprintf(stderr,"get_info(): connect: (%d) %s\n",errno,strerror(errno));
      exit(1);
   }
   
   grow=7;
   
   if (VERBOSE)
       printf("\nConnection Established...\n");
   
   if (!strcmp(command,"DIAL"))
       grow+=strlen(connection)+1;
   
   writedata = (char *)malloc(grow);
   writedata[0] = '\0';
   strcat(writedata,command);
   if (grow > 7) {
      strcat(writedata," ");
      strcat(writedata,connection);
   }
   strcat(writedata,"\r\n");
   write(sock,writedata,strlen(writedata));
   free(writedata);
   
   write(sock,"QUIT\r\n",6);
   
   if (VERBOSE && !strcmp(command,"DIAL"))
       printf("\nDialing...\n");
   
   fullbuf = (char *)malloc(1);
   fullbuf[0] = '\0';
   grow=1;
   while ((bytes = read(sock,buffer,BUFSIZE)) > 0 ) {
      grow+=bytes;
      fullbuf = (char *)realloc(fullbuf,grow);
      strncat(fullbuf,buffer,bytes);
   }
   if (!strcmp(command,"LIST"))
       print_list(fullbuf);
   else if (!strcmp(command,"STAT"))
       print_stat(fullbuf);
   else if (!strcmp(command,"KILL") && VERBOSE)
       print_kill(fullbuf);
   else if (!strcmp(command,"DIAL") && VERBOSE)
       print_dial(fullbuf);
//   printf("%s\n",fullbuf); 
   return(res);
}

/* Main Function */
int
main(int args, char **argv)
{
   char *p;
   extern char *optarg;
   extern int optind, opterr, optopt;
   int ch;
   int res=0;
   char *host=NULL;
   char *connection=NULL;
   char *command=NULL;
   
   progname = ((p = rindex(argv[0],'/')) ? p+1 : *argv);
   
   if (args < 2) usage();
   
   while ((ch = getopt(args,argv,"dkslvVp:h:c:")) != EOF) {
      switch(ch) {
      case 'v':
	 VERBOSE = 1;
	 break;
      case 'V':
	 printf("%s",VERSION);
	 return(0);
	 break;
      case 'd':
	 if (!command)
	     command=put_str("DIAL");
	 break;
      case 'k':
	 if (!command)
	     command=put_str("KILL");
	 break;	 
      case 's':
	 if (!command)
	     command=put_str("STAT");
	 break;	 
      case 'l':
	 if (!command)
	     command=put_str("LIST");
	 break;
      case 'p':
	 port = atoi(optarg);
	 break;
      case 'h':
	 host=get_host(optarg);
	 break;
      case 'c':
	 connection=get_connection(optarg);
	 break;
      default:
	 usage();
	 break;
      }
   }

   if (!host || !port)
       usage();
   
   if (!strcmp(command,"DIAL") && !connection)
       usage();
   
   if (!res) {
      do_command(host,command,connection);
   }
   
   return(0);
}
