/* Bezerk
 * Copyright (C) 1998 Tony Gale.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */
 
#include <glib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>

#include "bezerk.h"
#include "send.h"
#include "ch_utils.h"
#include "msg_utils.h"
#include "reply.h"
#include "ctcp.h"
#include "dcc.h"
#include "irc.h"
#include "menu.h"
#include "util.h"
#include "script.h"
#include "debug.h"
#include "version.h"

#ifdef HAVE_TCL
#include <tcl.h>
#include "tcl_script.h"

extern Tcl_Interp *interp;
#endif

extern GdkColor colour_white;
extern GdkColor colour_red;
extern GdkColor colour_yellow;
extern GdkColor colour_blue;
extern GdkColor colour_grey;
extern GdkColor colour_green;

GSList *aliases=NULL;
static Alias system_aliases[]= {
  { "ban", "/mode $# +b $1"},
  { "deop", "/mode $# -o $1"},
  { "devoice", "/mode $# -v $1"},
  { "op", "/mode $# +o $1"},
  { "umode", "/mode $m $1"},
  { "unban", "/mode $# -b $1"},  
  { "voice", "/mode $# +v $1"},
  { NULL, NULL },
};

static UserCommand user_command[] = {
  { "ABOUT",    "ABOUT",     bezerk_version },
  { "ADMIN",    "ADMIN",     send_args },
  { "ALIAS",    "ALIAS",     add_alias},
  { "AWAY",     "AWAY",      send_away },
  { "BACK",     "BACK",      send_away },
  { "BYE",      "QUIT",      send_quit },
  { "CHANNEL",  "JOIN",      send_join },
  { "CONNECT",  "CONNECT",   send_args },
  { "CTCP",     "CTCP",      process_ctcp_command },
  { "DATE",     "TIME",      send_command },
  { "DCC",      "DCC",       process_dcc_command },
  { "DIE",     "DIE",        send_command },
  { "EXIT",     "QUIT",      send_quit },
  { "INFO",    "INFO",       send_args },
  { "INVITE",   "INVITE",    send_invite },
  { "JOIN",     "JOIN",      send_join },
  { "KILL",     "KILL",      send_args },
  { "L",        "PART",      send_part },
  { "LEAVE",    "PART",      send_part },
  { "LINKS",   "LINKS",      send_args },
  { "LIST",     "LIST",      send_command },
  { "LUSERS",   "LUSERS",    send_args },
  { "ME",       "ME",        ctcp_send_action },
  { "M",        "MSG",       send_privmsg },
  { "MSG",      "MSG",       send_privmsg },
  { "MODE",     "MODE",      send_mode },
  { "MOTD",     "MOTD",      send_command },
  { "NICK",     "NICK",      send_nick },
  { "OPER",    "OPER",       send_args },
  { "P",        "PING",      ctcp_send_ping },
  { "PART",     "PART",      send_part },
  { "PING",     "PING",      ctcp_send_ping },
  { "QUERY",    "QUERY",     send_privmsg },
  { "QUIT",     "QUIT",      send_quit },
  { "QUOTE",    "QUOTE",     send_quote },
  { "REHASH",   "REHASH",    send_command },
  { "RESTART", "RESTART",    send_command },
  { "SIGNOFF",  "QUIT",      send_quit },
  { "SQUIT",    "SQUIT",     send_args },
  { "STATS",   "STATS",      send_args },
  { "T",        "TOPIC",     send_topic },
#ifdef HAVE_TCL
  { "TCL",      "TCL",       run_tcl },
#endif
  { "TIME",     "TIME",      send_args },
  { "TOPIC",    "TOPIC",     send_topic },
  { "TRACE",   "TRACE",      send_args },
  { "VER",      "VER",       ctcp_send_version },
  { "VERSION",  "VERSION",   send_args },
  { "WHOIS",    "WHOIS",     send_args},
  { "WHOWAS",   "WHOWAS",    send_args},
  { NULL,       NULL,        NULL },
};

USER_COMMAND(bezerk_version)
{
  Message *message;

  bs_function_enter();

  message = message_new(MT_CHANNEL, window->connection, NULL);
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_white, NULL, NULL, "Bezerk "));
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_white, NULL, NULL, BEZERK_VERSION));
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_white, NULL, NULL, " by Tony Gale [Trog]"));

  bs_function_leave();
  return(message->type);
}

#ifdef HAVE_TCL
USER_COMMAND(run_tcl)
{
  int rslt;

  bs_function_enter();

/*   if (window->current_channel == NULL) { */
/*     puts ("current channel is null!"); */
/*   } else { */
       /* set the current channel.. we could set other vars too  */
/*     Tcl_SetVar(interp, "current_chan", window->current_channel->name, TCL_GLOBAL_ONLY); */
/*   } */

  tclglobalwindow = window;
  rslt = Tcl_Eval(interp, args);
  if(rslt == TCL_ERROR) {
    fputs( interp->result, stderr);
  }

/*   Tcl_UnsetVar(interp, "current_chan",  TCL_GLOBAL_ONLY); */
  bs_function_leave();
  return (0);
}
#endif HAVE_TCL

USER_COMMAND(send_command)
{

  bs_function_enter();

  if (command)
    irc_send_message(BEZ_BASE_WINDOW(window)->connection->sd, command);
  
  bs_function_leave();
  return(0);
}

USER_COMMAND(send_args)
{

  bs_function_enter();

  if (!command) {
    bs_function_leave();
    return(0);
  }

  if (args) {
    irc_send_format(BEZ_BASE_WINDOW(window)->connection->sd, "%s %s", command, args);
  } else {
    irc_send_message(BEZ_BASE_WINDOW(window)->connection->sd, command);
  }

  bs_function_leave();
  return(0);
}

USER_COMMAND(send_colon_args)
{

  bs_function_enter();

  if (!command) {
    bs_function_leave();
    return(0);
  }

  if (args) {
    irc_send_format(BEZ_BASE_WINDOW(window)->connection->sd, "%s :%s", command, args);
  } else {
    irc_send_message(BEZ_BASE_WINDOW(window)->connection->sd, command);
  }

  bs_function_leave();
  return(0);
}

USER_COMMAND(send_away)
{

  bs_function_enter();

  if (!command) {
    bs_function_leave();
    return(0);
  }

  if (strcmp(command, "BACK") == 0) {
    args=NULL;
  }

  send_colon_args("AWAY", args, window);

  bs_function_leave();
  return(0);
}

USER_COMMAND(send_join)
{
  ChannelInfo *channel;
  int entry_num;
  char *channel_name;

  bs_function_enter();

  if (!args) {
    bs_function_leave();
    return(0);
  } 

  if (strlen(args) == 0) {
    bs_function_leave();
    return(0);
  }

  if ( (*args != '#') && (*args != '&') ) {
    channel_name = g_malloc(strlen(args)+2);
    strcpy(channel_name, "#");
    strcat(channel_name, args);
  } else {
    channel_name = g_strdup(args);
  }

  if ( (entry_num = find_channel_entry(BEZ_BASE_WINDOW(window)->connection->channels, channel_name)) >= 0 ) {
    channel = find_channel(BEZ_BASE_WINDOW(window)->connection->channels, channel_name);
    if ( channel->status == JOINED ) {
	gtk_clist_select_row( GTK_CLIST(channel->window->ch_clist), entry_num, 0);
	channel->window->current_channel = channel;
    }
  } else {
    irc_send_format(BEZ_BASE_WINDOW(window)->connection->sd, "JOIN %s", channel_name);
  }

  g_free(channel_name);

  bs_function_leave();
  return(0);
}

USER_COMMAND(send_nick)
{

  bs_function_enter();

  if (!args) {
    bs_function_leave();
    return(0);
  }

  /*TODO: check for invalid characters in nick */

  irc_send_format(BEZ_BASE_WINDOW(window)->connection->sd, "NICK %s", args);

  bs_function_leave();
  return(0);
}

USER_COMMAND(send_privmsg)
{
  Message *message=NULL;
  char *to, *msg;
  BezWindow *msg_window;

  bs_function_enter();

  if (!args) {
    bs_function_leave();
    return(0);
  }
  
  to = bs_strtok(args, " ");
  msg = bs_strtok(NULL, "\0");

  if (!to) {
    bs_function_leave();
    return(0);
  }

  if (!msg) {
    if ( find_message_entry(window->connection->messages, to) < 0 ) {
      msg_window = create_message_window(IRC_MSG, to);
      BEZ_MESSAGE_WINDOW(msg_window)->connection =  window->connection;
      window->connection->messages = g_slist_append(window->connection->messages, msg_window);
    } else {
    message = message_new(MT_CHANNEL, window->connection, NULL);
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_red, NULL, NULL,
						     "A message window to this nick is already open"));
    }
  } else {
    irc_send_format(BEZ_BASE_WINDOW(window)->connection->sd, "PRIVMSG %s :%s", to, msg);
    
    message = message_new(MT_CHANNEL, window->connection, NULL);
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_red, NULL, NULL, "msg("));
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_white, NULL, NULL, to));
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_red, NULL, NULL, "): "));
    message->parts = message_part_parse_new(message->parts, &colour_white, msg);
  }

  bs_function_leave();
  return( (message ? message->type : 0) );
}

USER_COMMAND(send_mode)
{

  bs_function_enter();

  if (!args) {
    bs_function_leave();
    return(0);
  }

  irc_send_format(BEZ_BASE_WINDOW(window)->connection->sd, "MODE %s", args);

  bs_function_leave();
  return(0);
}

USER_COMMAND(send_part)
{
  ChannelInfo *target_channel;

  bs_function_enter();

  /* && is correct here */
  if (!window && !args) {
    bs_function_leave();
    return(0);
  }

  if (window->type == BEZ_CHANNEL) {
    BezChannelWindow *channel_window = (BezChannelWindow *) window;    
    if (!args) {
      if ( channel_window->current_channel != NULL) {
	irc_send_format(BEZ_BASE_WINDOW(window)->connection->sd,
			"PART %s", channel_window->current_channel->name);
	BEZ_BASE_WINDOW(window)->connection->channels = remove_channel(BEZ_BASE_WINDOW(window)->connection->channels,
								       channel_window->current_channel->name);
      }
    } else if ( (target_channel = find_channel(BEZ_BASE_WINDOW(window)->connection->channels, args)) ) {
      irc_send_format(BEZ_BASE_WINDOW(window)->connection->sd, "PART %s", args);
      BEZ_BASE_WINDOW(window)->connection->channels =
	remove_channel(BEZ_BASE_WINDOW(window)->connection->channels, args);
    }
  } else if (window->type == BEZ_MESSAGE) {
    gtk_widget_destroy(window->window);
  }

  bs_function_leave();
  return(0);
}

USER_COMMAND(send_notice)
{

  bs_function_enter();

  if (!args) {
    bs_function_leave();
    return(0);
  }

  irc_send_format(BEZ_BASE_WINDOW(window)->connection->sd, "NOTICE %s", args);

  bs_function_leave();
  return(0);
}

USER_COMMAND(send_quit)
{

  bs_function_enter();

  if ( !window || !BEZ_BASE_WINDOW(window)->connection || (BEZ_BASE_WINDOW(window)->connection->sd < 0) ) {
    bs_function_leave();
    return(0);
  }

  if (args) {
    irc_send_format( BEZ_BASE_WINDOW(window)->connection->sd, "QUIT :%s", args);
  } else {
    irc_send_message( BEZ_BASE_WINDOW(window)->connection->sd, "Go Bezerk! [http://www.gtk.org/~trog]");
  }

  irc_disconnect(BEZ_BASE_WINDOW(window)->connection->sd);
  BEZ_BASE_WINDOW(window)->connection->sd = -1;
  gdk_input_remove(window->connection->tag);
  gtk_widget_destroy(BEZ_BASE_WINDOW(window)->window);

/*   irc_disconnect(BEZ_BASE_WINDOW(window)->connection->sd); */
/*   irc_serversd = 0; */
/*   g_free(connection.nick); */
/*   g_free(connection.name); */
/*   g_free(connection.server); */
/*   g_free(connection.port); */
/*   connection.status = NOTREG; */
/*   connection.mode[0] = '\0';  */
/*   if (connection.oldnick) { */
/*     g_free(connection.oldnick); */
/*     connection.oldnick = NULL; */
/*   } */

/*   menus_set_sensitive("<Main>/File/Disconnect", FALSE); */
/*   menus_set_sensitive("<Main>/File/Connect", TRUE); */

/*   message = message_new(MT_CHANNEL, connection, NULL); */
/*   message->parts = g_slist_append(message->parts,  */
/* 				  message_part_new(&colour_white, NULL, NULL, "Disconnected")); */

/*   clean_up(); */

  bs_function_leave();
  return(0);
}

USER_COMMAND(send_invite)
{
  Message *message;
  char *user, *tochannel;

  bs_function_enter();

  if (!window) {
    bs_function_leave();
    return(0);
  }

  if (!args) {
    message = message_new(MT_CHANNEL, window->connection, NULL);
    message->parts = g_slist_append(message->parts, message_part_new(&colour_blue,NULL, NULL, 
				   "Not enough arguments: INVITE <nickname> [<channel>]"));
    bs_function_leave();
    return(message->type);
  } else {
    user = bs_strtok(args, " \t");
    tochannel = bs_strtok(NULL, "\0");
    if (!tochannel) {
      if (window->type == BEZ_MESSAGE) {
	message = message_new(MT_NICK, window->connection, ((BezMessageWindow *) window)->nick);
	message->parts = g_slist_append(message->parts, message_part_new(&colour_blue,NULL, NULL, 
				   "Not enough arguments: INVITE <nickname> [<channel>]"));
	bs_function_leave();
	return(message->type);
      } else if (window->type == BEZ_CHANNEL) {	
	BezChannelWindow *channel_window = (BezChannelWindow *) window;
	if (channel_window->current_channel) {
	  tochannel = channel_window->current_channel->name;
	} else {
	  message = message_new(MT_CHANNEL, window->connection, NULL);
	  message->parts = g_slist_append(message->parts, message_part_new(&colour_blue,NULL, NULL, 
				       "Not enough arguments: INVITE <nickname> [<channel>]"));
	  bs_function_leave();
	  return(message->type);
	}
      }
    }
    irc_send_format(BEZ_BASE_WINDOW(window)->connection->sd, "INVITE %s %s", user, tochannel);
  }

  bs_function_leave();
  return(0);
}

USER_COMMAND(send_topic)
{
  Message *message;
  char *channel, *topic;
  bs_function_enter();

  if (!window && !args) {
    bs_function_leave();
    return(0);
  }

  if (window->type == BEZ_MESSAGE) {
    message = message_new(MT_CHANNEL, window->connection, NULL);
    message->parts = g_slist_append(message->parts, message_part_new(&colour_red, NULL, NULL, 
				       "Command cannot be used here"));
    bs_function_leave();
    return(message->type);
  } else if (window->type == BEZ_CHANNEL) {
    BezChannelWindow *channel_window = (BezChannelWindow *) window;
    if (channel_window->current_channel) {
      if (!args) {
	irc_send_format(BEZ_BASE_WINDOW(window)->connection->sd,
			"TOPIC %s", channel_window->current_channel->name);
      } else {
	channel = bs_strtok(args, " ");
	topic = bs_strtok(NULL, "\0");
	if (!topic)
	  irc_send_format(BEZ_BASE_WINDOW(window)->connection->sd, "TOPIC %s", channel);
	else
	  irc_send_format(BEZ_BASE_WINDOW(window)->connection->sd, "TOPIC %s :%s", channel, topic);
      }
    }
  }

  bs_function_leave();
  return(0);
}

USER_COMMAND(send_quote)
{

  bs_function_enter();

  if (args) {
    irc_send_message(BEZ_BASE_WINDOW(window)->connection->sd, args);
  }

  bs_function_leave();
  return(0);
}

USER_COMMAND(add_alias)
{
  Alias *new_alias;
  char *comm_name, *comm_string;
  char *args_cpy;

  bs_function_enter();

  if (!args) {
    bs_function_leave();
    return(0);
  }

  args_cpy = g_strdup(args);
  new_alias = g_malloc(sizeof(Alias));

  comm_name = strtok(args_cpy, " \t");
  comm_string  = strtok(NULL, "\0");

  if (!comm_name || !comm_string) {
      bs_function_leave();
      return(0);
  }

  new_alias->name = g_strdup(comm_name);
  new_alias->value = g_strdup(comm_string);

  aliases = g_slist_append(aliases, new_alias);
  prefs_add_alias(comm_name, comm_string);

  g_free(args_cpy);

  bs_function_leave();
  return(0);
}

int send_issue_command(char *text, BezWindow *window)
{
  char *command, *args;
  int command_num=0;
  int retval=0;
  GSList *alias_entry;
  Alias *alias_info;
  Message *message;

  bs_function_enter();

  if ( !text || !text[0] ) {
      bs_function_leave();
      return(0);
  }

  command = bs_strtok(text, " ");
  args = bs_strtok(NULL, "");

  while(user_command[command_num].name) {
    if (!strcasecmp(user_command[command_num].name, command)) {
      retval = user_command[command_num].func(user_command[command_num].command, args,
					      (BezBaseWindow *) window);
      bs_function_leave();
      return(retval);
    }
    command_num++;
  }

  command_num = 0;
  while (system_aliases[command_num].name != NULL) {
    if (!strcasecmp(system_aliases[command_num].name, command)) {
      retval = process_alias(BEZ_BASE_WINDOW(window)->connection,
			      system_aliases[command_num].value, args, window);
      bs_function_leave();
      return(retval);
    }
    command_num++;
  }

  alias_entry = aliases;
  while( alias_entry ) {
    alias_info = alias_entry->data;
    if (!strcasecmp(alias_info->name, command)) {
      retval = process_alias(BEZ_BASE_WINDOW(window)->connection,
			     alias_info->value, args, window);
      bs_function_leave();
      return(retval);
    }
    alias_entry = g_slist_next(alias_entry);
  }

  message = message_new(MT_CHANNEL, BEZ_BASE_WINDOW(window)->connection, NULL);
  message->parts = g_slist_append(message->parts, message_part_new(&colour_white, NULL, NULL, 
								   "Unrecognised/unimplemented command "));

  bs_function_leave();
  return(message->type);
}



