/* Bezerk
 * Copyright (C) 1998 Tony Gale.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */
 
#include <gtk/gtk.h>
#include <glib.h>
#include <string.h>
#include <time.h>

#include "reply.h"
#include "servers.h"
#include "message.h"
#include "dialogs.h"
#include "irc.h"
#include "recv.h"
#include "util.h"
#include "ch_utils.h"
#include "ctcp.h"
#include "prefs.h"
#include "bezerk.h"
#include "debug.h"

extern messageData irc_message;
extern struct UserData user;
extern GSList *messages;

GdkColormap *cmap;
GdkColor colour_white;
GdkColor colour_black;
GdkColor colour_blue;
GdkColor colour_green;
GdkColor colour_red;
GdkColor colour_brown;
GdkColor colour_purple;
GdkColor colour_orange;
GdkColor colour_yellow;
GdkColor colour_lightgreen;
GdkColor colour_cyan;
GdkColor colour_lightcyan;
GdkColor colour_lightblue;
GdkColor colour_pink;
GdkColor colour_grey;
GdkColor colour_lightgrey;
GdkFont *font_normal;
GdkFont *font_bold;
GdkFont *font_italic;
GdkFont *font_bold_italic;

GSList *joining_channels = NULL;

void format_reply();

void channel_init()
{

  bs_function_enter();

  cmap = gdk_colormap_get_system();

  extract_color(&colour_white, 255, 255, 255);
  if (!gdk_color_alloc(cmap, &colour_white)) {
    g_error("couldn't allocate colour");
  }

  extract_color(&colour_black, 0, 0, 0);
  if (!gdk_color_alloc(cmap, &colour_black)) {
    g_error("couldn't allocate colour");
  }

  extract_color(&colour_blue, 0, 100, 255);
  if (!gdk_color_alloc(cmap, &colour_blue)) {
    g_error("couldn't allocate colour");
  }

  extract_color(&colour_green, 0, 255, 0);
  if (!gdk_color_alloc(cmap, &colour_green)) {
    g_error("couldn't allocate colour");
  }

  extract_color(&colour_red, 255, 0, 0);
  if (!gdk_color_alloc(cmap, &colour_red)) {
    g_error("couldn't allocate colour");
  }

  extract_color(&colour_brown, 165, 42, 42);
  if (!gdk_color_alloc(cmap, &colour_brown)) {
    g_error("couldn't allocate colour");
  }

  extract_color(&colour_purple, 160, 32, 240);
  if (!gdk_color_alloc(cmap, &colour_purple)) {
    g_error("couldn't allocate colour");
  }

  extract_color(&colour_orange, 255, 165, 0);
  if (!gdk_color_alloc(cmap, &colour_orange)) {
    g_error("couldn't allocate colour");
  }

  extract_color(&colour_yellow, 255, 255, 0);
  if (!gdk_color_alloc(cmap, &colour_yellow)) {
    g_error("couldn't allocate colour");
  }

  extract_color(&colour_lightgreen, 144, 238, 144);
  if (!gdk_color_alloc(cmap, &colour_lightgreen)) {
    g_error("couldn't allocate colour");
  }

  extract_color(&colour_cyan, 0, 255, 255);
  if (!gdk_color_alloc(cmap, &colour_cyan)) {
    g_error("couldn't allocate colour");
  }

  extract_color(&colour_lightcyan, 224, 255, 255);
  if (!gdk_color_alloc(cmap, &colour_lightcyan)) {
    g_error("couldn't allocate colour");
  }

  extract_color(&colour_lightblue, 173, 216, 230);
  if (!gdk_color_alloc(cmap, &colour_lightblue)) {
    g_error("couldn't allocate colour");
  }

  extract_color(&colour_pink, 255, 192, 203);
  if (!gdk_color_alloc(cmap, &colour_pink)) {
    g_error("couldn't allocate colour");
  }

  extract_color(&colour_grey, 190, 190, 190);
  if (!gdk_color_alloc(cmap, &colour_grey)) {
    g_error("couldn't allocate colour");
  }

  extract_color(&colour_lightgrey, 211, 211, 211);
  if (!gdk_color_alloc(cmap, &colour_lightgrey)) {
    g_error("couldn't allocate colour");
  }

  font_normal = gdk_font_load ( preference_get("font.normal") );
  font_bold = gdk_font_load ( preference_get("font.bold") );
  font_italic = gdk_font_load ( preference_get("font.italic") );
  font_bold_italic = gdk_font_load ( preference_get("font.bolditalic") );

  bs_function_leave();
  return;
}

void rpl_namreply(Connection *connection)
{
  ChannelInfo *channel;
  char *nick;

  bs_function_enter();

  if ( !connection || !irc_message.args || !irc_message.args[0] || !irc_message.args[1] || 
       !irc_message.args[2] || !irc_message.args[3]) {
    bs_function_leave();
    return;
  }

  format_reply(connection);

  if ( !strcmp(irc_message.args[2], "*") ) {
    bs_function_leave();
    return;
  }
    
  if ( (channel = find_channel(joining_channels, irc_message.args[2])) != NULL) {
    nick = strtok(irc_message.args[3], " ");
    while (nick) {
      add_channel_member(channel, nick);
      nick = strtok(NULL, " ");
    }
  } else if ( ((channel = find_channel(connection->channels, irc_message.args[2])) == NULL) && 
	      !joining_channels ) {
    /* It's a channel we aren't on - either we're joining it or we've done a NAMES command */
    channel = (ChannelInfo *) g_malloc(sizeof(ChannelInfo));
    channel->name = g_strdup(irc_message.args[2]);
    channel->window = NULL;
    channel->topic = NULL;
    channel->members = NULL;
    channel->status = NOT_JOINING;
    channel->mode[0] = '\0';
    channel->selected_member = 0;
    channel->connection = connection;

    nick = strtok(irc_message.args[3], " ");
    while (nick) {
      add_channel_member(channel, nick);
      nick = strtok(NULL, " ");
    }
    joining_channels = g_slist_append(joining_channels, channel);
  }

  bs_function_leave();
  return;
}

void rpl_endofnames(Connection *connection)
{
  GSList *channels;
  ChannelInfo *target_channel;
  int cl_row;

  bs_function_enter();

  if ( !connection || !irc_message.args || !irc_message.args[0] || !irc_message.args[1] ) {
    bs_function_leave();
    return;
  }

  format_reply(connection);

  if ( !strcmp(irc_message.args[1], "*") ) {
    channels = joining_channels;
    while( joining_channels ) {
      target_channel = joining_channels->data;
      joining_channels = remove_channel(joining_channels, target_channel->name);
    }
    bs_function_leave();
    return;
  }

  if ( (target_channel = find_channel(joining_channels, irc_message.args[1])) != NULL) {
    if ( !is_channel_member(target_channel, connection->nick) ) {
      remove_members(target_channel->members);
      g_free(target_channel->name);
      g_free(target_channel);
    } else {
      target_channel->status = JOINED;
      target_channel->window = BEZ_CHANNEL_WINDOW(connection->console);
      target_channel->connection->channels = g_slist_append(target_channel->connection->channels, target_channel);
      cl_row = gtk_clist_append (GTK_CLIST (target_channel->window->ch_clist),
				 irc_message.args+1);
      gtk_clist_select_row(GTK_CLIST (target_channel->window->ch_clist), cl_row, 0);
      target_channel->window->current_channel = target_channel;
      target_channel->selected_member = 0;
      
      /* Make sure we get the topic */
      irc_send_format(connection->sd, "TOPIC %s", target_channel->name);
      irc_send_format(connection->sd, "MODE %s", target_channel->name);
    }
    joining_channels = g_slist_remove(joining_channels, target_channel);
  }

  bs_function_leave();
  return;
}   

void rpl_notopic(Connection *connection)
{
  Message *message;

  bs_function_enter();

  if ( !connection || !irc_message.args || !irc_message.args[0] || !irc_message.args[1] ) {
    bs_function_leave();
    return;
  }

  message = message_new(MT_CHANNEL, connection, irc_message.args[1]);
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_blue, NULL, NULL, "No topic is set on "));
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_blue, NULL, NULL, irc_message.args[1]));

  bs_function_leave();
  return;
}

void rpl_topic(Connection *connection)
{
  ChannelInfo *target_channel;
  char buff[BUFFLEN+20];
  Message *message;

  bs_function_enter();

  if ( !connection || !irc_message.args || !irc_message.args[0] ||
       !irc_message.args[1] || !irc_message.args[2]) {
    bs_function_leave();
    return;
  }

  if ( !(target_channel = find_channel(connection->channels, irc_message.args[1]))) {
    bs_function_leave();
    return;
  }

  if ( target_channel->topic ) {
    g_free(target_channel->topic);
  }

  target_channel->topic = g_strdup(irc_message.args[2]);
  
  if ( target_channel->window != NULL ) {
    if (target_channel == target_channel->window->current_channel) {
      g_snprintf(buff, BUFFLEN+20, "Bezerk: %s - %s", target_channel->name, target_channel->topic);
      gtk_window_set_title(GTK_WINDOW(target_channel->window->window), buff);
    }
  }

  message = message_new(MT_CHANNEL, connection, irc_message.args[1]);
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_blue, NULL, NULL, "Topic on "));
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_blue, NULL, NULL, irc_message.args[1]));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_blue, NULL, NULL, ": "));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_blue, NULL, NULL, irc_message.args[2]));

  bs_function_leave();
  return;
}

void rpl_topicset(Connection *connection)
{
  time_t settime;
  char *time_set;
  Message *message;

  bs_function_enter();

  if ( !connection || !irc_message.args || !irc_message.args[0] || !irc_message.args[1] || 
       !irc_message.args[2] || !irc_message.args[3] ) {
    bs_function_leave();
    return;
  }

  if ( !(find_channel(connection->channels, irc_message.args[1]))) {
    bs_function_leave();
    return;
  }

  settime = atol(irc_message.args[3]);
  time_set = ctime(&settime);
  time_set[strlen(time_set)-1] = '\0';

  message = message_new(MT_CHANNEL, connection, irc_message.args[1]);
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_blue, NULL, NULL, "Topic on channel "));
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_blue, NULL, NULL, irc_message.args[1]));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_blue, NULL, NULL, " set by "));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_blue, NULL, NULL, irc_message.args[2]));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_blue, NULL, NULL, " on "));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_blue, NULL, NULL, time_set));

  bs_function_leave();
  return;
}

void rpl_inviting(Connection *connection)
{
  Message *message;

  bs_function_enter();

  if ( !connection || !irc_message.args || !irc_message.args[0] || !irc_message.args[1] ||
       !irc_message.args[2]) {
    bs_function_leave();
    return;
  }

  message = message_new(MT_CHANNEL, connection, NULL);
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_blue, NULL, NULL, "Inviting "));
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_blue, NULL, NULL, irc_message.args[1]));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_blue, NULL, NULL, " to "));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_blue, NULL, NULL, irc_message.args[2]));

  bs_function_leave();
  return;
}

void rpl_time(Connection *connection)
{
  Message *message;

  bs_function_enter();

  if ( !connection || !irc_message.args || !irc_message.args[0] ||
       !irc_message.args[1] || !irc_message.args[2]) {
    bs_function_leave();
    return;
  }

  message = message_new(MT_CHANNEL, connection, NULL);
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_blue, NULL, NULL, "Time on "));
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_blue, NULL, NULL, irc_message.args[1]));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_blue, NULL, NULL, " is "));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_blue, NULL, NULL, irc_message.args[2]));

  bs_function_leave();
  return;
}

void rpl_away(Connection *connection)
{
  Message *message;

  bs_function_enter();

  if ( !connection || !irc_message.args || !irc_message.args[0] ||
       !irc_message.args[1] || !irc_message.args[2]) {
    bs_function_leave();
    return;
  }

  message = message_new(MT_CHANNEL, connection, NULL);
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_yellow, NULL, NULL, "away"));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_white, NULL, NULL, "("));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_white, NULL, NULL, irc_message.args[1]));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_white, NULL, NULL, "): "));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_white, NULL, NULL, irc_message.args[2]));

  bs_function_leave();
  return;
}

void rpl_unaway(Connection *connection)
{
  Message *message;

  bs_function_enter();

  if (!connection) {
    bs_function_leave();
    return;
  }

  connection->status = OKAY;
  set_user_status(connection);

  message = message_new(MT_CHANNEL, connection, NULL);
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_blue, NULL, NULL, 
					   "You are no longer marked as being away"));

  bs_function_leave();
  return;
}

void rpl_nowaway(Connection *connection)
{
  Message *message;

  bs_function_enter();

  if (!connection) {
    bs_function_leave();
    return;
  }

  connection->status = AWAY;
  set_user_status(connection);

  message = message_new(MT_CHANNEL, connection, NULL);
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_blue, NULL, NULL, 
					   "You have been marked as being away"));

  bs_function_leave();
  return;
}

void format_reply(Connection *connection)
{
  char buff[BUFFLEN];
  int argnum=1;
  Message *message;

  bs_function_enter();

  if ( !connection || !irc_message.nick || !irc_message.args || !irc_message.args[0] ||
       !irc_message.args[1]) {
    bs_function_leave();
    return;
  }

  buff[0]= '\0';
  while(irc_message.args[argnum]) {
    strcat(buff, irc_message.args[argnum++]);
    strcat(buff, " ");
  }
  buff[strlen(buff)] = '\0';

  bs_printf(3, "  buff<%s>", buff);

  message = message_new(MT_CHANNEL, connection, NULL);
  message->parts = g_slist_append(message->parts, message_part_new(&colour_blue, NULL, NULL, buff));

  bs_function_leave();
  return;
}

void err_nickname(Connection *connection)
{

  bs_function_enter();

  if (!connection) {
    bs_function_leave();
    return;
  }

  if (connection->status == NOTREG) {
    nick_dialog(connection, "Nickname is in use. Please enter another:");
  } else {
    format_reply(connection);
  }

  bs_function_leave();
  return;
}

void err_channel(Connection *connection)
{
  Message *message;

  bs_function_enter();

  if ( !connection || !irc_message.args || !irc_message.args[0] ||
       !irc_message.args[1] || !irc_message.args[2]) {
    bs_function_leave();
    return;
  }

  connection->channels = remove_channel(connection->channels, irc_message.args[1]);

  message = message_new(MT_CHANNEL, connection, NULL);
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_blue, NULL, NULL, irc_message.args[2]));

  bs_function_leave();
  return;
}

void process_error(Connection *connection)
{

  bs_function_enter();

  switch(irc_message.number)
    {
    case 432:
    case 433:
    case 436:
      err_nickname(connection);
      break;
    case 437:
    case 471:
    case 473:
    case 474:
    case 475:
    case 476:
      err_channel(connection);
      break;
    default:
      format_reply(connection);
    }

  bs_function_leave();
  return;
}

void process_reply(Connection *connection)
{

  bs_function_enter();

   switch(irc_message.number)
    {
    case 001:
      /* Can probably get away without doing this */
      /* connection->server = g_strdup(irc_message.nick); */

      connection->status = OKAY;
      set_user_status(connection);
      /* Allow this to fall through */
    case 002:
    case 003:
    case 004: 
      format_reply(connection);
      break;
    case 251:
    case 252:
    case 253:
    case 254:
    case 255:
      format_reply(connection);
      break;
    case 301:
      rpl_away(connection);
      break;
    case 305:
      rpl_unaway(connection);
      break;
    case 306:
      rpl_nowaway(connection);
      break;
    case 331:
      rpl_notopic(connection);
      break;
    case 332:
      rpl_topic(connection);
      break;
    case 333:
      rpl_topicset(connection);
      break;
    case 341:
      rpl_inviting(connection);
      break;
    case 353: /* Names list */
      rpl_namreply(connection);
      break;
    case 366: /* End of names list */
      rpl_endofnames(connection);
      break;
    case 372:
    case 375:
    case 376:
      format_reply(connection);
      break;
    case 391:
      rpl_time(connection);
      break;
    default:
      format_reply(connection);
    }

  bs_function_leave();
  return;
}

void channel_issue_message(Connection *connection)
{

  bs_function_enter();

  if (irc_message.type == MSG_NUMERIC) {
    if ((irc_message.command[0] == '4') || (irc_message.command[0] == '5')) {
      process_error(connection);
    } else {
      process_reply(connection);
    }
  } else {
    process_command(connection);
  }

  messages_write();

  bs_function_leave();
  return;
}
