/*  tide  Command-line client for dusty old TTYs and line printers.
    Last modified 1998-05-31

    Copyright (C) 1998  David Flater.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "common.hh"

// The user_io_ptr feature of libpng doesn't seem to work.
static FILE *png_io_ptr;
static void
write_data_fn (png_structp png_ptr, png_bytep b_ptr, png_size_t sz) {
  fwrite (b_ptr, 1, sz, png_io_ptr);
}

#define fmtbarf { \
        Dstr details ("Can't do format "); \
        details += format; \
        details += " in mode "; \
        details += mode; \
        barf (BAD_FORMAT, details); \
      }

void
do_location (const Dstr &name, TideContext *context, Interval step,
FILE *outfp, char mode, char format) {
  StationRef *sr;
  CommandLineSettings *cls = context->cls;
  Settings *settings = context->settings;
  if (cls->l_len <= maxfastload)
    sr = context->fastload (name);
  else
    sr = (*(context->stationIndex()))[name];
  if (sr) {
    Station *s = sr->load(context);

    // Install mark level, if applicable.
    if (!(cls->ml_isnull)) {
      s->markLevel = new PredictionValue (cls->ml);
      if (cls->ml.Units() != s->myUnits)
        s->markLevel->Units (s->myUnits);
    }

    // Find start and end times.
    // This needs to be done here due to the timezones.
    Timestamp startt, endt;
    if (cls->b.isNull())
      startt = Timestamp (time (NULL));
    else
      startt = Timestamp (cls->b, sr->timezone, context->settings);
    if (cls->e.isNull())
      endt = startt + Interval(defpredictinterval);
    else
      endt = Timestamp (cls->e, sr->timezone, context->settings);
    if (startt > endt) {
      Timestamp tt = startt;
      startt = endt;
      endt = tt;
    }

    switch (mode) {
    case 'p':
      if (format != 't')
        fmtbarf;
      {
        Dstr text_out;
        context->textMode (s, startt, endt, text_out);
        fprintf (outfp, "%s", text_out.aschar());
      }
      break;

    case 's':
      if (format != 't')
        fmtbarf;
      {
        Dstr text_out;
        context->statsMode (s, startt, endt, text_out);
        fprintf (outfp, "%s", text_out.aschar());
      }
      break;

    case 'c':
      switch (format) {
      case 'h':
      case 't':
        {
          Dstr text_out;
          context->calendarMode (s, startt, endt, text_out, (format=='h'));
          fprintf (outfp, "%s", text_out.aschar());
        }
        break;
      default:
        fmtbarf;
      }
      break;

    case 'r':
      if (format != 't')
        fmtbarf;
      {
        Dstr text_out;
        context->rawMode (s, startt, endt, step, text_out);
        fprintf (outfp, "%s", text_out.aschar());
      }
      break;

    case 'b':
      if (format != 't')
        fmtbarf;
      {
        Dstr text_out;
        context->bannerMode (s, startt, endt, text_out);
        fprintf (outfp, "%s", text_out.aschar());
      }
      break;

    case 'g':
      switch (format) {
      case 'p':
        {
          RGBGraph g (settings->gw, settings->gh, context->colors);
          g.drawTides (s, startt);
          png_io_ptr = outfp;
	  g.writeAsPNG (write_data_fn);
        }
        break;
      case 't':
        {
          Dstr text_out;
          TTYGraph g (settings->tw, settings->th);
          g.drawTides (s, startt);
	  g.writeAsText (text_out);
          fprintf (outfp, "%s", text_out.aschar());
        }
        break;
      default:
        fmtbarf;
      }
      break;

    default:
      {
        Dstr details ("Unsupported mode: ");
        details += mode;
        barf (BAD_MODE, details);
      }
    }

    delete s;

  } else {
    Dstr details ("Could not find: ");
    details += name;
    barf (STATION_NOT_FOUND, details, 0);
  }
}

void
loop_locations (TideContext *context, FILE *outfp) {
  CommandLineSettings *cls = context->cls;
  int is_first = 1;
  struct CommandLineSettings::lnode *l = cls->l;

  Interval step (HOURSECONDS);
  if (!(cls->s.isNull())) {
    step = Interval (cls->s);
    if (step <= Interval (0)) {
      Dstr details ("You must specify a positive step.  You tried ");
      details += cls->s;
      barf (NUMBER_RANGE_ERROR, details);
    }
  }

  char mode, format;
  if (cls->m.isNull())
    mode = 'p';
  else
    mode = cls->m[0];
  if (cls->f.isNull())
    format = 't';
  else
    format = cls->f[0];

  if (mode != 'l') {
    if (!l)
      cerr << "Warning:  No locations specified with -l; hence, no output." << endl;
    while (l) {
      if (is_first)
	is_first = 0;
      else
	fprintf (outfp, "%s", stationsep);
      do_location (l->name, context, step, outfp, mode, format);
      l = l->next;
    }

  } else { // List mode
    switch (format) {
    case 't':
      {
	Dstr text_out;
	context->listModePlain (text_out);
	fprintf (outfp, "%s", text_out.aschar());
      }
      break;
    case 'h':
      {
	Dstr text_out;
	context->listModeHTML (text_out);
	fprintf (outfp, "%s", text_out.aschar());
      }
      break;
    default:
      fmtbarf;
    }
  }
}

int main (int argc, char **argv)
{
  // So much for ANSI.
  // sync_with_stdio();

  Settings settings;
  {
    ConfigDefaults cd;
    UserDefaults ud;
    CommandLineSettings cls (argc, argv);
    settings.supersedeBy (cd);
    settings.supersedeBy (ud);
    settings.supersedeBy (cls);
  }

  TideContext *context = new TideContext (argc, argv, new Colors (settings),
    &settings);

  struct stat buf;
  if (stat (context->disabledisclaimerfile.aschar(), &buf)) {
    cerr << "\
-----------------------------------------------------------------------------\n\
             XTide   Copyright (C) 1998 David Flater.\n\
\n\
This software is provided under the terms of the GNU General Public\n\
License, either version 2 of the License, or (at your option) any later\n\
version.\n\
\n\
                        NOT FOR NAVIGATION\n\
\n\
This program is distributed in the hope that it will be useful, but\n\
WITHOUT ANY WARRANTY; without even the implied warranty of\n\
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  The author\n\
assumes no liability for damages arising from use of this program OR\n\
of the 'harmonics data' that is distributed with it.  For details, see\n\
the verbose documentation at http://www.universe.digex.net/~dave/xtide/.\n\
\n\
This obnoxious message will go away permanently if you click \"Don't show\n\
this again\" in the disclaimer window of the X windows client (xtide), or\n\
if you create a file in your home directory called \".disableXTidedisclaimer\".\n\
-----------------------------------------------------------------------------\n\
\n";
  }

  if (argc < 2) {
    cerr << "\
Minimal usage:  tide -l \"Location name\"\n\
Other switches:\n\
  -b \"YYYY-MM-DD HH:MM\"\n\
      Specify the begin (start) time for predictions.\n\
  -e \"YYYY-MM-DD HH:MM\"\n\
      Specify the end (stop) time for predictions.\n\
  -f h|p|t\n\
      Specify the output format as HTML, PNG, or text.  The default is text.\n\
      Only calendar and list modes can do HTML; only graph mode can do PNG.\n\
  -m b|c|g|l|p|r|s\n\
      Specify mode to be banner, calendar, graph, list, plain, raw, or\n\
      stats.  The default is plain.\n\
  -o \"filename\"\n\
      Redirect output to the specified file.\n\
  -s \"HH:MM\"\n\
      Specify the step interval, in hours and minutes, for raw\n\
      mode predictions.  The default is one hour.\n\
  -v\n\
      Print version string and exit.\n\
\n\
NOTE:  These are only the most important switches.  For information on\n\
all of the switches, please read the verbose documentation at:\n\
    http://www.universe.digex.net/~dave/xtide/" << endl;
    exit (-1);
  }

  FILE *outfp = stdout;
  if (!(context->cls->o.isNull())) {
    if (!(outfp = fopen (context->cls->o.aschar(), "w"))) {
      Dstr details (context->cls->o);
      details += ": ";
      details += strerror (errno);
      details += ".";
      barf (CANT_OPEN_FILE, details);
    }
  }

  loop_locations (context, outfp);

  exit (0);
}
