/*
 * Copyright 1996 Thierry Bousch
 * Licensed under the Gnu Public License, Version 2
 *
 * $Id: simple-parser.c,v 1.6 1996/12/28 16:27:53 bousch Exp $
 *
 * A simple recursive descent parser for algebraic expressions
 */

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "saml.h"
#include "saml-errno.h"
#include "saml-parse.h"

/*
 * Here is the grammar:
 *
 * expr -> term
 *       | `+' term
 *       | `-' term
 *       | expr `+' term
 *       | expr `-' term
 *
 * term -> factor
 *       | term `*' factor
 *       | term `/' factor
 *
 * factor -> atom
 *         | atom `^' signed_int
 *
 * atom -> `(' expr `)'
 *       | INTEGER
 *       | IDENTIFIER
 *       | IDENTIFIER `!' signed_int
 *
 * signed_int -> INTEGER
 *             | `+' INTEGER
 *             | `-' INTEGER
 */

static mref_t model;
static int last_token;

static int parse_expr (mref_t result);
static int parse_factor (mref_t result);
static int parse_term (mref_t result);
static int parse_atom (mref_t result);

int saml_parse (mref_t result, mref_t _model)
{
	int ret;

	/* Beware of aliasing, result and _model might be the same mref */
	model = mref_new();
	mref_copy(model, _model);
	last_token = (*saml_lexer)();
	ret = parse_expr(result);
	mref_free(model);

	/* Have we reached the end of the input? */
	if (last_token != STOK_EOF)
		ret = -1;
	if (ret < 0)
		mref_error(result, SE_STRING, "saml_parse");
	return ret;
}

static int parse_expr (mref_t result)
{
	mref_t tmp0 = mref_new(), tmp1 = mref_new(), tmp2 = mref_new();
	int ret, op;
	unsigned int terms = 0;

	while(1) {
		if (last_token == '+' || last_token == '-') {
			op = last_token;
			last_token = (*saml_lexer)();
		}
		else if (terms == 0) {
			/* The first term doesn't need a sign */
			op = '+';
		}
		else break;

		ret = parse_term(tmp0);
		if (ret < 0) goto end_parse;

		if (terms == 0) {
			/*
			 * In the case of tensors, there are different kinds
			 * of zeros, so we must initialize tmp1,tmp2,result
			 * from the first term, not from the model.
			 */
			mref_zero(tmp1,   tmp0);
			mref_zero(tmp2,   tmp0);
			mref_zero(result, tmp0);
		}

		if (op == '+')
		  mref_add(tmp1, tmp1, tmp0);
		else
		  mref_sub(tmp1, tmp1, tmp0);

		/* After having read 32 terms, move them to tmp2 */
		terms++;
		if (terms % 32 == 0) {
			mref_add(tmp2, tmp2, tmp1);
			mref_zero(tmp1, tmp1);
			/* After 1024 terms, move them to result */
			if (terms % 1024 == 0) {
				mref_add(result, result, tmp2);
				mref_zero(tmp2, tmp2);
			}
		}
	}
	mref_add(tmp2, tmp2, tmp1);
	mref_add(result, result, tmp2);
	ret = 0;  /* to make GCC happy */
end_parse:
	mref_free(tmp0); mref_free(tmp1); mref_free(tmp2);
	return ret;
}

static int parse_term (mref_t result)
{
	mref_t tmp = mref_new();
	int ret, op;

	ret = parse_factor(result);
	if (ret < 0) goto end_parse;

	while (last_token == '*' || last_token == '/') {
		op = last_token;
		last_token = (*saml_lexer)();

		ret = parse_factor(tmp);
		if (ret < 0) goto end_parse;

		if (op == '*')	mref_mul(result, result, tmp);
		else		mref_div(result, result, tmp);
	}
end_parse:
	mref_free(tmp);
	return ret;
}

int parse_factor (mref_t result)
{
	int ret, sign, expo;

	ret = parse_atom(result);
	if (ret < 0)
		return ret;

	/* If there isn't a caret after the atom, exit now */
	if (last_token != '^')
		return ret;
		
	last_token = (*saml_lexer)();
	sign = '+';
	if (last_token == '+' || last_token == '-') {
		sign = last_token;
		last_token = (*saml_lexer)();
	}
	if (last_token != STOK_INTEGER) {
		/* Syntax error, no exponent found */
		return -1;
	}
	expo = atoi(saml_token); /* FIXME */
	last_token = (*saml_lexer)();
	if (sign == '-')
		expo = -expo;
	mref_power(result, result, expo);
	return 0;
}
	
int parse_atom (mref_t result)
{
	int ret, sign;
	char *litname;

	switch (last_token) {
	    case '(':
		last_token = (*saml_lexer)();
		ret = parse_expr(result);
		if (last_token != ')')
			ret = -1;  /* no matching parenthesis */
		last_token = (*saml_lexer)();
		return ret;

	    case STOK_INTEGER:
		mref_build(result, ST_INTEGER, saml_token);
		mref_promote(result, model);
		last_token = (*saml_lexer)();
		return 0;

	    case STOK_LITERAL:
	    	litname = alloca(strlen(saml_token) + 32);
	    	strcpy(litname, saml_token);
	    	last_token = (*saml_lexer)();
	    	if (last_token != '!') {
	    		/* Simply a literal */
	    		mref_build(result, ST_LITERAL, litname);
	    		mref_promote(result, model);
	    		return 0;
	    	}
	    	/* This is tensorid!number */
	    	last_token = (*saml_lexer)();
	    	sign = '+';
	    	if (last_token == '+' || last_token == '-') {
	    		sign = last_token;
	    		last_token = (*saml_lexer)();
	    	}
	    	if (last_token != STOK_INTEGER)
	    		return -1;	/* missing vector number */

	    	strcat(litname, "!");
	    	if (sign == '-')
	    		strcat(litname, "-");
	    	/* FIXME -- possible stack overflow */
	    	strcat(litname, saml_token);
	    	last_token = (*saml_lexer)();
	    	
	    	mref_build(result, ST_TENSOR, litname);
	    	mref_promote(result, model);
	    	return 0;

	    default:
	    	/* Syntax error, not an atom */
	    	return -1;
	}
}
