/***************************************************************
LanceMan's quickplot --- a fast interactive 2D plotter

Copyright (C) 1998  Lance Arsenault

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; version 2
of the License.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

***********************************************************************/
/* $Id: read_ascii_file.c,v 1.2 1998/03/11 03:34:56 lance Exp $ */
#include <stdio.h>
#include <assert.h>

#include "data.h"

#define DEBUG 0
#define DELTA_STR_LENGTH 20 /* must be > 1 */ 

static int is_it_seperater_char(char c)
{
  return (c == ' ' || c == '\t' || c == ',');
}

static int file_cleanup_return(int return_val, char *linein_str,
			       Data **d, double *readin)
{
  free(linein_str);
  free(d);
  free(readin);

  return return_val;
}

/* make the string linein_str bigger if it needs it */
static void check_linein_str_length(int current_line_length,char **linein_str)
{
  static int linein_str_length;

  if(*linein_str == NULL)
    linein_str_length = 0;

  if(linein_str_length <= current_line_length)
    assert(NULL != (*linein_str = (char *) realloc((void *) *linein_str,
	 sizeof(char)*(
	 linein_str_length = current_line_length + DELTA_STR_LENGTH))));
}

static int get_next_line(FILE *fileptr, char **linein_str)
{
  int line_length,i,j;
  
  for(line_length=0;(i = getc(fileptr))!=EOF && i!='\n';line_length++)
    {
      check_linein_str_length(line_length,linein_str);
      (*linein_str)[line_length] = i;
    }
  check_linein_str_length(line_length,linein_str);
  (*linein_str)[line_length] = '\0';

  return line_length;
}

static void skip_lines(FILE *fileptr, int num, char *file)
{
  int i,j;

  for(i=0;i<num;i++)
    while( (j =getc(fileptr)) != '\n' && j != EOF);
}

static int count_fields(const char *str)
{
  int i,num_fields,j;

  num_fields = 0;
  for(i = str[j=0]; i != '\0';)
    {   
      /* skip spacer */
      while(i != '\0' && is_it_seperater_char(i))
	i =str[++j];
      if(i != '\0' && !is_it_seperater_char(i))
	num_fields++; /* got one */ 
      /* go to next spacer */
      while(i != '\0' && !is_it_seperater_char(i)) 
	i =str[++j];
    }
  return num_fields;
}


/* put in ' ' in place of the first num strings
 * that are seperated by  seperater_char
 * i.e. is_it_seperater_char()=TRUE
 */
static void strip_strings(char *str, int num)
{
  int i,j;
  
  for(j=0;is_it_seperater_char(str[j])
	&& str[j] != '\0';j++)
    str[j] = ' ';
  for(i=0;i<num && str[j] != '\0';i++)
    {
      while(!is_it_seperater_char(str[j])
	    && str[j] != '\0')
	str[j++] = ' ';
      while(is_it_seperater_char(str[j])
	    && str[j] != '\0')
	  str[j++] = ' ';
    }
}

/* returns number of doubles read in = num_fields on success
 * or     -1 no more data in file 
 * or     -2 for bad data line error and set linein_str
 *
 * x is x[num_fields] is filled with the data in the current line
 * from the file input in the stream of fileptr
 */
static int readin_data_line(FILE *fileptr, char *data_line_in,
			    double *x,int num_fields)
{
  int i,j;

  if(count_fields(data_line_in) == 0 && feof(fileptr))
    return -1; /* no more data in file */
  for(i=0;i<num_fields && 1 == (j= sscanf(data_line_in,"%lf",&x[i]));i++)
    strip_strings(data_line_in,1);
  if(j != 1)
    return -2; /* bad data line error in string data_line_in */
  return num_fields; /* success */
}


int read_ascii_file(FILE *fileptr, char *file, Plot *plot)
{
  int i,j,k,num_fields,n;
  unsigned long num_points /* number of points read in so far */
               ,num_Data;  /* number of elements in the Data array plot->data[] */
  double *readin;
  Data **d;
  char *line_in_str;

  readin = NULL;
  d = NULL;
  line_in_str = NULL;

  /* skip lines if option is set */
  /*******************************/
  skip_lines(fileptr,plot->skip_lines,file);
  if((plot->flag & VERBOSE) && plot->skip_lines > 0)
    fprintf(stderr,
	    "quickplot Info: skipped the first %d line(s) in reading in file: %s\n",
	    plot->skip_lines,file);

  if(plot->flag & READ_LABELS)
    {
fprintf(stderr,"file: %s\n",file);
      if(!(plot->flag & LABELS))/* no labels on the command line */
      {
	int i;

	get_next_line(fileptr,&line_in_str);
	if(plot->labels == NULL)
	  {
	    assert(NULL != (plot->labels = (char *) malloc((strlen(line_in_str) +1)
							   *sizeof(char))));
	    plot->labels[0] = '\0';
	  }
	else
	  {
            int j;
	    assert(NULL != (plot->labels = (char *)
			  realloc((void *) plot->labels ,
				  ((j=strlen(plot->labels))+strlen(line_in_str)+2)
				  *sizeof(char))));
	    plot->labels[j++] = plot->label_seperator;
	    plot->labels[j] = '\0';
	  }
	
	strcpy(&(plot->labels[i = strlen(plot->labels)]),line_in_str);
	if(plot->flag & VERBOSE)
	  fprintf(stderr,
		  "quickplot Info: read in labels <%s> from file: %s\n",
		  line_in_str,file);
      }
      else
	get_next_line(fileptr,&line_in_str);/* skip the label line */
    }
  /* count the number of data fields in the file */
  /***********************************************/
  get_next_line(fileptr,&line_in_str);
  if((num_fields = count_fields(line_in_str)) < 1)
    {
      fprintf(stderr,
	    "quickplot ERROR: can't read any data from line number %d (shown below) from file: %s\nIs it binary data?\n",
	    plot->skip_lines+1,file);
      return file_cleanup_return(0, line_in_str,d,readin);
    }
    
#if(DEBUG)
  printf("the first line read in %s in file %s has %d fields\n",
	 line_in_str,file,num_fields);
#endif

  plot->dim += num_fields;

  realloc_plot_data_and_max_min(plot,num_fields);

  /* need array of Data pointers to move through data with */
  assert((d = (Data **)malloc(sizeof(Data *)*num_fields)) != NULL);

  /* malloc dummy readin for reading in data */
  assert(NULL !=(readin = (double *) malloc(sizeof(double)*num_fields)));

  /* read in data and allocate data struct as we go */
  num_Data = 0;/* size of Data array */
  add_data_size(plot,d,num_fields,&num_Data);

  /* get plot->min and plot->max initialed to the first value */
  /************************************************************/
  if((i = readin_data_line(fileptr,line_in_str,readin,num_fields)) == num_fields)
    for(j=plot->dim-num_fields;j<plot->dim;j++)
      plot->min[j] = plot->max[j] = readin[j + num_fields - plot->dim];/* ERROR HERE */
  else
    {
      fprintf(stderr,
	      "quickplot ERROR: can't read any data from line number %d from file: %s\nIs it binary data?\n",
	      plot->skip_lines+1,file);
      return file_cleanup_return(1, line_in_str,d,readin);
    }

  /** read in data **/
  n = plot->dim - num_fields;
  num_points = 0;
  while(i == num_fields)
    { 
      if(d[0] == NULL)
	add_data_size(plot,d,num_fields,&num_Data);
      for(j=0;j<num_fields;j++)
	{
	  k = j+n;
	  d[j]->val = readin[j];
	  if(readin[j] > plot->max[k])
	    plot->max[k] = readin[j];
	  if(readin[j] < plot->min[k])
	    plot->min[k] = readin[j];

	  d[j] = d[j]->next;
	}
      num_points++;
      get_next_line(fileptr,&line_in_str);
      i = readin_data_line(fileptr,line_in_str,readin,num_fields);
    }

  if(i == -2)/* error in readin_data_line */
    {
      fprintf(stderr,
	      "quickplot ERROR: Error reading data in line number %d in file: %s\n",
	      num_points+plot->skip_lines+1,file);
      return file_cleanup_return(1, line_in_str,d,readin);
    }
  check_data_size(plot,num_fields,num_points);
      
  assert((plot->num_points = (unsigned long *) realloc(
	   (void *)plot->num_points,
	   sizeof(unsigned long)*plot->dim)) != NULL);
  for(i=plot->dim-num_fields;i<plot->dim;i++)
    plot->num_points[i] = num_points;

  read_report(plot,num_points,num_fields,file);

#if(0)
  for(i=0;i<num_fields;i++)
    d[i] = plot->data[plot->dim-num_fields +i];
  while(d[0]!=NULL)
    {
      for(i=0;i<num_fields;i++)
	{
	  printf("%g @%d ",d[i]->val,d[i]);
	  d[i] = d[i]->next;
	  printf("next %d ",d[i]);
	}
      printf("\n");
    }
#endif

  /* free memory and return */
  return file_cleanup_return(0, line_in_str,d,readin);
}
