/*--------------------------------------------------------------------------
  ----- File:        t1aaset.c 
  ----- Author:      Rainer Menzner (rmz@neuroinformatik.ruhr-uni-bochum.de)
                     Subsampling-code by Raph Levien (raph@acm.org)
  ----- Date:        04/18/1998
  ----- Description: This file is part of the t1-library. It contains
                     functions for antialiased setting of characters
		     and strings of characters.
  ----- Copyright:   t1lib is copyrighted (c) Rainer Menzner, 1996-1998. 
                     As of version 0.5, t1lib is distributed under the
		     GNU General Public Library Lincense. The
		     conditions can be found in the files LICENSE and
		     LGPL, which should reside in the toplevel
		     directory of the distribution.  Please note that 
		     there are parts of t1lib that are subject to
		     other licenses:
		     The parseAFM-package is copyrighted by Adobe Systems
		     Inc.
		     The type1 rasterizer is copyrighted by IBM and the
		     X11-consortium.
  ----- Warranties:  Of course, there's NO WARRANTY OF ANY KIND :-)
  ----- Credits:     I want to thank IBM and the X11-consortium for making
                     their rasterizer freely available.
		     Also thanks to Piet Tutelaers for his ps2pk, from
		     which I took the rasterizer sources in a format
		     independent from X11.
                     Thanks to all people who make free software living!
--------------------------------------------------------------------------*/
  
#define T1AASET_C


#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#include "../type1/ffilest.h"
#include "../type1/types.h"
#include "parseAFM.h" 
#include "../type1/objects.h"
#include "../type1/spaces.h"
#include "../type1/util.h"
#include "../type1/fontfcn.h"
#include "../type1/regions.h"


#include "t1types.h"
#include "t1extern.h"
#include "t1aaset.h"
#include "t1set.h"
#include "t1load.h"
#include "t1finfo.h"
#include "t1misc.h"
#include "t1base.h"


#define DEFAULTBPP 8


/* As a fall back */
#ifndef T1_AA_TYPE16 
#define T1_AA_TYPE16    short
#endif
#ifndef T1_AA_TYPE32 
#define T1_AA_TYPE32    int
#endif


unsigned T1_AA_TYPE32 T1aa_gray_val0;   /* black value */
unsigned T1_AA_TYPE32 T1aa_gray_val1;   /* gray 25% value */
unsigned T1_AA_TYPE32 T1aa_gray_val2;   /* gray 50% value */   
unsigned T1_AA_TYPE32 T1aa_gray_val3;   /* gray 75% value */
unsigned T1_AA_TYPE32 T1aa_gray_val4;   /* white value */

static int T1aa_level;
static T1_AA_TYPE32 T1aa_lut[625];
static int T1aa_count[256];


/* The following pixels are for antialiasing under X11; they simply
   represent a bit plane and not yet associated with a specific
   color. Note that the highest bit is always set if a pixel other
   than background is set -> clipping operations in transparent
   mode */
#define PLANE_0 0x00      /* the background color (this means no bit is set) */
#define PLANE_1 0xF1      /* the closest color to background */
#define PLANE_2 0xF2      /*  ...  */
#define PLANE_3 0xF4      /*  ...  */
#define PLANE_4 0xF8      /* the foreground color */



/* T1_AAInit: This function must be called whenever the T1aa_gray_val
   or T1aa_bpp variables change, or the level changes. */
static void T1_AAInit ( int level )
{
  int i;
  int i0, i1, i2, i3;
  unsigned T1_AA_TYPE32 gv[5];

  gv[0] = T1aa_gray_val4;
  gv[1] = T1aa_gray_val3;
  gv[2] = T1aa_gray_val2;
  gv[3] = T1aa_gray_val1;
  gv[4] = T1aa_gray_val0;

  if (level == 2 && T1aa_bpp == 8) {
    for (i0 = 0; i0 < 5; i0++)
      for (i1 = 0; i1 < 5; i1++)
	for (i2 = 0; i2 < 5; i2++)
	  for (i3 = 0; i3 < 5; i3++) {
	    ((char *)T1aa_lut)[(((i0 * 5 + i1) * 5 + i2) * 5 + i3) * 4] = gv[i3];
	    ((char *)T1aa_lut)[(((i0 * 5 + i1) * 5 + i2) * 5 + i3) * 4 + 1] = gv[i2];
	    ((char *)T1aa_lut)[(((i0 * 5 + i1) * 5 + i2) * 5 + i3) * 4 + 2] = gv[i1];
	    ((char *)T1aa_lut)[(((i0 * 5 + i1) * 5 + i2) * 5 + i3) * 4 + 3] = gv[i0];
	  }
    for (i = 0; i < 256; i++) {
      T1aa_count[i] = 0;
      if (i & 0x80) T1aa_count[i] += 125;
      if (i & 0x40) T1aa_count[i] += 125;
      if (i & 0x20) T1aa_count[i] += 25;
      if (i & 0x10) T1aa_count[i] += 25;
      if (i & 0x08) T1aa_count[i] += 5;
      if (i & 0x04) T1aa_count[i] += 5;
      if (i & 0x02) T1aa_count[i] += 1;
      if (i & 0x01) T1aa_count[i] += 1;
    }
  } else if (level == 2 && T1aa_bpp == 16) {
    for (i0 = 0; i0 < 5; i0++)
      for (i1 = 0; i1 < 5; i1++) {
	((short *)T1aa_lut)[(i0 * 5 + i1) * 2] = gv[i1];
	((short *)T1aa_lut)[(i0 * 5 + i1) * 2 + 1] = gv[i0];
      }
    for (i = 0; i < 256; i++) {
      T1aa_count[i] = 0;
      if (i & 0x80) T1aa_count[i] += 160;
      if (i & 0x40) T1aa_count[i] += 160;
      if (i & 0x20) T1aa_count[i] += 32;
      if (i & 0x10) T1aa_count[i] += 32;
      if (i & 0x08) T1aa_count[i] += 5;
      if (i & 0x04) T1aa_count[i] += 5;
      if (i & 0x02) T1aa_count[i] += 1;
      if (i & 0x01) T1aa_count[i] += 1;
    }
  } else if (level == 2 && T1aa_bpp == 32) {
    for (i0 = 0; i0 < 5; i0++)
      ((long *)T1aa_lut)[i0] = gv[i0];
    for (i = 0; i < 256; i++) {
      T1aa_count[i] = 0;
      if (i & 0x80) T1aa_count[i] += 512;
      if (i & 0x40) T1aa_count[i] += 512;
      if (i & 0x20) T1aa_count[i] += 64;
      if (i & 0x10) T1aa_count[i] += 64;
      if (i & 0x08) T1aa_count[i] += 8;
      if (i & 0x04) T1aa_count[i] += 8;
      if (i & 0x02) T1aa_count[i] += 1;
      if (i & 0x01) T1aa_count[i] += 1;
    }
  } else {
    /* unsupported combination of level and bpp - how to report the error? */
  }
  T1aa_level = level;
}

/* T1_AADoLine: Create a single scanline of antialiased output. The
   (x, y) arguments refer to the number of pixels in the input image
   to convert down. The width argument is the number of bytes
   separating scanlines in the input. */
static void T1_AADoLine ( int level, int x, int y, int width,
			  char *c_in_ptr,
			  char *target_ptr )
{
  int i;
  int size;
  char buf[8];
  int count;
  
  static char *align_buf = NULL;
  static int align_buf_size = 0;
  unsigned char *in_ptr;
  
  int new_size=55;
  char *optr;

  

  /* Perhaps we should mask the last byte when the line end isn't aligned
     with a byte, but it always seems to be zero, so we don't bother. */

  /* We convert the input pointer to unsigned since we use it as index! */
  in_ptr=(unsigned char*)c_in_ptr;
  
  if (T1aa_level != level){
    T1_AAInit (level);
  }


  if ((long)target_ptr & 3){
    /* calculate new_size (size in bytes of output buffer */
    if (level == 2)
      new_size = ((x + 1) >> 1) * (T1aa_bpp >> 3);
    if (new_size > align_buf_size)
      {
	if (align_buf)
	  free (align_buf);
	align_buf = (char *)malloc(new_size);
	align_buf_size = new_size;
      }
    optr = align_buf;
  }
  else
    optr = target_ptr;
  
  
  if (level == 2) {
    size = (x + 1) >> 3;
    
    if (T1aa_bpp == 8) {
      if (y == 2){
	for (i = 0; i < size; i++) {
	  ((T1_AA_TYPE32 *)optr)[i] = T1aa_lut[(T1aa_count[in_ptr[i]] +
					T1aa_count[in_ptr[i + width]])];
	}
      }
      else if (y == 1){
	for (i = 0; i < size; i++) {
	  /* could collapse the luts here, but it would be a marginal
	     performance gain at best. */
	  ((T1_AA_TYPE32 *)optr)[i] = T1aa_lut[(T1aa_count[in_ptr[i]])];
	}
      }
      /* Note: (x+1)&6 is an identical statement to ((x+1)>>1)&3 which
	 in turn is the horizontal pixelnumber in the aaglyph */
      if ((x+1) & 6) {
	if (y == 2)
	  ((T1_AA_TYPE32 *)buf)[0] = T1aa_lut[(T1aa_count[in_ptr[i]] +
				       T1aa_count[in_ptr[i + width]])];
	else if (y == 1)
	  ((T1_AA_TYPE32 *)buf)[0] = T1aa_lut[(T1aa_count[in_ptr[i]])];
	memcpy (optr + i * 4, buf, ((x+1)>>1)&3 );
      }
      
    } else if (T1aa_bpp == 16) {
      if (y == 2)
	for (i = 0; i < size; i++) {
	  count = T1aa_count[in_ptr[i]] + T1aa_count[in_ptr[i + width]];
	  ((T1_AA_TYPE32 *)optr)[i * 2] = T1aa_lut[count & 31];
	  ((T1_AA_TYPE32 *)optr)[i * 2 + 1] = T1aa_lut[count >> 5];
	}
      else if (y == 1)
	for (i = 0; i < size; i++) {
	  count = T1aa_count[in_ptr[i]];
	  ((T1_AA_TYPE32 *)optr)[i * 2] = T1aa_lut[count & 31];
	  ((T1_AA_TYPE32 *)optr)[i * 2 + 1] = T1aa_lut[count >> 5];
	}
      if ((x + 1) & 6) {
	if (y == 2)
	  count = T1aa_count[in_ptr[i]] + T1aa_count[in_ptr[i + width]];
	else if (y == 1)
	  count = T1aa_count[in_ptr[i]];
	((T1_AA_TYPE32 *)buf)[0] = T1aa_lut[count & 31];
	((T1_AA_TYPE32 *)buf)[1] = T1aa_lut[count >> 5];
	memcpy (optr + i * 8, buf, (((x+1)>>1)&3)<<1 );
      }
    } else if (T1aa_bpp == 32) {
      if (y == 2)
	for (i = 0; i < size; i++) {
	  count = T1aa_count[in_ptr[i]] + T1aa_count[in_ptr[i + width]];
	  ((T1_AA_TYPE32 *)optr)[i * 4] = T1aa_lut[count & 7];
	  ((T1_AA_TYPE32 *)optr)[i * 4 + 1] = T1aa_lut[(count >> 3) & 7];
	  ((T1_AA_TYPE32 *)optr)[i * 4 + 2] = T1aa_lut[(count >> 6) & 7];
	  ((T1_AA_TYPE32 *)optr)[i * 4 + 3] = T1aa_lut[(count >> 9) & 7];
	}
      else if (y == 1)
	for (i = 0; i < size; i++) {
	  count = T1aa_count[in_ptr[i]];
	  ((T1_AA_TYPE32 *)optr)[i * 4] = T1aa_lut[count & 7];
	  ((T1_AA_TYPE32 *)optr)[i * 4 + 1] = T1aa_lut[(count >> 3) & 7];
	  ((T1_AA_TYPE32 *)optr)[i * 4 + 2] = T1aa_lut[(count >> 6) & 7];
	  ((T1_AA_TYPE32 *)optr)[i * 4 + 3] = T1aa_lut[(count >> 9) & 7];
	}
      if ((x + 1) & 6) {
	if (y == 2)
	  count = T1aa_count[in_ptr[i]] + T1aa_count[in_ptr[i + width]];
	else if (y == 1)
	  count = T1aa_count[in_ptr[i]];
	((T1_AA_TYPE32 *)optr)[i * 4] = T1aa_lut[count & 7];
	if (((x + 1) & 6) > 2) {
	  ((T1_AA_TYPE32 *)optr)[i * 4 + 1] = T1aa_lut[(count >> 3) & 7];
	  if (((x + 1) & 6) > 4) {
	    ((T1_AA_TYPE32 *)optr)[i * 4 + 2] = T1aa_lut[(count >> 6) & 7];
	  }
	}
      }
    }
  }

  if ((long)target_ptr & 3){
    memcpy (target_ptr, align_buf, new_size);
  }
  
}


/* T1_AASetChar(...): Generate the anti-aliased bitmap for a character */
GLYPH *T1_AASetChar( int FontID, char charcode, float size,
		     float angle)
{
  
  GLYPH *glyph;   /* pointer to bitmap glyph */
  static GLYPH aaglyph={NULL,{0,0,0,0,0},NULL,DEFAULTBPP};/* The anti-aliased glyph */
  long asc, dsc, ht, wd;
  long i, k;
  long n_horz, n_horz_pad, n_vert, n_asc, n_dsc;
  long v_start_exact, v_end_exact;
  char *target_ptr;
  long offset;
  char *ptr;
  int y;

  int memsize;
  LONG paddedW;
 

  /* Reset character glyph, if necessary */
  if (aaglyph.bits!=NULL){
    free(aaglyph.bits);
    aaglyph.bits=NULL;
  }
  aaglyph.metrics.leftSideBearing=0;
  aaglyph.metrics.rightSideBearing=0;
  aaglyph.metrics.advanceX=0;
  aaglyph.metrics.advanceY=0;
  aaglyph.metrics.ascent=0;
  aaglyph.metrics.descent=0;
  aaglyph.pFontCacheInfo=NULL;
  aaglyph.bpp=T1aa_bpp;
  
  /* First, call routine to rasterize character, all error checking is
     done in this function: */ 
  if ((glyph=T1_SetChar( FontID, charcode, 2*size, angle))==NULL)
    return(NULL); /* An error occured */
  
  /* Get dimensions of bitmap: */
  asc=glyph->metrics.ascent;
  dsc=glyph->metrics.descent;
  ht=asc-dsc;
  wd=glyph->metrics.rightSideBearing-glyph->metrics.leftSideBearing;
  
  /* Set some looping parameters for subsampling */
  /* The horizontal number of pixels in aaglyph */
  n_horz=(wd+1)>>1;
  /* And the padded value */
  n_horz_pad=PAD( n_horz*T1aa_bpp, pFontBase->bitmap_pad )>>3;
  
  /* vertical number of steps: */
  if (asc % 2){ /* odd ascent */
    n_asc= asc > 0 ? asc/2+1 : asc/2-1;
    v_start_exact=0;
  }
  else{
    n_asc=asc/2;
    v_start_exact=1;
  }
  if (dsc % 2){ /* odd descent */
    n_dsc= dsc > 0 ? dsc/2+1 : dsc/2-1;
    v_end_exact=0;
  }
  else{
    n_dsc=dsc/2;
    v_end_exact=1;
  }
  /* the total number of lines: */
  n_vert=n_asc-n_dsc;

  /* Allocate memory for glyph */
  memsize = n_horz_pad*n_vert;
  
  aaglyph.bits = (char *)malloc(memsize*sizeof( char));
  if (aaglyph.bits == NULL) {
    T1_errno=T1ERR_ALLOC_MEM;
    return(NULL);
  }
  

  paddedW=PAD(wd,pFontBase->bitmap_pad)/8;
  offset=0;
  target_ptr=aaglyph.bits;
  
  ptr = glyph->bits;
  for (i = 0; i < n_vert; i++) {
    if (((i==0)&&(v_start_exact==0)) || ((i==n_vert-1)&&(v_end_exact==0)))
      y=1;
    else
      y=2;

    T1_AADoLine ( 2, wd, y, paddedW, ptr, target_ptr );
    ptr += y * paddedW;
    target_ptr += n_horz_pad;
  }
  k = n_horz;
  
  /* .. and set them in aaglyph */
  aaglyph.metrics.leftSideBearing=(int) floor(glyph->metrics.leftSideBearing / 2.0 +0.5);
  aaglyph.metrics.rightSideBearing=aaglyph.metrics.leftSideBearing + k;
  aaglyph.metrics.advanceX=(int) floor(glyph->metrics.advanceX / 2.0 +0.5);
  aaglyph.metrics.advanceY=(int) floor(glyph->metrics.advanceY / 2.0 +0.5);
  aaglyph.metrics.ascent=n_asc;
  aaglyph.metrics.descent=n_dsc;
  aaglyph.pFontCacheInfo=NULL;

  return(&aaglyph);
}



/* T1_AASetString(...): Generate the antialiased bitmap for a
   string of characters */
GLYPH *T1_AASetString( int FontID, char *string, int len, 
		       long spaceoff, int modflag, float size,
		       float angle)
{
  GLYPH *glyph;   /* pointer to bitmap glyph */
  static GLYPH aastring_glyph={NULL,{0,0,0,0,0},NULL,DEFAULTBPP};/* The anti-aliased glyph */
  long asc, dsc, ht, wd;
  long i, k;
  long n_horz, n_horz_pad, n_vert, n_asc, n_dsc;
  long v_start_exact, v_end_exact;
  char *target_ptr;
  long offset;
  char *ptr;
  int y;
  
  int memsize;
  LONG paddedW;

  
  /* Reset character glyph, if necessary */
  if (aastring_glyph.bits!=NULL){
    free(aastring_glyph.bits);
    aastring_glyph.bits=NULL;
  }
  aastring_glyph.metrics.leftSideBearing=0;
  aastring_glyph.metrics.rightSideBearing=0;
  aastring_glyph.metrics.advanceX=0;
  aastring_glyph.metrics.advanceY=0;
  aastring_glyph.metrics.ascent=0;
  aastring_glyph.metrics.descent=0;
  aastring_glyph.pFontCacheInfo=NULL;
  aastring_glyph.bpp=T1aa_bpp;
  
  /* First, call routine to rasterize character, all error checking is
     done in this function: */ 
  if ((glyph=T1_SetString( FontID, string, len, spaceoff,
			   modflag, 2*size, angle))==NULL)
    return(NULL); /* An error occured */
  
  /* Get dimensions of bitmap: */
  asc=glyph->metrics.ascent;
  dsc=glyph->metrics.descent;
  ht=asc-dsc;
  wd=glyph->metrics.rightSideBearing-glyph->metrics.leftSideBearing;
  
  /* Set some looping parameters for subsampling */
  /* The horizontal number of steps: */
  n_horz=(wd+1)>>1;
  /* And the padded value */
  n_horz_pad=PAD( n_horz*T1aa_bpp, pFontBase->bitmap_pad )>>3;

  /* vertical number of steps: */
  if (asc % 2){ /* odd ascent */
    n_asc= asc > 0 ? asc/2+1 : asc/2-1;
    v_start_exact=0;
  }
  else{
    n_asc=asc/2;
    v_start_exact=1;
  }
  if (dsc % 2){ /* odd descent */
    n_dsc= dsc > 0 ? dsc/2+1 : dsc/2-1;
    v_end_exact=0;
  }
  else{
    n_dsc=dsc/2;
    v_end_exact=1;
  }
  /* the total number of lines: */
  n_vert=n_asc-n_dsc;
  
  
  /* Allocate memory for glyph */
  memsize = n_horz_pad*n_vert;
  
  aastring_glyph.bits = (char *)malloc(memsize*sizeof( char));
  if (aastring_glyph.bits == NULL) {
    T1_errno=T1ERR_ALLOC_MEM;
    return(NULL);
  }
  
  
  paddedW=PAD(wd,pFontBase->bitmap_pad)/8;
  offset=0;
  target_ptr=aastring_glyph.bits;
  
  ptr = glyph->bits;
  for (i = 0; i < n_vert; i++) {
    if (((i==0)&&(v_start_exact==0)) || ((i==n_vert-1)&&(v_end_exact==0)))
      y=1;
    else
      y=2;
    T1_AADoLine ( 2, wd, y, paddedW, ptr, target_ptr );
    ptr += y * paddedW;
    target_ptr += n_horz_pad;
  }
  k = n_horz;
  
  /* .. and set them in aastring_glyph */
  aastring_glyph.metrics.leftSideBearing=(int) floor(glyph->metrics.leftSideBearing / 2.0 + 0.5);
  aastring_glyph.metrics.rightSideBearing=aastring_glyph.metrics.leftSideBearing + k;
  aastring_glyph.metrics.advanceX=(int) floor(glyph->metrics.advanceX / 2.0 +0.5);
  aastring_glyph.metrics.advanceY=(int) floor(glyph->metrics.advanceY / 2.0 +0.5);
  aastring_glyph.metrics.ascent=n_asc;
  aastring_glyph.metrics.descent=n_dsc;
  aastring_glyph.pFontCacheInfo=NULL;

  return(&aastring_glyph);
}



/* T1_AASetGrayValues(): Sets the byte values that are put into the
   pixel position for the respective entries:
   Returns 0 if successfull.
   */
int T1_AASetGrayValues(unsigned long white,
		       unsigned long gray75,
		       unsigned long gray50,
		       unsigned long gray25,
		       unsigned long black)
{
  
   if (CheckForInit()){
     T1_errno=T1ERR_OP_NOT_PERMITTED;
     return(-1);
   }
   
   T1aa_gray_val0=(unsigned T1_AA_TYPE32)black;    /* black value */
   T1aa_gray_val1=(unsigned T1_AA_TYPE32)gray25;   /* gray 25% value */
   T1aa_gray_val2=(unsigned T1_AA_TYPE32)gray50;   /* gray 50% value */   
   T1aa_gray_val3=(unsigned T1_AA_TYPE32)gray75;   /* gray 75% value */
   T1aa_gray_val4=(unsigned T1_AA_TYPE32)white;    /* white value */

   T1aa_level = 0;

   return(0);
   
}

		     
/* T1_AASetBitsPerPixel(): Sets the depths of the antialiased glyph
   pixel. Returns 0 if bpp is valid and -1 otherwise. If 24 is
   specified, meaning to be the depth rather than the bpp-value,
   automatically 32 bpp is chosen. */
int  T1_AASetBitsPerPixel( int bpp)
{
  
  if (CheckForInit()){
    T1_errno=T1ERR_OP_NOT_PERMITTED;
    return(-1);
  }
  

  T1aa_level = 0;

  if (bpp==8){
    T1aa_bpp=8;
    return(0);
  }
  if (bpp==16){
    T1aa_bpp=16;
    return(0);
  }
  if ((bpp==32)|(bpp==24)){
    T1aa_bpp=32;
    return(0);
  }

  T1_errno=T1ERR_INVALID_PARAMETER;
  return(-1);
}
