// fl_rect.C

// These routines from fl_draw.H are used by the standard boxtypes
// and thus are always linked into an fltk program.

// Also all fl_clip routines, since they are always linked in so
// that minimal update works.

#include <FL/Fl_Widget.H>
#include <FL/fl_draw.H>
#include <FL/x.H>

void fl_rect(int x, int y, int w, int h) {
  if (w<=0 || h<=0) return;
#ifdef WIN32
  MoveToEx(fl_gc, x, y, 0L); 
  LineTo(fl_gc, x+w-1, y);
  LineTo(fl_gc, x+w-1, y+h-1);
  LineTo(fl_gc, x, y+h-1);
  LineTo(fl_gc, x, y);
#else
  XDrawRectangle(fl_display, fl_window, fl_gc, x, y, w-1, h-1);
#endif
}

void fl_rectf(int x, int y, int w, int h) {
  if (w<=0 || h<=0) return;
#ifdef WIN32
  RECT rect;
  rect.left = x; rect.top = y;  
  rect.right = x + w; rect.bottom = y + h;
  FillRect(fl_gc, &rect, fl_brush());
#else
  if (w && h) XFillRectangle(fl_display, fl_window, fl_gc, x, y, w, h);
#endif
}

void fl_xyline(int x, int y, int x1) {
#ifdef WIN32
  MoveToEx(fl_gc, x, y, 0L); LineTo(fl_gc, x1+1, y);
#else
  XDrawLine(fl_display, fl_window, fl_gc, x, y, x1, y);
#endif
}

void fl_xyline(int x, int y, int x1, int y2) {
#ifdef WIN32
  if (y2 < y) y2--;
  MoveToEx(fl_gc, x, y, 0L); 
  LineTo(fl_gc, x1, y);
  LineTo(fl_gc, x1, y2);
#else
  XPoint p[3];
  p[0].x = x;  p[0].y = p[1].y = y;
  p[1].x = p[2].x = x1; p[2].y = y2;
  XDrawLines(fl_display, fl_window, fl_gc, p, 3, 0);
#endif
}

void fl_xyline(int x, int y, int x1, int y2, int x3) {
#ifdef WIN32
  MoveToEx(fl_gc, x, y, 0L); 
  LineTo(fl_gc, x1, y);
  LineTo(fl_gc, x1, y2);
  LineTo(fl_gc, x3, y2);
#else
  XPoint p[4];
  p[0].x = x;  p[0].y = p[1].y = y;
  p[1].x = p[2].x = x1; p[2].y = p[3].y = y2;
  p[3].x = x3;
  XDrawLines(fl_display, fl_window, fl_gc, p, 4, 0);
#endif
}

void fl_yxline(int x, int y, int y1) {
#ifdef WIN32
  if (y1 < y) y1--;
  MoveToEx(fl_gc, x, y, 0L); LineTo(fl_gc, x, y1);
#else
  XDrawLine(fl_display, fl_window, fl_gc, x, y, x, y1);
#endif
}

void fl_yxline(int x, int y, int y1, int x2) {
#ifdef WIN32
  if (x2 > x) x2++;
  MoveToEx(fl_gc, x, y, 0L); 
  LineTo(fl_gc, x, y1);
  LineTo(fl_gc, x2, y1);
#else
  XPoint p[3];
  p[0].x = p[1].x = x;  p[0].y = y;
  p[1].y = p[2].y = y1; p[2].x = x2;
  XDrawLines(fl_display, fl_window, fl_gc, p, 3, 0);
#endif
}

void fl_yxline(int x, int y, int y1, int x2, int y3) {
#ifdef WIN32
  MoveToEx(fl_gc, x, y, 0L); 
  LineTo(fl_gc, x, y1);
  LineTo(fl_gc, x2, y1);
  LineTo(fl_gc, x2, y3);
#else
  XPoint p[4];
  p[0].x = p[1].x = x;  p[0].y = y;
  p[1].y = p[2].y = y1; p[2].x = p[3].x = x2;
  p[3].y = y3;
  XDrawLines(fl_display, fl_window, fl_gc, p, 4, 0);
#endif
}

void fl_line(int x, int y, int x1, int y1) {
#ifdef WIN32
  MoveToEx(fl_gc, x, y, 0L); 
  LineTo(fl_gc, x1, y1);
#else
  XDrawLine(fl_display, fl_window, fl_gc, x, y, x1, y1);
#endif
}

void fl_line(int x, int y, int x1, int y1, int x2, int y2) {
#ifdef WIN32
  MoveToEx(fl_gc, x, y, 0L); 
  LineTo(fl_gc, x1, y1);
  LineTo(fl_gc, x2, y2);
#else
  XPoint p[3];
  p[0].x = x;  p[0].y = y;
  p[1].x = x1; p[1].y = y1;
  p[2].x = x2; p[2].y = y2;
  XDrawLines(fl_display, fl_window, fl_gc, p, 3, 0);
#endif
}

void fl_loop(int x, int y, int x1, int y1, int x2, int y2) {
#ifdef WIN32
  MoveToEx(fl_gc, x, y, 0L); 
  LineTo(fl_gc, x1, y1);
  LineTo(fl_gc, x2, y2);
  LineTo(fl_gc, x, y);
#else
  XPoint p[4];
  p[0].x = x;  p[0].y = y;
  p[1].x = x1; p[1].y = y1;
  p[2].x = x2; p[2].y = y2;
  p[3].x = x;  p[3].y = y;
  XDrawLines(fl_display, fl_window, fl_gc, p, 4, 0);
#endif
}

void fl_loop(int x, int y, int x1, int y1, int x2, int y2, int x3, int y3) {
#ifdef WIN32
  MoveToEx(fl_gc, x, y, 0L); 
  LineTo(fl_gc, x1, y1);
  LineTo(fl_gc, x2, y2);
  LineTo(fl_gc, x3, y3);
  LineTo(fl_gc, x, y);
#else
  XPoint p[5];
  p[0].x = x;  p[0].y = y;
  p[1].x = x1; p[1].y = y1;
  p[2].x = x2; p[2].y = y2;
  p[3].x = x3; p[3].y = y3;
  p[4].x = x;  p[4].y = y;
  XDrawLines(fl_display, fl_window, fl_gc, p, 5, 0);
#endif
}

void fl_polygon(int x, int y, int x1, int y1, int x2, int y2) {
  XPoint p[3];
  p[0].x = x;  p[0].y = y;
  p[1].x = x1; p[1].y = y1;
  p[2].x = x2; p[2].y = y2;
#ifdef WIN32
  SelectObject(fl_gc, fl_brush());
  Polygon(fl_gc, p, 3);
#else
  XFillPolygon(fl_display, fl_window, fl_gc, p, 3, Convex, 0);
#endif
}

void fl_polygon(int x, int y, int x1, int y1, int x2, int y2, int x3, int y3) {
  XPoint p[4];
  p[0].x = x;  p[0].y = y;
  p[1].x = x1; p[1].y = y1;
  p[2].x = x2; p[2].y = y2;
  p[3].x = x3; p[3].y = y3;
#ifdef WIN32
  SelectObject(fl_gc, fl_brush());
  Polygon(fl_gc, p, 4);
#else
  XFillPolygon(fl_display, fl_window, fl_gc, p, 4, Convex, 0);
#endif
}

////////////////////////////////////////////////////////////////

#ifdef WIN32

static struct rect {int x, y, r, b;} rstack[10];
static int rstackptr;
int fl_clip_state_number; // used by gl_begin.C to update GL clip
extern char fl_direct_paint; // in Fl_win32.C

void fl_clip(int x, int y, int w, int h) {
  fl_clip_state_number++;
  int r = x+w;
  int b = y+h;
  if (rstackptr) {
    rect& rr = rstack[rstackptr-1];
    if (rr.x > x) x = rr.x;
    if (rr.y > y) y = rr.y;
    if (rr.r < r) r = rr.r;
    if (rr.b < b) b = rr.b;
  }
  rect* rp = &rstack[rstackptr++];
  rp->x = x;
  rp->y = y;
  rp->r = r;
  rp->b = b;
  if (rstackptr == 1 && fl_direct_paint) return;
  HRGN R = CreateRectRgn(x,y,r,b);
  SelectClipRgn(fl_gc, R);
  DeleteObject(R);
}

void fl_pop_clip() {
  fl_clip_state_number++;
  rstackptr--;
  HRGN R;
  if (rstackptr) {
    rect& r = rstack[rstackptr-1];
    R = CreateRectRgn(r.x, r.y, r.r, r.b);
  } else {
    R = 0L;
  }
  SelectClipRgn(fl_gc, R);
  if (R) DeleteObject(R);
}

// does this rectangle intersect current clip?
int fl_not_clipped(int x, int y, int w, int h) {
  if (!rstackptr) return 2;
  rect& r = rstack[rstackptr-1];
  return (x < r.r && x+w > r.x && y < r.b && y+h > r.y);
  return 1;
}

// return rectangle surrounding intersection of this rectangle and clip:
int fl_clip_box(int x, int y, int w, int h, int& X, int& Y, int& W, int& H){
  X = x; Y = y; W = w; H = h;
  if (!rstackptr) return 0;
  int R = x+w;
  int B = y+h;
  rect& r = rstack[rstackptr-1];
  int ret = 0;
  if (r.x > x) {X = r.x; ret = 1;}
  if (r.y > y) {Y = r.y; ret = 1;}
  if (r.r < R) {R = r.r; ret = 1;}
  if (r.b < B) {B = r.b; ret = 1;}
  if (B <= Y || R <= X) {W = H = 0; return 2;}
  W = R-X;
  H = B-Y;
  return ret;
}

#else

static Region rstack[10];
static int rstackptr;
int fl_clip_state_number; // used by gl_begin.C to update GL clip

// Set the region before drawing a window due to expose events:
void fl_clip_region(Region r) {
  if (r) {
    fl_clip_state_number++;
    rstack[0] = r;
    rstackptr = 1;
    XSetRegion(fl_display, fl_gc, r);
  }
}

// Missing X call: (is this the fastest way to init a 1-rectangle region?)
static Region XRectangleRegion(int x, int y, int w, int h) {
  XRectangle R;
  R.x = x; R.y = y; R.width = w; R.height = h;
  Region r = XCreateRegion();
  XUnionRectWithRegion(&R, r, r);
  return r;
}

// Intersect & push a new clip rectangle:
void fl_clip(int x, int y, int w, int h) {
  fl_clip_state_number++;
  Region r;
  if (w > 0 && h > 0) {
    r = XRectangleRegion(x,y,w,h);
    if (rstackptr) {
      Region temp = XCreateRegion();
      XIntersectRegion(rstack[rstackptr-1], r, temp);
      XDestroyRegion(r);
      r = temp;
    }
  } else { // make empty clip region:
    r = XCreateRegion();
  }
  rstack[rstackptr++] = r;
  XSetRegion(fl_display, fl_gc, r);
}

// pop back to previous clip:
void fl_pop_clip() {
  fl_clip_state_number++;
  XDestroyRegion(rstack[--rstackptr]);
  if (rstackptr) XSetRegion(fl_display, fl_gc, rstack[rstackptr-1]);
  else XSetClipMask(fl_display, fl_gc, 0);
}

// does this rectangle intersect current clip?
int fl_not_clipped(int x, int y, int w, int h) {
  return rstackptr ? XRectInRegion(rstack[rstackptr-1], x, y, w, h) : 1;
}

// return rectangle surrounding intersection of this rectangle and clip:
int fl_clip_box(int x, int y, int w, int h, int& X, int& Y, int& W, int& H){
  X = x; Y = y; W = w; H = h;
  if (!rstackptr) return 0;
  Region r = rstack[rstackptr-1];
  switch (XRectInRegion(r, x, y, w, h)) {
  case 0: // completely outside
    W = H = 0;
    return 2;
  case 1: // completely inside:
    return 0;
  default: // partial:
    break;
  }
  Region rr = XRectangleRegion(x,y,w,h);
  Region temp = XCreateRegion();
  XIntersectRegion(r, rr, temp);
  XRectangle rect;
  XClipBox(temp, &rect);
  X = rect.x; Y = rect.y; W = rect.width; H = rect.height;
  XDestroyRegion(temp);
  XDestroyRegion(rr);
  return 1;
}

#endif

// end of fl_rect.C
