/* 
 * This code is based on the ISO filesystem support in MILO (by
 * Dave Rusling).
 *
 * This is a set of functions that provides minimal filesystem
 * functionality to the Linux bootstrapper.  All we can do is
 * open and read files... but that's all we need 8-)
 */
#include <linux/kernel.h>
#include <linux/config.h>
#include <linux/string.h>
#include <linux/stat.h>
#include <linux/kdev_t.h>

#include <iso.h>
#include <isolib.h>

extern int printf (const char * format, ...);

#undef DEBUG_ISO 

/* iso9660 support code */

#define MAX_OPEN_FILES 5

static struct inode_table_entry {
    struct iso_inode inode;
    int inumber;
    int free;
    unsigned short old_mode;
    unsigned size;
    int nlink;
    int mode;
    void *start;
} inode_table[MAX_OPEN_FILES];

static unsigned long root_inode = 0;
static struct isofs_super_block sb;
static char data_block[1024];
static char big_data_block[2048];

#ifndef S_IRWXUGO
# define S_IRWXUGO	(S_IRWXU|S_IRWXG|S_IRWXO)
# define S_IXUGO	(S_IXUSR|S_IXGRP|S_IXOTH)
# define S_IRUGO	(S_IRUSR|S_IRGRP|S_IROTH)
#endif


int
isonum_711 (char * p)
{
	return (*p & 0xff);
}


int
isonum_712 (char * p)
{
	int val;

	val = *p;
	if (val & 0x80)
		val |= 0xffffff00;
	return val;
}


int
isonum_721 (char * p)
{
	return ((p[0] & 0xff) | ((p[1] & 0xff) << 8));
}

int
isonum_722 (char * p)
{
	return (((p[0] & 0xff) << 8) | (p[1] & 0xff));
}


int
isonum_723 (char * p)
{
	return isonum_721(p);
}


int
isonum_731 (char * p)
{
	return ((p[0] & 0xff)
		| ((p[1] & 0xff) << 8)
		| ((p[2] & 0xff) << 16)
		| ((p[3] & 0xff) << 24));
}


int
isonum_732 (char * p)
{
	return (((p[0] & 0xff) << 24)
		| ((p[1] & 0xff) << 16)
		| ((p[2] & 0xff) << 8)
		| (p[3] & 0xff));
}


int
isonum_733 (char * p)
{
	return isonum_731(p);
}


static int
iso_breadi (long blkno, long nblks, char * buffer)
{
	extern long iso_dev_read (void * buf, long offset, long size);
	long offset, size;

#ifdef DEBUG_ISO
	printf("iso_breadi: reading blocks %ld..%ld\n",
	       blkno, blkno + nblks - 1);
#endif
	offset = blkno * sb.s_blocksize;
	size   = nblks * sb.s_blocksize;
	if (iso_dev_read(buffer, offset, size) != size) {
		printf("iso_breadi: read error on block %ld\n", blkno);
		return -1;
	}
	return 0;
}


/*
 * Release our hold on an inode.  Since this is a read-only application,
 * don't worry about putting back any changes...
 */
static void
iso_iput (struct iso_inode *ip)
{
	struct inode_table_entry *itp;

	/* Find and free the inode table slot we used... */
	itp = (struct inode_table_entry *) ip;

	itp->inumber = 0;
	itp->free = 1;
}


/*
 * Read the specified inode from the disk and return it to the user.
 * Returns NULL if the inode can't be read...
 */
static struct iso_inode *
iso_iget (int ino)
{
	int i;
	struct iso_inode *inode;
	struct inode_table_entry *itp;
	struct iso_directory_record * raw_inode;
	unsigned char *pnt = NULL;
	void *cpnt = NULL;
	int high_sierra;
	int block;

#ifdef DEBUG_ISO
	printf("iso_iget(ino=%d)\n", ino);
#endif

	/* find a free inode to play with */
	inode = NULL;
	itp = NULL;
	for (i = 0; i < MAX_OPEN_FILES; i++) {
		if (inode_table[i].free) {
			itp = &(inode_table[i]);
			inode = &(itp->inode);
			break;
		}
	}
	if ((inode == NULL) || (itp == NULL)) {
		printf("iso9660 (iget): no free inodes\n");
		return (NULL);
	}

	block = ino >> sb.s_blocksize_bits;
#ifdef DEBUG_ISO
	printf("iso_inode: reading ino %d, block %d at offset %d\n",
	       ino, block, sb.s_blocksize * ino);
#endif
	if (iso_breadi(block, 1, data_block) < 0) {
		printf("iso9660: unable to read i-node block");
		return NULL;
	}

	pnt = ((unsigned char *) data_block + (ino & (sb.s_blocksize - 1)));
	raw_inode = ((struct iso_directory_record *) pnt);
	high_sierra = sb.s_high_sierra;

	if ((ino & (sb.s_blocksize - 1)) + *pnt > sb.s_blocksize){
		int frag1, offset;

		offset = (ino & (sb.s_blocksize - 1));
		frag1 = sb.s_blocksize - offset;
		cpnt = big_data_block;
		memcpy(cpnt, data_block + offset, frag1);
		if (iso_breadi(++block, 1, data_block) < 0) {
			printf("unable to read i-node block");
			return NULL;
		}
		offset += *pnt - sb.s_blocksize;
		memcpy((char *)cpnt+frag1, data_block, offset);
		pnt = ((unsigned char *) cpnt);
		raw_inode = ((struct iso_directory_record *) pnt);
	}

	if (raw_inode->flags[-high_sierra] & 2) {
		itp->mode = S_IRUGO | S_IXUGO | S_IFDIR;
		itp->nlink = 1; /* Set to 1.  We know there are 2, but
				   the find utility tries to optimize
				   if it is 2, and it screws up.  It is
				   easier to give 1 which tells find to
				   do it the hard way. */
	} else {
		itp->mode = sb.s_mode; /* Everybody gets to read the file. */
		itp->nlink = 1;
		itp->mode |= S_IFREG;
		/*
		 * If there are no periods in the name, then set the
		 * execute permission bit
		 */
		for(i=0; i< raw_inode->name_len[0]; i++)
			if(raw_inode->name[i]=='.' || raw_inode->name[i]==';')
				break;
		if(i == raw_inode->name_len[0] || raw_inode->name[i] == ';') 
			itp->mode |= S_IXUGO; /* execute permission */
	}

	itp->size = isonum_733 (raw_inode->size);

	/* There are defective discs out there - we do this to protect
	   ourselves.  A cdrom will never contain more than 700Mb */
	if((itp->size < 0 || itp->size > 700000000) &&
	   sb.s_cruft == 'n')
	{
		printf("Warning: defective cdrom.  "
		       "Enabling \"cruft\" mount option.\n");
		sb.s_cruft = 'y';
	}

	/*
	 * Some dipshit decided to store some other bit of information
	 * in the high byte of the file length.  Catch this and
	 * holler.  WARNING: this will make it impossible for a file
	 * to be > 16Mb on the CDROM!!!
	 */
	if(sb.s_cruft == 'y' && 
	   itp->size & 0xff000000)
	{
		itp->size &= 0x00ffffff;
	}

	if (raw_inode->interleave[0]) {
		printf("Interleaved files not (yet) supported.\n");
		itp->size = 0;
	}

	/* I have no idea what file_unit_size is used for, so
	   we will flag it for now */
	if (raw_inode->file_unit_size[0] != 0){
		printf("File unit size != 0 for ISO file (%d).\n", ino);
	}

	/* I have no idea what other flag bits are used for, so
	   we will flag it for now */
#ifdef DEBUG
	if ((raw_inode->flags[-high_sierra] & ~2)!= 0){
		printf("Unusual flag settings for ISO file (%d %x).\n",
		       ino, raw_inode->flags[-high_sierra]);
	}
#endif

	inode->i_first_extent = (isonum_733 (raw_inode->extent) + 
				 isonum_711 (raw_inode->ext_attr_length))
					 << sb.s_log_zone_size;

	/* Will be used for previous directory */
	inode->i_backlink = 0xffffffff;
	switch (sb.s_conversion) {
	      case 'a':
		inode->i_file_format = ISOFS_FILE_UNKNOWN; /* File type */
		break;
	      case 'b':
		inode->i_file_format = ISOFS_FILE_BINARY; /* File type */
		break;
	      case 't':
		inode->i_file_format = ISOFS_FILE_TEXT; /* File type */
		break;
	      case 'm':
		inode->i_file_format = ISOFS_FILE_TEXT_M; /* File type */
		break;
	}

#if 0
	/* Now test for possible Rock Ridge extensions which will
	   override some of these numbers in the inode structure. */

	if (!high_sierra)
		parse_rock_ridge_inode(raw_inode, inode);
#endif

	/* keep our inode table correct */
	itp->free = 0;
	itp->inumber = ino;

	/* return a pointer to it */
	return inode;
}


/*
 * ok, we cannot use strncmp, as the name is not in our data space.
 * Thus we'll have to use isofs_match. No big problem. Match also makes
 * some sanity tests.
 *
 * NOTE! unlike strncmp, isofs_match returns 1 for success, 0 for failure.
 */
static int
iso_match (int len, const char *name, const char *compare, int dlen)
{
	if (!compare)
		return 0;

#ifdef DEBUG_ISO
	printf("iso_match: comparing %s with %s\n", name, compare);
#endif

	/* check special "." and ".." files */
	if (dlen == 1) {
		/* "." */
		if (compare[0] == 0) {
			if (!len)
				return 1;
			compare = ".";
		} else if (compare[0] == 1) {
			compare = "..";
			dlen = 2;
		}
	}
	if (dlen != len)
		return 0;
	return !memcmp(name, compare, len);
}


static int
iso_bmap (struct iso_inode *inode, int block)
{
	if (block < 0) {
		printf("iso_bmap: block<0");
		return 0;
	}
	return (inode->i_first_extent >> sb.s_blocksize_bits) + block;
}


/*
 * Find an entry in the specified directory with the wanted name. It
 * returns the cache buffer in which the entry was found, and the entry
 * itself (as an inode number). It does NOT read the inode of the
 * entry - you'll have to do that yourself if you want to.
 */
static int
iso_find_entry (struct iso_inode *dir, const char *name, int namelen,
		unsigned long *ino, unsigned long *ino_back)
{
	unsigned long bufsize = sb.s_blocksize;
	unsigned char bufbits = sb.s_blocksize_bits;
	unsigned int block, f_pos, offset, inode_number;
	void * cpnt = NULL;
	unsigned int old_offset;
	unsigned int backlink;
	int dlen, rrflag, match, i;
	char * dpnt;
	struct iso_directory_record * de;
	char c;
	struct inode_table_entry *itp = (struct inode_table_entry *) dir;

	*ino = 0;
	if (!dir) return -1;
	
	if (!(block = dir->i_first_extent)) return -1;
  
	f_pos = 0;
	offset = f_pos & (bufsize - 1);
	block = iso_bmap(dir,f_pos >> bufbits);
	if (!block) return -1;

	if (iso_breadi(block, 1, data_block) < 0) return -1;
  
	while (f_pos < itp->size) {
		de = (struct iso_directory_record *) (data_block + offset);
		backlink = itp->inumber;
		inode_number = (block << bufbits) + (offset & (bufsize - 1));

		/* If byte is zero, this is the end of file, or time to move to
		   the next sector. Usually 2048 byte boundaries. */
		
		if (*((unsigned char *) de) == 0) {
			offset = 0;
			f_pos = ((f_pos & ~(ISOFS_BLOCK_SIZE - 1))
				 + ISOFS_BLOCK_SIZE);
			block = iso_bmap(dir,f_pos>>bufbits);
			if (!block) return -1;
			if (iso_breadi(block, 1, data_block) < 0) return -1;
  			continue; /* Will kick out if past end of directory */
		}

		old_offset = offset;
		offset += *((unsigned char *) de);
		f_pos += *((unsigned char *) de);

		/* Handle case where the directory entry spans two blocks.
		   Usually 1024 byte boundaries */
		if (offset >= bufsize) {
		        unsigned int frag1;
			frag1 = bufsize - old_offset;
			cpnt = big_data_block;
			memcpy(cpnt, data_block + old_offset, frag1);

			de = (struct iso_directory_record *) cpnt;
			offset = f_pos & (bufsize - 1);
			block = iso_bmap(dir,f_pos>>bufbits);
			if (!block) return -1;
			if (iso_breadi(block, 1, data_block) < 0) return 0;
			memcpy((char *)cpnt+frag1, data_block, offset);
		}
		
		/* Handle the '.' case */
		
		if (de->name[0]==0 && de->name_len[0]==1) {
		  inode_number = itp->inumber;
		  backlink = 0;
		}
		
		/* Handle the '..' case */

		if (de->name[0]==1 && de->name_len[0]==1) {

#if 0
		  printf("Doing .. (%d %d)",
			 sb.s_firstdatazone,
			 itp->i_inumber);
#endif
		  if((int) sb.s_firstdatazone != itp->inumber)
		    inode_number = dir->i_backlink;
		  else
		    inode_number = itp->inumber;
		  backlink = 0;
		}
    
		dlen = de->name_len[0];
		dpnt = de->name;
		/* Now convert the filename in the buffer to lower case */
#if 0
		rrflag = get_rock_ridge_filename(de, &dpnt, &dlen, dir);
#else
		rrflag = 0;
#endif
		if (rrflag) {
		  if (rrflag == -1) return -1; /* Relocated deep directory */
		} else {
		  if(sb.s_mapping == 'n') {
		    for (i = 0; i < dlen; i++) {
		      c = dpnt[i];
		      if (c >= 'A' && c <= 'Z') c |= 0x20;  /* lower case */
		      if (c == ';' && i == dlen-2 && dpnt[i+1] == '1') {
			dlen -= 2;
			break;
		      }
		      if (c == ';') c = '.';
		      de->name[i] = c;
		    }
		    /* This allows us to match with and without a trailing
		       period.  */
		    if(dpnt[dlen-1] == '.' && namelen == dlen-1)
		      dlen--;
		  }
		}
		/*
		 * Skip hidden or associated files unless unhide is set 
		 */
		match = 0;
		if(   !(de->flags[-sb.s_high_sierra] & 5)
		   || sb.s_unhide == 'y' )
		{
		  match = iso_match(namelen,name,dpnt,dlen);
		}

		if (cpnt) cpnt = NULL;

#if 0
		if(rrflag) kfree(dpnt);
#endif
		if (match) {
		  if ((int) inode_number == -1) {
#if 0
		    /* Should only happen for the '..' entry */
		    inode_number = 
		      iso_lookup_grandparent(dir,
					       find_rock_ridge_relocation(de,dir));
		    if(inode_number == -1){
		      /* Should never happen */
		      printf("Backlink not properly set.\n");
		      return -1;
		    }
#else 
		    printf("iso9660: error inode_number = -1\n");
		    return -1;
		  }
#endif
		  *ino = inode_number;
		  *ino_back = backlink;
#ifdef DEBUG_ISO
		  printf("iso_find_entry returning successfully (ino = %d)\n",
			 inode_number);
#endif
		  return 0;
		}
	}
	return -1;
}


/*
 *  Look up name in the current directory and return its corresponding
 *  inode if it can be found.
 */
struct iso_inode *
iso_lookup(struct iso_inode *dir, const char *name)
{
	struct inode_table_entry *itp = (struct inode_table_entry *) dir;
	unsigned long ino, ino_back;
	struct iso_inode *result = NULL;
	int first, last;

#ifdef DEBUG_ISO
	printf("iso_lookup: %s\n", name);
#endif

	/* is the current inode a directory? */
	if (!S_ISDIR(itp->mode)) {
#ifdef DEBUG_ISO
		printf("iso_lookup: inode %d not a directory\n", itp->inumber);
#endif
		iso_iput(dir);
		return NULL;
	}

	/* work through the name finding each directory in turn */
	ino = 0;
	first = last = 0;
	while (last < (int) strlen(name)) {
		if (name[last] == '/') {
			if (iso_find_entry(dir, &name[first], last - first, 
					   &ino, &ino_back)) 
				return NULL;
			/* throw away the old directory inode, we
                           don't need it anymore */
			iso_iput(dir);
			if (!(dir = iso_iget(ino))) 
				return NULL;
			first = last + 1;
			last = first;
		} else
			last++;
	}
	if (iso_find_entry(dir, &name[first], last - first, &ino, &ino_back)) {
		iso_iput(dir);
		return NULL;
	}
	if (!(result = iso_iget(ino))) {
		iso_iput(dir);
		return NULL;
	}
	/*
	 * We need this backlink for the ".." entry unless the name
	 * that we are looking up traversed a mount point (in which
	 * case the inode may not even be on an iso9660 filesystem,
	 * and writing to u.isofs_i would only cause memory
	 * corruption).
	 */
	result->i_backlink = ino_back; 
	
	iso_iput(dir);
	return result;
}


/*
 * look if the driver can tell the multi session redirection value
 */
static inline unsigned int
iso_get_last_session (void)
{
#ifdef DEBUG_ISO 
	printf("iso_get_last_session() called\n");
#endif	
	return 0;
}


int
iso_read_super (void *data, int silent)
{
	static int first_time = 1;
	unsigned int blocksize_bits;
	int high_sierra;
	unsigned int iso_blknum, vol_desc_start;
	char rock = 'y';
	int i;

	struct iso_volume_descriptor *vdp;
	struct hs_volume_descriptor *hdp;

	struct iso_primary_descriptor *pri = NULL;
	struct hs_primary_descriptor *h_pri = NULL;

	struct iso_directory_record *rootp;

	/* Initialize the inode table */
	for (i = 0; i < MAX_OPEN_FILES; i++) {
		inode_table[i].free = 1;
		inode_table[i].inumber = 0;
	}

#ifdef DEBUG_ISO 
	printf("iso_read_super() called\n");
#endif	

	/* set up the block size */
	blocksize_bits = 0;
	{
		int i = 1024;
		while (i != 1){
			blocksize_bits++;
			i >>=1;
		};
	}
	sb.s_blocksize = 1024;
	sb.s_blocksize_bits = blocksize_bits;

	sb.s_high_sierra = high_sierra = 0; /* default is iso9660 */

	vol_desc_start = iso_get_last_session();
	
	for (iso_blknum = vol_desc_start+16; iso_blknum < vol_desc_start+100; 
	     iso_blknum++) {
#ifdef DEBUG_ISO
		printf("iso_read_super: iso_blknum=%d\n", iso_blknum);
#endif
		if (iso_breadi(iso_blknum << (11 - blocksize_bits), 1,
			       data_block) < 0)
		{
			printf("iso_read_super: bread failed, dev "
			       "iso_blknum %d\n", iso_blknum);
			return -1;
		}
		vdp = (struct iso_volume_descriptor *)data_block;
		hdp = (struct hs_volume_descriptor *)data_block;

		if (strncmp (hdp->id, HS_STANDARD_ID, sizeof hdp->id) == 0) {
			if (isonum_711 (hdp->type) != ISO_VD_PRIMARY)
				return -1;
			if (isonum_711 (hdp->type) == ISO_VD_END)
				return -1;

			sb.s_high_sierra = 1;
			high_sierra = 1;
			rock = 'n';
			h_pri = (struct hs_primary_descriptor *)vdp;
			break;
		}

		if (strncmp (vdp->id, ISO_STANDARD_ID, sizeof vdp->id) == 0) {
			if (isonum_711 (vdp->type) != ISO_VD_PRIMARY)
				return -1;
			if (isonum_711 (vdp->type) == ISO_VD_END)
				return -1;
			
			pri = (struct iso_primary_descriptor *)vdp;
			break;
		}
	}
	if(iso_blknum == vol_desc_start + 100) {
		if (!silent)
			printf("iso: Unable to identify CD-ROM format.\n");
		return -1;
	}
	
	if (high_sierra) {
		rootp = (struct iso_directory_record *)
			h_pri->root_directory_record;
		if (isonum_723 (h_pri->volume_set_size) != 1) {
			printf("Multi-volume disks not (yet) supported.\n");
			return -1;
		};
		sb.s_nzones = isonum_733 (h_pri->volume_space_size);
		sb.s_log_zone_size = 
			isonum_723 (h_pri->logical_block_size);
		sb.s_max_size = isonum_733(h_pri->volume_space_size);
	} else {
		rootp = (struct iso_directory_record *)
			pri->root_directory_record;
		if (isonum_723 (pri->volume_set_size) != 1) {
			printf("Multi-volume disks not (yet) supported.\n");
			return -1;
		}
		sb.s_nzones = isonum_733 (pri->volume_space_size);
		sb.s_log_zone_size = isonum_723 (pri->logical_block_size);
		sb.s_max_size = isonum_733(pri->volume_space_size);
	}

	sb.s_ninodes = 0; /* No way to figure this out easily */

	/* RDE: convert log zone size to bit shift */

	switch (sb.s_log_zone_size) {
	      case  512: sb.s_log_zone_size =  9; break;
	      case 1024: sb.s_log_zone_size = 10; break;
	      case 2048: sb.s_log_zone_size = 11; break;

	      default:
		printf("Bad logical zone size %ld\n", sb.s_log_zone_size);
		return -1;
	}

	/* RDE: data zone now byte offset! */

	sb.s_firstdatazone = (isonum_733( rootp->extent) 
					 << sb.s_log_zone_size);
	/*
	 * The CDROM is read-only, has no nodes (devices) on it, and
	 * since all of the files appear to be owned by root, we
	 * really do not want to allow suid.  (suid or devices will
	 * not show up unless we have Rock Ridge extensions).
	 */
	if (first_time) {
		first_time = 0;
		printf("iso: Max size:%ld   Log zone size:%ld\n",
		       sb.s_max_size, 1UL << sb.s_log_zone_size);
		printf("iso: First datazone:%ld   Root inode number %d\n",
		       sb.s_firstdatazone >> sb.s_log_zone_size,
		       isonum_733 (rootp->extent) << sb.s_log_zone_size);
		if (high_sierra)
			printf("iso: Disc in High Sierra format.\n");
	}

	/* set up enough so that it can read an inode */

	sb.s_mapping = 'n';
	sb.s_rock = (rock == 'y' ? 1 : 0);
	sb.s_conversion = 'b';
	sb.s_cruft = 'n';
	sb.s_unhide = 'n';
	/*
	 * It would be incredibly stupid to allow people to mark every file 
	 * on the disk as suid, so we merely allow them to set the default 
	 * permissions.
	 */
	sb.s_mode = S_IRUGO & 0777;
	sb.s_blocksize = 1024;
	sb.s_blocksize_bits = blocksize_bits;

	/* return successfully */
	root_inode = isonum_733 (rootp->extent) << sb.s_log_zone_size;
	return 0;
}


int
iso_bread (int fd, long blkno, long nblks, char * buffer)
{
	struct iso_inode *inode;
	unsigned int block;

	/* find the inode for this file */
	inode = &inode_table[fd].inode;

	/* do some error checking */
	if (!inode || !inode->i_first_extent)
		return 0;

	/* figure out which iso block number(s) we're being asked for */
	block = iso_bmap(inode, blkno);
	if (!block)
		return 0;

	/* now try and read them... */
	if (iso_breadi(block, nblks, buffer) < 0)
		return -1;

	return nblks * sb.s_blocksize;
}


int
iso_open (const char *filename)
{
	struct iso_inode *result;
	struct iso_inode *root;

	/* get the root directory */
	root = iso_iget(root_inode);
	if (!root) {
		printf("iso9660: get root inode failed\n");
		return -1;
	}

	/* lookup the file */
	result = iso_lookup(root, filename);
	if (result == NULL) 
		return -1;
	else {
		struct inode_table_entry * itp =
			(struct inode_table_entry *) result;
		return (itp - inode_table);
	}
}


void
iso_close (int fd)
{
	iso_iput(&inode_table[fd].inode);
}


long
iso_map (int fd, long block)
{
	return iso_bmap(&inode_table[fd].inode, block) * sb.s_blocksize;
}


size_t
iso_fsize (int fd)
{
	return inode_table[fd].size;
}
