// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: x11xpm.cc,v 1.1 1998/07/13 05:06:47 jgg Exp $
/* ######################################################################

   XWindowsGC - XPM Drawing functions for X

   XPM's are handled much like text elements. You can draw an XPM with
   it's background colour filled with the GC's background or you can draw
   it transparent. Unfortunately it is required that the X server have a 
   copy of the XPM (in PixMap format) before it can be rendered. It is
   necessary to cache PixMap's that have had their background colours 
   generated to advoid excessive server IO.
   
   This is done with a linked list attached to each pixmap.
   
   ##################################################################### */
									/*}}}*/
// Include Files							/*{{{*/
#include <deity/x11dev.h>
#include <deity/xpm.h>
#include <X11/xpm.h>
#include <stdlib.h>
#include <iostream.h>
									/*}}}*/

// XPMPreload Class							/*{{{*/
// ---------------------------------------------------------------------
/* */
class XWindowsGC::XPMPreload : public XPMImage::PreloadData
{
   public:
   
   // Various data thingies
   XpmImage Image;              // This is a X11/xpm.h structure
   XpmInfo Info;
   XWindowsGC *Gc;
   
   XPMPreload(XPMImage &Src);
   virtual ~XPMPreload();
};
									/*}}}*/
// XPMPreload::XPMPreload - Constructor					/*{{{*/
// ---------------------------------------------------------------------
/* */
XWindowsGC::XPMPreload::XPMPreload(XPMImage &Src)
{
   memset(&Info,0,sizeof(Info));
   memset(&Image,0,sizeof(Image));
   if (Src.IsFile() == true)
   {
      int Res = XpmReadFileToXpmImage((char *)Src.FileName().c_str(),
				      &Image,&Info);
      if (Res != 0)
	 clog << "Error " << Res << " loading XPMImage from " << Src.FileName() << endl;
   }
   else
   {
      int Res = XpmCreateXpmImageFromData((char **)Src.XPMData(),&Image,&Info);
      if (Res != 0)
	 clog << "Error " << Res << " loading internal XPMImage" << endl;
   }
   Dim = Point(Image.width,Image.height);
}
									/*}}}*/
// XPMPreload::~XPMPreload - Destructor					/*{{{*/
// ---------------------------------------------------------------------
/* Dump the loaded structures */
XWindowsGC::XPMPreload::~XPMPreload()
{
   XpmFreeXpmImage(&Image);
   XpmFreeXpmInfo(&Info);
}
									/*}}}*/

// XPMData Class							/*{{{*/
// ---------------------------------------------------------------------
/* */
class XWindowsGC::XPMData : public XPMImage::Data
{
   static int MyColorAlloc(Display *,Colormap,char *Name,XColor *Clr,void *Me);
   static int MyColorFree(Display *,Colormap,Pixel *,int,void *) {return 1;};
   
   public:
   
   // Various data thingies
   Pixmap Pix;
   XpmAttributes *Attr;
   
   // Source xpm and background colour of Pix
   XPMImage &Src;
   Pixel Background;
   XWindowsGC *Gc;
   
   // Next link from src
   XPMData *Next;
       
   XPMData(XPMImage &Src,Pixel Background,XWindowsGC *Gc);
   virtual ~XPMData();
};
									/*}}}*/
// XPMData::MyColorAlloc - Custom colour allocator			/*{{{*/
// ---------------------------------------------------------------------
/* This uses the matching function the GC to provide colour translation
   this takes advantage of our much better colour handling */
int XWindowsGC::XPMData::MyColorAlloc(Display *,Colormap Map,char *Name,
				      XColor *Clr,void *Me)
{
   XPMData &This = *(XPMData *)Me;
   XColor Jnk;
   XLookupColor(*This.Gc,Map,Name,Clr,&Jnk);
   Clr->pixel = This.Gc->MatchColor(Color(Clr->red>>8,Clr->green>>8,Clr->blue>>8,Color::Blue));
   return 1;
}
									/*}}}*/
// XPMData::XPMData - Constructor					/*{{{*/
// ---------------------------------------------------------------------
/* Any colors that have the symbol name of Background are translated into
   the current background colour. To do this make sure a line like:
     b c none s Background
   Is present. (Add 's Background' to the end of the clear color) */
XWindowsGC::XPMData::XPMData(XPMImage &Src,Pixel Background,XWindowsGC *Gc) :
            Src(Src), Background(Background), Gc(Gc)
{
   // Allocate the attributes structure
   Attr = (XpmAttributes *)calloc(XpmAttributesSize(),1);
 
   /* Setup background colour symbol translation.. Now that we have preloading
      this can be generated directly by editing the ximage structure.. */
   Attr->colorsymbols = new XpmColorSymbol;
   Attr->colorsymbols->name = "Background";
   Attr->colorsymbols->value = 0;
   Attr->colorsymbols->pixel = Background;
   Attr->numsymbols = 1;
   Attr->colormap = Gc->PrivColors;
   Attr->alloc_color = MyColorAlloc;
   Attr->free_colors = MyColorFree;
   Attr->color_closure = this;
   Attr->valuemask = XpmColorSymbols | XpmColormap | XpmAllocCloseColors | 
                     XpmAllocColor | XpmFreeColors | XpmColorClosure;

   // Load the XPM
   Pix = 0;
   XPMPreload *Pre = (XPMPreload *)Src.iPreload;
   if (Pre == 0)
      return;
   
   Attr->width = Pre->Dim.x;
   Attr->height = Pre->Dim.y;
   
   int Res = XpmCreatePixmapFromXpmImage(*GC,Gc->CurWin,&Pre->Image,
					 &Pix,0,Attr);
   if (Res != 0)
   {
      clog << "Error " << Res << " creating pixmap from preloaded XPM" << endl;
      return;
   }
}
									/*}}}*/
// XPMData::~XPMData - Destructor					/*{{{*/
// ---------------------------------------------------------------------
/* */
XWindowsGC::XPMData::~XPMData()
{
   delete []Attr->colorsymbols;
   Attr->colorsymbols = 0;
   XpmFreeAttributes(Attr);
   free(Attr);
   
   // Need to free pixmap+colours too
}
									/*}}}*/

// XWindowsGC::GetXPMData - Aquire a pixmap				/*{{{*/
// ---------------------------------------------------------------------
/* This will look for a pixmap with a matching background pixel. If one isn't
   found then it will be created. */
XWindowsGC::XPMData *XWindowsGC::GetXPMData(XPMImage &Image)
{
   // Ready the image
   if (Image.iPreload == 0)
      if (ImagePreload(Image) == false)
	 return 0;
	  
   // Look over Image for a matching color
   XPMData *I = (XPMData *)Image.DrawData;
   for(; I != 0 && I->Background != BackColor; I = I->Next)
      if (I->Pix == 0)
	 return 0;

   // Need a new pixmap
   if (I == 0)
   {
      // Construct the pixmap
      I = new XPMData(Image,BackColor,this);      
      I->Next = (XPMData *)Image.DrawData;
      Image.DrawData = I;
    
      /* Hm, some kind of loader error. We cache the load error so that
         missing pixmaps don't wreck the whole program.. */
      if (I->Pix == 0)
	 return 0;
   }
   return I;
}
									/*}}}*/
// XWindowsGC::ImagePreload - Preload the image data			/*{{{*/
// ---------------------------------------------------------------------
/* */
bool XWindowsGC::ImagePreload(XPMImage &Image)
{
   if (Image.iPreload == 0)
      Image.iPreload = new XPMPreload(Image);
   if (Image.iPreload->Dim == Point(0,0))
      return false;
   return true;
}
									/*}}}*/
// XWindowsGC::DrawBitmap - Draw a bitmap at a specified point		/*{{{*/
// ---------------------------------------------------------------------
/* Draw a bitmap normally, this is just like the similar DrawText routine */
void XWindowsGC::DrawBitmap(Point Pos,XPMImage &Image,unsigned long Flags)
{
   // Grab the pixmap
   XPMData *I = GetXPMData(Image);
   if (I == 0 || I->Pix == 0)
      return;
   
   // Compute the location
   Rect Loc = Rect(Pos.x,Pos.y,I->Attr->width,I->Attr->height);
   FlagPosition(Loc,Flags);
   
   // Draw the pixmap.
   XCopyArea(*this,I->Pix,CurWin,CurGC,0,0,Loc.w,Loc.h,
	     Loc.x+Clip.x,Loc.y+Clip.y);
}
									/*}}}*/
// XWindowsGC::DrawBitmap - Draw a bitmap inside a rectangle		/*{{{*/
// ---------------------------------------------------------------------
/* This, like its DrawText cousin, draws a bitmap inside a clipping 
   rectangle and fills the whole rectangle with the background colour
   in a flicker free manner. */
void XWindowsGC::DrawBitmap(Rect ROuter,Point Pos,XPMImage &Image,
			    unsigned long Flags)
{
   // Grab the pixmap
   XPMData *I = GetXPMData(Image);
   if (I == 0 || I->Pix == 0)
      return;
   
   // Compute the real location
   Rect RInner = Rect(Pos.x,Pos.y,I->Attr->width,I->Attr->height);
   FlagPosition(RInner,Flags);

   // Clip the outer rectangle
   Rect VOuter = ROuter;
   VOuter.Clip(Clip);
   
   // Adjust the inner rectangle and clip it too
   RInner.x += ROuter.x - VOuter.x;
   RInner.y += ROuter.y - VOuter.y;
   Rect VInner = RInner;
   VInner.Clip(VOuter);
   
   // Empty inner rectangle?
   if (VInner.w == 0 || VInner.h == 0)
   {
      BFill(VOuter);
      return;
   }   
       
   // Draw the pixmap.
   XCopyArea(*this,I->Pix,CurWin,CurGC,VInner.x - RInner.x,
	     VInner.y - RInner.y,VInner.w,VInner.h,
	     VInner.x + VOuter.x + Clip.x,VInner.y + VOuter.y + Clip.y);

   MaskFill(VOuter,VInner);
}
									/*}}}*/
// XWindowsGC::DrawBitmapT - Draw a transparent bitmap			/*{{{*/
// ---------------------------------------------------------------------
/* */
void XWindowsGC::DrawBitmapT(Point /*Pos*/,XPMImage &/*Image*/,
			     unsigned long /*Flags*/)
{
}
									/*}}}*/

