// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#ifndef _VerticalRenderer_h_
#define _VerticalRenderer_h_

#include <qlist.h>
#include "SgmlRenderer.h"

class TextRenderer;

//: The VerticalRenderer class manages a list of Renderers that are laid out top to bottom.
//. A VerticalRenderer receives start tag, end tag, and content signals from an
//. SgmlParser, and creates appropriate Renders to display the content.  It
//. also manages their geometry and insures that the proper amount of padding
//. exists betweem each of its children.
class VerticalRenderer : public SgmlRenderer {
    Q_OBJECT
    SgmlRenderer* _sgmlRenderer;
    TextRenderer* _textRenderer;
    QString       _titleString;
    int           _marginLeft;
    int           _marginRight;
    int           _vspace;
    int           _sgmlDepth;

    struct ChildInfo {
        Renderer* renderer;
        int       marginLeft;
        int       marginRight;
        int       marginTop;
        int       marginBottom;
        ChildInfo( Renderer* _renderer, int _marginLeft, int _marginRight, int _marginTop, int _marginBottom );
    };
    
    QList<ChildInfo> _renderers;
public:
    //. Create a VerticalRenderer.
    VerticalRenderer( Canvas*     canvas,
                      SgmlParser* parser,
                      int         clipWidth=0,
                      QObject*    parent=0,
                      const char* name=0 );

    //. This is a virtual destructor.
    virtual ~VerticalRenderer();

    //. Reposition my children so that they do not overlap.  Child widgets are
    //. laid out from top to bottom, with vertical space between them.
    virtual bool redraw();

    bool findAnchor( const QString& name, int& x, int& y );

    void repaint( QPainter& p, const Rect& r );
public slots:
    //. Process the content signal from the SgmlParser.  Forward the content to
    //. the current text renderer.  If there is no current text renderer, create
    //. one based on the current style's whitespace setting (Normal or Pre).
    virtual void content( QString text );

    //. Forward the endOfData signal to the open child renderer, if any.
    virtual void endOfData();

    //. Adjust the current left and right margins.  If there is an open table
    //. renderer, forward the end tag to it.  If the end tag is for the open
    //. table renderer, close the table renderer.  Otherwise, if the end tag is
    //. for an inline element, forward it to the open text renderer, if any.
    //. If the end tag is for a block-level element, close the open text renderer,
    //. if any.
    virtual void endTag();

    //. Adjust the current left and right margins.  If there is an open table
    //. renderer then forward the start tag to it, and return.  Otherwise take
    //. the following actions, based on the display type of the start tag:
    //. <DL>
    //. <DT>Block<DD>Close any open text renderer.
    //. <DT>Inline<DD>Forward the start tag to the open text renderer, if any.
    //. <DT>Table<DD>Create a new table renderer.
    //. <DT>Image<DD>Create a new text renderer, and forward the start tag.
    //. <DT>HLine<DD>Create a new horizontal line renderer.
    //. <DT>Title<DD>Clear the title text buffer.
    //. </DL>
    virtual void startTag();

    //. Notify each of the child renderers of their new preferred widths.
    virtual void widthChanged( int w );
};

#endif
