// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#ifndef _Request_h_
#define _Request_h_

#include <qobject.h>
#include "Url.h"

class Canvas;

//: The Request class provides one-stop-shopping for data retrieval.
//. Any object that needs to download data from a URL in a non-blocking way
//. can use a request.  The request uses the Cache for efficiency.
class Request : public QObject {
    Q_OBJECT
    Url        _url;
    int        _totalSize;
    int        _totalRead;
    Canvas*    _canvas;
protected:
    //. Process a delayed abort request.  Abort the open connection, and die.
    virtual void timerEvent( QTimerEvent* e );
public:
    //. Create a request for the given URL, after parsing it.
    Request( QString url, Canvas* canvas=0 );

    //. Create a request for the given URL.
    Request( const Url& url, Canvas* canvas=0 );

    //. Destroy the request and abort any open connection.
    ~Request();

    //. Open a new connection.
    bool open();
public slots:
    //. Abort the current connection, and delete the Request.  This action is
    //. deferred (using startTimer(0)) in order to give the caller a chance to
    //. clean up first.
    void abort();

    //. Forward the data signals from the connection.
    void fwdData( const char* bytes, int length );

    //. Forward the endOfData signal from the connection.
    void fwdEndOfData();

    //. Forward the startOfData signal from the connection.
    void fwdStartOfData( QString mediaType, QString mediaSubtype, int totalSize );

    //. Forward the status signal from the connection.
    void fwdStatus( QString msg );
    
    //. Forward the urlChanged signal from the connection.
    void fwdUrlChanged( const Url& url );
signals:
    //. This signal is used to forward the data signals from the connection.
    void data( const char* bytes, int length );

    //. This signal is used to forward the endOfData signal from the
    //. connection.
    void endOfData();

    //. This signal is used to forward the startOfData signal from the
    //. connection.
    void startOfData( Request* req, QString mediaType, QString mediaSubtype, int totalSize );

    //. This signal is emitted whenever the connection has something
    //. interesting to put in the status line of the browser.
    void status( QString msg );

    //. This signal is emitted when the Request is informed that the URL for
    //. its connection has changed.
    void urlChanged( const Url& url );

    //. This signal is emitted just before the Request deletes itself.
    void done( Request* req );
};

#endif
