// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#include <qmsgbox.h>
#include <qobject.h>
#include "SgmlCatalog.h"
#include "DtdManager.h"
#include "DtdParser.h"
#include "Options.h"

DtdManager::DtdManager()
{
//    _dtds.setAutoDelete( TRUE );
}

DtdManager::~DtdManager()
{
    while ( _dtds.first() ) {
	delete _dtds.first();
	_dtds.remove();
    }
    while ( _parsers.first() ) {
	delete _parsers.first();
	_parsers.remove();
    }
    while ( _requests.first() ) {
	delete _requests.first();
	_requests.remove();
    }
}

Dtd* DtdManager::findDtd( const QString& soi )
{
    QString tmp = "file:";
    tmp += options->catalogFile();
    Url url( tmp );
    Url dtdUrl( &url, soi );
    QListIterator<Dtd> i( _dtds );
    for ( i.toFirst(); i.current(); ++i ) {
	if ( i.current()->soi() == dtdUrl.url() ) {
	    return i.current();
	}
    }

    return 0;
}

void DtdManager::request( QObject* parser, QString doctype, QString id )
{
    Dtd* dtd;

    // Look for the public ID in the catalog, to get the SOI.
    CatalogEntry* e = sgmlCatalog->find( CatalogEntry::Doctype, id );
    if ( !e ) {
        // Try doctype.
        if ( !( e = sgmlCatalog->find( CatalogEntry::Doctype, doctype ) ) ) {
            QString error;
            error.sprintf( "Cannot resolve DTD reference '%s'.\nPlease check the entry in the catalog.", id.data() );
            QMessageBox::message( "QWeb: Error", error );
            return;
        }
    }
    
    if ( ( dtd = findDtd( e->soi() ) ) ) {
        connect( this, SIGNAL( dtdReady( Dtd* ) ), parser, SLOT( dtd( Dtd* ) ) );
        emit dtdReady( dtd );
        disconnect( parser );
        return;
    }
	
    // Create a new request.
    Parser* par = new Parser( parser, e->soi() );
	
    // Has the same DTD already been requested?
    for ( DtdRequest* r = _requests.first(); r; r = _requests.next() ) {
        if ( r->soi == e->soi() ) {
            _parsers.append( par );
            return;
        }
    }
	
    // Search the catalog for the dtd.
    dtd = new Dtd;
    par->dtd = dtd;
    DtdParser* dtdParser = new DtdParser( dtd );
    connect( dtdParser, SIGNAL( done( Dtd* ) ), this, SLOT( done( Dtd* ) ) );

    // Resolve the DTD URL relative to the location of the catalog.
    QString tmp = "file:";
    tmp += options->catalogFile();
    Url url( tmp );
    Url dtdUrl( &url, e->soi() );
    dtd->soi( dtdUrl.url() );

    // Issue request for the DTD.
    Request* req = new Request( dtdUrl );
    connect( req, SIGNAL( data( const char*, int ) ), dtdParser  , SLOT( fwdData( const char*, int ) ) );
    connect( req, SIGNAL( endOfData() )             , dtdParser  , SLOT( fwdEndOfData() ) );

    if ( req->open() ) {
        _requests.append( new DtdRequest( req, dtd, e->soi() ) );
        _parsers.append( par );
    } else {
        // Cannot open connection.
        delete req;
        delete dtdParser;
        delete dtd;
        delete par;
            
        QString error;
        error.sprintf( "Cannot open DTD '%s' for id '%s'.\nPlease check the entry in the catalog.", dtdUrl.url().data(), id.data() );
        QMessageBox::message( "QWeb: Error", error );
            
        connect( this, SIGNAL( dtdReady( Dtd* ) ), parser, SLOT( dtd( Dtd* ) ) );
        emit dtdReady( 0 );
        disconnect( parser );
    }
}

void DtdManager::done( Dtd* dtd )
{
    for ( DtdRequest* r = _requests.first(); r; r = _requests.next() ) {
        if ( r->dtd == dtd ) {
            _dtds.append( dtd );
            _requests.remove( r );
            delete r;
        }
    }
	
    for ( Parser* p = _parsers.first(); p; ) {
        if ( p->dtd == dtd ) {
            connect( this, SIGNAL( dtdReady( Dtd* ) ), p->parser, SLOT( dtd( Dtd* ) ) );
            emit dtdReady( dtd );
            disconnect( p->parser );
            _parsers.remove();
            delete p;
            p = _parsers.current();
        } else {
            p = _parsers.next();
        }
    }
}

void DtdManager::clear()
{
    while ( _dtds.first() ) {
	delete _dtds.first();
	_dtds.remove();
    }
}

Dtd* DtdManager::find( QString dtdName )
{
    // Look for the public ID in the catalog, to get the SOI.
    CatalogEntry* e = sgmlCatalog->find( CatalogEntry::Doctype, dtdName );
    if ( !e ) {
        QString error;
        error.sprintf( "Cannot resolve DTD reference '%s'.\nPlease check the entry in the catalog.", dtdName.data() );
        QMessageBox::message( "QWeb: Error", error );
        return 0;
    }

    return findDtd( e->soi() );
}
