/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@medizin.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: action.c,v 1.4 94/07/17 16:03:42 nau Exp $";

/* action routines
 */

#include "global.h"

#include "action.h"
#include "command.h"
#include "create.h"
#include "cursor.h"
#include "data.h"
#include "dialog.h"
#include "draw.h"
#include "error.h"
#include "find.h"
#include "memory.h"
#include "misc.h"
#include "pinout.h"
#include "remove.h"
#include "search.h"
#include "transform.h"

/* ---------------------------------------------------------------------------
 * some local defines
 */
#define	TAN_30_DEGREE	0.577350269		/* tan(30) */
#define	TAN_60_DEGREE	1.732050808		/* tan(60) */

#define	F_SET_GRID							1
#define	F_SET_ZOOM							2
#define	F_SET_LINESIZE						3
#define	F_SET_VIASIZE						4
#define	F_SET_PINSIZE						5
#define	F_PASTE								6
#define	F_LINESTACK							7
#define	F_VIA								8
#define	F_LINE								9
#define	F_RECTANGLE							10
#define	F_ELEMENT							11
#define	F_PIN								12
#define	F_TEXT								13
#define	F_TOP_OF_LINESTACK					14
#define	F_BLOCK								15
#define	F_VIAS_IN_BLOCK						16
#define	F_PINS_IN_BLOCK						17
#define	F_MOVE								18
#define	F_SETORIGIN							19
#define	F_ROTATE							20
#define	F_MIRROR							21
#define	F_FIND								22
#define	F_RESET_FOUND_LINES_AND_RECTANGLES	23
#define	F_RESET_FOUND_PINS_AND_VIAS			24
#define	F_RESET								25
#define	F_DISPLAY_CANONICAL_NAME			26
#define	F_DISPLAY_NAME_ON_PCB				27
#define	F_ENABLE							28
#define	F_DISABLE							29
#define	F_LINES_IN_BLOCK					30
#define	F_NOTIFY_CREATE						31
#define	F_ALL_IN_BLOCK						32
#define	F_ABSOLUTE							33
#define	F_RELATIVE							34

/* ---------------------------------------------------------------------------
 * some local types
 */
typedef struct					/* used to identify subfunctions */
{
	char	*Identifier;
	int		ID;
} FunctionType, *FunctionTypePtr;

/* ---------------------------------------------------------------------------
 * some local identifiers
 */
static	Boolean					IgnoreMoveEvents = 0;
static	FunctionType			Functions[] = {
	{ "Grid", F_SET_GRID },
	{ "Zoom", F_SET_ZOOM },
	{ "LineSize", F_SET_LINESIZE },
	{ "ViaSize", F_SET_VIASIZE },
	{ "PinSize", F_SET_PINSIZE },
	{ "Via", F_VIA },
	{ "Line", F_LINE },
	{ "Rectangle", F_RECTANGLE },
	{ "Element", F_ELEMENT },
	{ "Pin", F_PIN },
	{ "Text", F_TEXT },
	{ "TopOfLineStack", F_TOP_OF_LINESTACK },
	{ "PinsInBlock", F_PINS_IN_BLOCK },
	{ "ViasInBlock", F_VIAS_IN_BLOCK },
	{ "LinesInBlock", F_LINES_IN_BLOCK },
	{ "Move", F_MOVE },
	{ "SetOrigin", F_SETORIGIN },
	{ "Rotate", F_ROTATE },
	{ "Mirror", F_MIRROR },
	{ "Find", F_FIND },
	{ "ResetLinesAndRectangles", F_RESET_FOUND_LINES_AND_RECTANGLES },
	{ "ResetPinsAndVias", F_RESET_FOUND_PINS_AND_VIAS },
	{ "Reset", F_RESET },
	{ "DisplayCanonicalName", F_DISPLAY_CANONICAL_NAME },
	{ "DisplayNameOnPCB", F_DISPLAY_NAME_ON_PCB },
	{ "Enable", F_ENABLE },
	{ "Disable", F_DISABLE },
	{ "Paste", F_PASTE },
	{ "LineStack", F_LINESTACK },
	{ "Block", F_BLOCK },
	{ "AllInBlock", F_ALL_IN_BLOCK },
	{ "Absolute", F_ABSOLUTE },
	{ "Relative", F_RELATIVE },
	{ "NotifyCreate", F_NOTIFY_CREATE }};

/* ---------------------------------------------------------------------------
 * some local routines
 */
static	int		GetFunctionID(String);
static	void	NotifyLine(void);
static	void	NotifyBlock(void);
static	void	CreateObject(int);

/* ---------------------------------------------------------------------------
 * get function ID of passed string
 */
static int GetFunctionID(String Ident)
{
	int		i;

	for (i = ENTRIES(Functions) -1; i != -1; i--)
		if (!strcmp(Ident, Functions[i].Identifier))
			return(Functions[i].ID);
	return(-1);
}

/* ---------------------------------------------------------------------------
 * makes the 'marked line' fit into a 45 degree direction
 * routine is called for every new cursor coordinate by ActionMoveCursor()
 *
 * directions:
 *
 *           0
 *          7 1
 *         6   2
 *          5 3
 *           4
 */
static void NotifyLine(void)
{
	Position	dx, dy, min;
	float		m;

		/* line input mode must be switched on */
	if (MyCursor.Mode != MODE_LINE ||
		MyCursor.MarkedLine.State == MARKED_STATE_RESET)
		return;

		/* first calculate direction of line */
	dx = MyCursor.X -MyCursor.MarkedLine.X1;
	dy = MyCursor.Y -MyCursor.MarkedLine.Y1;
	if (!dx)
	{
		if (!dy)
		{
				/* zero length line, don't draw anything */
			MyCursor.MarkedLine.Direction = 0;
			return;
		}
		else
			MyCursor.MarkedLine.Direction = dy > 0 ? 0 : 4;
	}
	else
	{
		m = (float) dy / (float) dx;
		MyCursor.MarkedLine.Direction = 2;
		if (m > TAN_30_DEGREE)
			MyCursor.MarkedLine.Direction = m > TAN_60_DEGREE ? 0 : 1;
		else
			if (m < -TAN_30_DEGREE)
				MyCursor.MarkedLine.Direction = m < -TAN_60_DEGREE ? 0 : 3;
	}
	if (dx < 0)
		MyCursor.MarkedLine.Direction += 4;

	dx = abs(dx);
	dy = abs(dy);
	min = (dy < dx) ? dy : dx;

		/* now set up the second pair of coordinates */
	switch (MyCursor.MarkedLine.Direction)
	{
		case 0:
		case 4:
			MyCursor.MarkedLine.X2 = MyCursor.MarkedLine.X1;
			MyCursor.MarkedLine.Y2 = MyCursor.Y;
			break;

		case 2:
		case 6:
			MyCursor.MarkedLine.X2 = MyCursor.X;
			MyCursor.MarkedLine.Y2 = MyCursor.MarkedLine.Y1;
			break;

		case 1:
			MyCursor.MarkedLine.X2 = MyCursor.MarkedLine.X1 +min;
			MyCursor.MarkedLine.Y2 = MyCursor.MarkedLine.Y1 +min;
			break;

		case 3:
			MyCursor.MarkedLine.X2 = MyCursor.MarkedLine.X1 +min;
			MyCursor.MarkedLine.Y2 = MyCursor.MarkedLine.Y1 -min;
			break;

		case 5:
			MyCursor.MarkedLine.X2 = MyCursor.MarkedLine.X1 -min;
			MyCursor.MarkedLine.Y2 = MyCursor.MarkedLine.Y1 -min;
			break;

		case 7:
			MyCursor.MarkedLine.X2 = MyCursor.MarkedLine.X1 -min;
			MyCursor.MarkedLine.Y2 = MyCursor.MarkedLine.Y1 +min;
			break;
	}
}

/* ---------------------------------------------------------------------------
 * sets new coordinates for marked block
 * routine is called for every new cursor coordinate by ActionMoveCursor()
 */
static void NotifyBlock(void)
{
	if (MyCursor.Mode == MODE_BLOCK && 
		MyCursor.MarkedBlock.State == MARKED_STATE_STARTED)
	{
		MyCursor.MarkedBlock.X2 = MyCursor.X;
		MyCursor.MarkedBlock.Y2 = MyCursor.Y;
	}	
}

/* ---------------------------------------------------------------------------
 * routine to create vias, lines ...
 */
static void CreateObject(int Type)
{
	Position		x1, y1,
					x2, y2;
	ElementTypePtr	element;
	LineTypePtr		line;
	char			*string;
	Cardinal		n;

	HideCursor();
	switch(Type)
	{
		case F_VIA:
				/* create a new via only if there isn't one yet */
			if (!SearchVia(MyCursor.X, MyCursor.Y) && PCB->ViaOn)
			{
				DrawNewVia(CreateNewVia(PCB, MyCursor.X, MyCursor.Y,
					Settings.ViaThickness, NULL, VIAFLAG));
				SetChangedFlag(True);
			}	
			break;

		case F_LINE:
			if (MyCursor.Mode != MODE_LINE)
				break;
			if (MyCursor.MarkedLine.State == MARKED_STATE_RESET)
			{
					/* first point of polygon */
				MyCursor.MarkedLine.State = MARKED_STATE_STARTED;
				MyCursor.MarkedLine.X1 = MyCursor.MarkedLine.X2 = MyCursor.X;
				MyCursor.MarkedLine.Y1 = MyCursor.MarkedLine.Y2 = MyCursor.Y;
			}
			else
			{
				line = CreateNewLineOnLayer(PCB, CURRENT,
					MyCursor.MarkedLine.X1, MyCursor.MarkedLine.Y1,
					MyCursor.MarkedLine.X2, MyCursor.MarkedLine.Y2,
					Settings.LineThickness, NOFLAG);
				DrawLineOnLayer(line, CURRENT);
				if (MyCursor.LineStack.On)
					AddLineToStack(line);
				SetChangedFlag(True);

					/* swap the coordinates */
				MyCursor.MarkedLine.X1 = MyCursor.MarkedLine.X2;
				MyCursor.MarkedLine.Y1 = MyCursor.MarkedLine.Y2;
			}
			break;

		case F_BLOCK:
			if (MyCursor.Mode == MODE_BLOCK)
				switch(MyCursor.MarkedBlock.State)
				{
					case MARKED_STATE_RESET:		/* setup first point */
						MyCursor.MarkedBlock.X1 = MyCursor.MarkedBlock.X2 = MyCursor.X;
						MyCursor.MarkedBlock.Y1 = MyCursor.MarkedBlock.Y2 = MyCursor.Y;
						MyCursor.MarkedBlock.State = MARKED_STATE_STARTED;
						break;

					case MARKED_STATE_STARTED:		/* setup second point */
						MyCursor.MarkedBlock.State = MARKED_STATE_ENDED;
						MyCursor.MarkedBlock.X = MIN(MyCursor.MarkedBlock.X1, MyCursor.MarkedBlock.X2);
						MyCursor.MarkedBlock.Y = MIN(MyCursor.MarkedBlock.Y1, MyCursor.MarkedBlock.Y2);
						break;
				}
			break;

		case F_RECTANGLE:
			if (MyCursor.Mode == MODE_BLOCK &&
				MyCursor.MarkedBlock.State == MARKED_STATE_ENDED)
			{
					/* end block mode */
				MyCursor.Mode = MODE_NONE;
				x1 = MIN(MyCursor.MarkedBlock.X1, MyCursor.MarkedBlock.X2);
				y1 = MIN(MyCursor.MarkedBlock.Y1, MyCursor.MarkedBlock.Y2);
				x2 = MAX(MyCursor.MarkedBlock.X1, MyCursor.MarkedBlock.X2);
				y2 = MAX(MyCursor.MarkedBlock.Y1, MyCursor.MarkedBlock.Y2);
				DrawRectOnLayer(CreateNewRect(PCB, CURRENT, x1, y1,
					x2-x1, y2-y1, NOFLAG), CURRENT);
				SetChangedFlag(True);
			}
			break;

		case F_ELEMENT:
			if (MyCursor.Mode == MODE_ELEMENT &&
				MyCursor.MarkedElement.Valid &&
				(PCB->ElementOn || PCB->PinOn))
			{
					/* copy element data, set surrounding rectangle
				 	* and move it to the current pointer location
				 	*/
				element = CopyElement(PCB, &MyCursor.MarkedElement.Element, NULL);
				MoveElement(element, MyCursor.X, MyCursor.Y);
				SetElementInfo(element, 1);
				DrawNewElement(element);
				SetChangedFlag(True);
			}
			break;

		case F_TEXT:
			if ((string = GetUserInput(Output.Toplevel, False, "Enter text:", NULL)) != NULL)
			{
				DrawTextOnLayer(CreateNewText(PCB, CURRENT,
					MyCursor.X, MyCursor.Y, 0, string, NOFLAG), CURRENT);

					/* free memory allocated by GetUserInput() */
				SaveFree(string);
				SetChangedFlag(True);
			}
			break;

		case F_PASTE:	/* paste line stack buffer */
			if (MyCursor.Mode == MODE_LINESTACK)
				for (n = MyCursor.LineStack.LineN, line = MyCursor.LineStack.Line; n; n--, line++)
					DrawLineOnLayer(CreateNewLineOnLayer(PCB, CURRENT,
						MyCursor.X -MyCursor.LineStack.OriginX +line->X1,
						MyCursor.Y -MyCursor.LineStack.OriginY +line->Y1,
						MyCursor.X -MyCursor.LineStack.OriginX +line->X2,
						MyCursor.Y -MyCursor.LineStack.OriginY +line->Y2,
						line->Thickness, NOFLAG), CURRENT);
			break;
	}
	RestoreCursor();
}

/* ---------------------------------------------------------------------------
 * action routine to move to X pointer relative to the current position
 * syntax: MovePointer(deltax,deltay)
 */
void ActionMovePointer(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	Position	old_x, old_y;

	if (*Num != 2)
		return;

	HideCursor();

		/* save old pointer position, move to new one and send
		 * pointer move request to server
		 */
	old_x = TO_SCREEN(MyCursor.X);
	old_y = TO_SCREEN(MyCursor.Y);
	MoveCursorRelative(atoi(*Params) *PCB->Grid, atoi(*(Params+1)) *PCB->Grid);
	old_x = TO_SCREEN(MyCursor.X) -old_x;
	old_y = TO_SCREEN(MyCursor.Y) -old_y;
	XWarpPointer(Dpy, Output.OutputWindow, None, 0, 0, 0, 0, old_x, old_y);

		/* XWarpPointer creates Motion events normally bound to
		 * ActionSetCursor. We don't do any updates when ActionSetCursor
		 * is called the next time to prevent from rounding errors
		 */
	IgnoreMoveEvents = True;

		/* update marked line and block coordinates */
	NotifyBlock();
	NotifyLine();
	SetCursorLine();
	RestoreCursor();
}

/* ---------------------------------------------------------------------------
 * action routine to set the cursor according to the X pointer position
 * syntax: SetCursor()
 */
void ActionSetCursor(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	int	x, y;

	if (*Num != 0 || PointerPosition(Output.OutputWindow, &x, &y))
		return;

		/* ignore events that are probably caused by ActionMovePointer */
	if (IgnoreMoveEvents)
	{
		IgnoreMoveEvents = False;
		return;
	}

	HideCursor();

		/* correct the values with zoom factor */
	MoveCursorAbsolute(TO_PCB(x) +PCB->Grid/2, TO_PCB(y) +PCB->Grid/2);

		/* update marked line and block coordinates */
	NotifyBlock();
	NotifyLine();
	SetCursorLine();
	RestoreCursor();
}

/* ---------------------------------------------------------------------------
 * action routine to change the grid, zoom and sizes
 * first parameter identifies the operation,
 * the second the type of object and the third is passed to
 * the specific routine
 * syntax: SetSetting(Absolute|Relative, Grid|Zoom|LineSize|ViaSize, value)
 */
void ActionSetSetting(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	Boolean	rel;
	int		value;

	if (*Num != 3)
		return;

	rel = (GetFunctionID(*Params) == F_RELATIVE);
	value = atoi(*(Params+2));
	switch(GetFunctionID(*(Params+1)))
	{
		case F_SET_GRID:
			SetGrid(rel ? value +PCB->Grid : value);
			break;

		case F_SET_ZOOM:
			SetZoom(rel ? value +PCB->Zoom : value);
			break;

		case F_SET_LINESIZE:
			SetLineSize(rel ? value +Settings.LineThickness : value);
			break;

		case F_SET_VIASIZE:
			SetViaSize(rel ? value +Settings.ViaThickness : value);
			break;
	}
}

/* ---------------------------------------------------------------------------
 * action routine to create objects
 * syntax: Create(Line|Block|Via|Text|Element|Rectangle)
 */
void ActionCreate(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	int		type;

	if (*Num == 1)
	{
		HideCursor();
		SwitchDrawingWindow(PCB->Zoom, Output.OutputWindow);
		switch(type = GetFunctionID(*Params))
		{
				/* create a new block */
			case F_BLOCK:

				/* create all 'normal drawable' objects */
			case F_VIA:
			case F_TEXT:
			case F_ELEMENT:
			case F_RECTANGLE:
			case F_LINE:
				CreateObject(type);
				break;
		}
		RestoreCursor();
	}
}

/* ---------------------------------------------------------------------------
 * action routine to remove vias, lines ...
 * its necessary to check the state of the blockmode for some functions
 * syntax: Remove(Line|Via|Text|Element|Rectangle|AllInBlock|TopOfLineStack)
 */
void ActionRemove(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	LayerTypePtr	layer;
	LineTypePtr		line;
	RectTypePtr		rect;
	TextTypePtr		text;

	HideCursor();
	SwitchDrawingWindow(PCB->Zoom, Output.OutputWindow);
	switch(GetFunctionID(*Params))
	{
		case F_VIA:
			if (RemoveVia(SearchVia(MyCursor.X, MyCursor.Y)))
				SetChangedFlag(True);
			break;

		case F_LINE:
			if (SearchLine(&layer, &line, MyCursor.X, MyCursor.Y))
			{
				RemoveLine(layer, line);
				SetChangedFlag(True);
			}
			break;

		case F_RECTANGLE:
			if (SearchRect(&layer, &rect, MyCursor.X, MyCursor.Y))
			{
				RemoveRect(layer, rect);
				SetChangedFlag(True);
			}
			break;

		case F_TEXT:
			if (SearchText(&layer, &text, MyCursor.X, MyCursor.Y))
			{
				RemoveText(layer, text);
				SetChangedFlag(True);
			}
			break;

		case F_ELEMENT:
			if(RemoveElement(SearchElement(MyCursor.X, MyCursor.Y)))
				SetChangedFlag(True);
			break;

		case F_ALL_IN_BLOCK:
			if (MyCursor.Mode == MODE_BLOCK &&
				MyCursor.MarkedBlock.State == MARKED_STATE_ENDED &&
				RemoveAllInRectangle(MyCursor.MarkedBlock.X1, MyCursor.MarkedBlock.Y1,
				MyCursor.MarkedBlock.X2, MyCursor.MarkedBlock.Y2))
			{
				SetChangedFlag(True);
				RedrawOutput();
			}
			MyCursor.Mode = MODE_NONE;
			SetStatusLine();
			break;

		case F_TOP_OF_LINESTACK:
			if (MyCursor.LineStack.LineN)
				MyCursor.LineStack.LineN--;
			SetStatusLine();
			break;
	}
	RestoreCursor();
}

/* ---------------------------------------------------------------------------
 * action routine to set a new mode or to create/delete objects 
 * according to the currently selected mode
 * syntax: Mode(Line|Block|Element|LineStack|ResetNotify|Create)
 */
void ActionMode(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	if (*Num != 1)
		return;

	switch(GetFunctionID(*Params))
	{
		case F_LINE:		SetMode(MODE_LINE); break;
		case F_BLOCK:		SetMode(MODE_BLOCK); break;
		case F_ELEMENT:		SetMode(MODE_ELEMENT); break;
		case F_LINESTACK:	SetMode(MODE_LINESTACK); break;
		case F_RESET:		SetMode(MODE_NONE); break;

		case F_NOTIFY_CREATE:		/* create object depending on mode */
			switch(MyCursor.Mode)
			{
				case MODE_LINE:			CreateObject(F_LINE); break;
				case MODE_BLOCK:		CreateObject(F_BLOCK); break;
				case MODE_ELEMENT:		CreateObject(F_ELEMENT); break;
				case MODE_LINESTACK:	CreateObject(F_PASTE); break;
			}
			break;
	}
}

/* ---------------------------------------------------------------------------
 * misc operations on objects in marked block
 * syntax: Block(Move, n), Block(SetOrigin)
 */
void ActionBlock(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	if (MyCursor.Mode != MODE_BLOCK ||
		MyCursor.MarkedBlock.State != MARKED_STATE_ENDED)
		return;

	HideCursor();
	SwitchDrawingWindow(PCB->Zoom, Output.OutputWindow);
	switch(GetFunctionID(*Params))
	{
		case F_MOVE:
				/* move marked block by
				 * (MyCursorMarkedBlock.X -x, MyCursorMarkedBlock.Y -y)
				 * followed by a rotation with center (MyCursor.X, MyCursor.Y)
				 */
			if (*Num == 2 &&
				MoveAndRotateAllInRectangle(
				MyCursor.MarkedBlock.X1, MyCursor.MarkedBlock.Y1,
				MyCursor.MarkedBlock.X2, MyCursor.MarkedBlock.Y2,
				MyCursor.X -MyCursor.MarkedBlock.X, MyCursor.Y -MyCursor.MarkedBlock.Y,
				MyCursor.X, MyCursor.Y, atoi(*(Params+1))))
			{
				SetChangedFlag(True);
				RedrawOutput();
			}
			MyCursor.Mode = MODE_NONE;
			SetStatusLine();
			break;

		case F_SETORIGIN:	/* sets a new origin for the marked block */
			if (*Num == 1)
			{
				MyCursor.MarkedBlock.X = MyCursor.X;
				MyCursor.MarkedBlock.Y = MyCursor.Y;
			}
			break;
	}
	RestoreCursor();
}

/* ---------------------------------------------------------------------------
 * rotate loaded element
 * syntax: RotateElement()
 */
void ActionRotateElement(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	if (MyCursor.Mode == MODE_ELEMENT &&
		MyCursor.MarkedElement.Valid &&
		*Num == 0)
	{
		HideCursor();
		SwitchDrawingWindow(PCB->Zoom, Output.OutputWindow);

			/* the element has a virtual origin of (0,0)
			 * so this is the fixpoint for our rotation
			 */
		RotateElement(&MyCursor.MarkedElement.Element, 0, 0, 1);
		RestoreCursor();
	}
}

/* ---------------------------------------------------------------------------
 * changes the size of lines, vias ...
 * syntax: Change(Text|Line|Via|Pin|LinesInBlock|ViasInBlock|PinsInBlock)
 */
void ActionChange(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	PinTypePtr		via,
					pin;
	LineTypePtr		line;
	LayerTypePtr	layer;
	ElementTypePtr	element;
	TextTypePtr		text;
	char			*string;

	if (*Num == 0 || *Num > 2)
		return;

	HideCursor();
	SwitchDrawingWindow(PCB->Zoom, Output.OutputWindow);
	switch(GetFunctionID(*Params))
	{
		case F_VIA:
			if ((via = SearchVia(MyCursor.X, MyCursor.Y)) != NULL)
				if (ChangeViaSize(via, atoi(*(Params+1))))
					SetChangedFlag(True);
			break;

		case F_PIN:
			if (SearchPin(&element, &pin, MyCursor.X, MyCursor.Y))
				if (ChangePinSize(pin, atoi(*(Params+1))))
					SetChangedFlag(True);
			break;

		case F_LINE:
			if (SearchLine(&layer, &line, MyCursor.X, MyCursor.Y))
				if (ChangeLineSize(line, layer, atoi(*(Params+1))))
					SetChangedFlag(True);
			break;

		case F_LINES_IN_BLOCK:
			if (MyCursor.Mode == MODE_BLOCK &&
				MyCursor.MarkedBlock.State == MARKED_STATE_ENDED)
				if (ChangeLineSizeInRectangle( MyCursor.MarkedBlock.X1,
					MyCursor.MarkedBlock.Y1, MyCursor.MarkedBlock.X2,
					MyCursor.MarkedBlock.Y2, atoi(*(Params+1))))
				{
					SetChangedFlag(True);
				}
			break;

		case F_VIAS_IN_BLOCK:
			if (MyCursor.Mode == MODE_BLOCK &&
				MyCursor.MarkedBlock.State == MARKED_STATE_ENDED)
				if (ChangeViaSizeInRectangle( MyCursor.MarkedBlock.X1,
					MyCursor.MarkedBlock.Y1, MyCursor.MarkedBlock.X2,
					MyCursor.MarkedBlock.Y2, atoi(*(Params+1))))
				{
					SetChangedFlag(True);
				}
			break;

		case F_PINS_IN_BLOCK:
			if (MyCursor.Mode == MODE_BLOCK &&
				MyCursor.MarkedBlock.State == MARKED_STATE_ENDED)
				if (ChangePinSizeInRectangle( MyCursor.MarkedBlock.X1,
					MyCursor.MarkedBlock.Y1, MyCursor.MarkedBlock.X2,
					MyCursor.MarkedBlock.Y2, atoi(*(Params+1))))
				{
					SetChangedFlag(True);
				}
			break;

		case F_TEXT:
			if (SearchText(&layer, &text, MyCursor.X, MyCursor.Y) &&
				(string = GetUserInput(Output.Toplevel, False, "Enter text:", text->TextString)) != NULL)
			{
				EraseText(text);
				AddStringToText(PCB, text, string);
				SaveFree(string);
				DrawTextOnLayer(text, layer);
				SetChangedFlag(True);
			}
			break;
	}
	RestoreCursor();
}

/* ---------------------------------------------------------------------------
 * screen refresh
 * syntax: Redraw()
 */
void ActionRedraw(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	if (*Num == 0)
		RedrawOutput();
}

/* ---------------------------------------------------------------------------
 * center display
 * syntax: Center()
 */
void ActionCenter(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	if (*Num == 0)
		CenterDisplay(TO_SCREEN(MyCursor.X), TO_SCREEN(MyCursor.Y));
}

/* ---------------------------------------------------------------------------
 * grabs an element into 'MarkedElement' which may then be positioned
 * if successful, update some other stuff and reposition the loaded element
 * syntax: GrabElement()
 */
void ActionGrabElement(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	ElementTypePtr	element;

	if (*Num != 0 || (element = SearchElement(MyCursor.X, MyCursor.Y)) == NULL)
		return;

		/* copy element data and reposition the upper left corner to (0,0)
		 * element points to MyCursor.MarkedElement.Element after
		 * CopyElement()
		 */
	HideCursor();
	SwitchDrawingWindow(PCB->Zoom, Output.OutputWindow);
	element = CopyElement(NULL, element, &MyCursor.MarkedElement.Element);
	SetElementInfo(element, 1);
	MoveElement(element, -element->Rect.X, -element->Rect.Y);
	MyCursor.MarkedElement.Valid = True;
	SetElementnameField();
	RestoreCursor();
}

/* ---------------------------------------------------------------------------
 * sets the name of elements, vias and pins
 * the returned string might be emtpy, it this case 
 * we free the allocated memory
 * syntax: Name(Via|Pin|Element)
 */
void ActionName(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	PinTypePtr		via,
					pin;
	ElementTypePtr	element;
	char			*name;

	if (*Num != 1)
		return;

	HideCursor();
	SwitchDrawingWindow(PCB->Zoom, Output.OutputWindow);
	switch(GetFunctionID(*Params))
	{
		case F_VIA:
			if ((via = SearchVia(MyCursor.X, MyCursor.Y)) != NULL)
			{
				name = GetUserInput(Output.Toplevel, True,
					"Enter the vias name:", via->Name);
				if (name != NULL)
				{
					RENAME(via->Name, name);
					SetChangedFlag(True);
				}
			}
			break;

		case F_PIN:
			if (SearchPin(&element, &pin, MyCursor.X, MyCursor.Y))
			{
				name = GetUserInput(Output.Toplevel, True,
					"Enter the pins name:", pin->Name);
				if (name != NULL)
				{
					RENAME(pin->Name, name);
					SetChangedFlag(True);
				}
			}
			break;

		case F_ELEMENT:
			if ((element = SearchElement(MyCursor.X, MyCursor.Y)) != NULL)
			{
				name = GetUserInput(Output.Toplevel, True,
					"Enter the elements name:", element->NameOnPCB);
				if (name != NULL)
				{
					EraseElementName(element);
					RENAME(element->NameOnPCB, name);
					DrawElementName(element);
					SetChangedFlag(True);
				}
			}
			break;
	}
	RestoreCursor();
}

/* ---------------------------------------------------------------------------
 * triggered by user input ascii text widget
 * syntax: EndCommand(<string>)
 */
void ActionEndCommand(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	if (*Num == 1)
		EndCommand(!strcmp("OK", *Params));
}

/* ---------------------------------------------------------------------------
 * some 'text processing'
 * syntax: Text(Rotate|Mirror)
 */
void ActionText(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	LayerTypePtr	layer;
	TextTypePtr		text;

	if (*Num == 0 || *Num > 2 ||
		!SearchText(&layer, &text, MyCursor.X, MyCursor.Y))
		return;

	HideCursor();
	SwitchDrawingWindow(PCB->Zoom, Output.OutputWindow);
	switch(GetFunctionID(*Params))
	{
		case F_ROTATE:		/* rotate text */
			EraseText(text);
			RotateText(text, text->X, text->Y, *Num == 1 ? 1 : atoi(*(Params+1)));
			DrawTextOnLayer(text, layer);
			SetChangedFlag(True);
			break;

		case F_MIRROR:		/* mirror text */
			EraseText(text);
			CHANGE_FLAG(MIRRORFLAG, text);
			SetChangedFlag(True);
			DrawTextOnLayer(text, layer);
			break;
	}
	RestoreCursor();
}

/* ---------------------------------------------------------------------------
 * searches connections of the pin or via at the cursor position
 * syntax: Connection(Find|ResetLinesAndRectangles|ResetPinsAndVias|Reset)
 */
void ActionConnection(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	if (*Num != 1)
		return;

	HideCursor();
	SwitchDrawingWindow(PCB->Zoom, Output.OutputWindow);
	switch(GetFunctionID(*Params))
	{
		case F_FIND:					/* find connections */
			if (FindConnection(MyCursor.X, MyCursor.Y))
				SetChangedFlag(True);
			break;

		case F_RESET_FOUND_LINES_AND_RECTANGLES:
			ResetFoundLinesAndRectangles();
			SetChangedFlag(True);
			RedrawOutput();
			break;

		case F_RESET_FOUND_PINS_AND_VIAS:
			ResetFoundPinsAndVias();
			SetChangedFlag(True);
			RedrawOutput();
			break;

		case F_RESET:
			ResetFoundPinsAndVias();
			ResetFoundLinesAndRectangles();
			SetChangedFlag(True);
			RedrawOutput();
			break;
	}
	RestoreCursor();
}

/* ---------------------------------------------------------------------------
 * changes between display modes of Element names
 * syntax: ElementNameMode(DisplayCanonicalName|DisplayNameOnPCB)
 */
void ActionElementNameMode(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	int				id;
	ElementTypePtr	element;
	Cardinal		i;

	if (*Num != 1 || !PCB->ElementOn)
		return;

	HideCursor();
	SwitchDrawingWindow(PCB->Zoom, Output.OutputWindow);
	switch(id = GetFunctionID(*Params))
	{
		case F_DISPLAY_NAME_ON_PCB:
		case F_DISPLAY_CANONICAL_NAME:
			for (i = PCB->ElementN, element = PCB->Element; i; i--, element++)
				EraseElementName(element);
			if (id == F_DISPLAY_NAME_ON_PCB)
				SET_FLAG(CANONICALFLAG, PCB);
			else
				CLEAR_FLAG(CANONICALFLAG, PCB);
			for (i = PCB->ElementN, element = PCB->Element; i; i--, element++)
				DrawElementName(element);
			break;
	}
	RestoreCursor();
}

/* ---------------------------------------------------------------------------
 * uallows the user to enter a commandline
 * syntax: StartCommand()
 */
void ActionStartCommand(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	if (*Num == 0)
	{
		HideCursor();
		SwitchDrawingWindow(PCB->Zoom, Output.OutputWindow);
		ReadAndExecuteCommand();
		RestoreCursor();
	}
}

/* ---------------------------------------------------------------------------
 * performs linestack handling
 * syntax: LineStack(Reset|Enable|Disable|SetOrigin|Paste)
 */
void ActionLineStack(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	if (*Num != 1)
		return;

	HideCursor();
	SwitchDrawingWindow(PCB->Zoom, Output.OutputWindow);
	switch(GetFunctionID(*Params))
	{
		case F_RESET:
			MyCursor.LineStack.On = False;
			if (MyCursor.Mode == MODE_LINESTACK)
				MyCursor.Mode = MODE_NONE;
			MyCursor.LineStack.LineN = 0;
			MyCursor.LineStack.OriginX = 0;
			MyCursor.LineStack.OriginY = 0;
			SetStatusLine();
			break;

		case F_ENABLE:
			MyCursor.LineStack.On = True;
			MyCursor.LineStack.OriginX = MyCursor.X;
			MyCursor.LineStack.OriginY = MyCursor.Y;
			SetStatusLine();
			break;

		case F_DISABLE:
			MyCursor.LineStack.On = False;
			SetStatusLine();
			break;

		case F_SETORIGIN:
			MyCursor.LineStack.OriginX = MyCursor.X;
			MyCursor.LineStack.OriginY = MyCursor.Y;
			break;

		case F_PASTE:
			CreateObject(F_PASTE);
			break;
	}
	RestoreCursor();
}

/* ---------------------------------------------------------------------------
 * opens a new window to display the pinout of the element at
 * the current cursor location
 * syntax: DisplayPinout()
 */
void ActionDisplayPinout(Widget W, XEvent *Event, String *Params, Cardinal *Num)
{
	ElementTypePtr	element;

	if (*Num != 0 || (element = SearchElement(MyCursor.X, MyCursor.Y)) == NULL)
		return;

	CreatePinoutWindow(Output.Toplevel, element);
}

