/*
** Double buffering for KBackup
** 
** (c) 1996 Mario Weilguni <e8732250@stud1.tuwien.ac.at>
**
*/

#include <stdlib.h>
#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include <getopt.h>

#define DEFAULT_BUFSIZE      4096
#define DEFAULT_BUFCOUNT     32
#define ERR_OPT 1
#define ERR_MEM 2
#define ERR_WRITE 3
#define ERR_READ 4


typedef struct {
    char *data;
    unsigned buflen;
    unsigned nbytes;
} BUFFER;


typedef struct {
    unsigned buf_count;
    unsigned buf_size;
    int verbose;
    int	sync;
} OPTIONS;

OPTIONS opt = {DEFAULT_BUFCOUNT, DEFAULT_BUFSIZE, 0, 0};
extern char *optarg;
extern int optind, opterr, optopt;


BUFFER *alloc_buffer(int size) {
    BUFFER *p;

    p = malloc(sizeof(BUFFER));
    if(p == NULL) 
	return NULL;
    else 
    {
	p->data = malloc(size);
	if(p->data == NULL) 
	{
	    free(p);
	    return NULL;
	} 
	else 
	{
	    memset(p->data, 0, size);
	    p->buflen = size;
	    return p;
	}
    }
}


void free_buffer(BUFFER *p) {
    if(p) 
    {
	if(p->data)
	    free(p->data);
	free(p);
    }
}


void showhelp(FILE *stream) {
    fprintf(stream, "Usage:\n\tdblbuf [-v] -s [-b buffersize] [-n buffers]\n");
}


void check_args(int argc, char **argv) {
    char *eptr;

    while(1)
    {
	switch(getopt(argc, argv, "hsvb:n:"))
	{
	case -1:
	    return;
	    break; /* for Lint */
	case 'v':
	    opt.verbose = 1;
	    break;
	case 'b':
	    opt.buf_size = strtol(optarg, &eptr, 10);
	    if(*eptr != '\0') 
	    {
		fprintf(stderr, "%s: invalid argument '%s'\n", argv[0], optarg);
		showhelp(stderr);
		exit(ERR_OPT);
	    }
	    break;
	case 'n':
	    opt.buf_count = strtol(optarg, &eptr, 10);
	    if(*eptr != '\0') 
	    {
		fprintf(stderr, "%s: invalid argument '%s'\n", argv[0], optarg);
		showhelp(stderr);
		exit(ERR_OPT);
	    }
	    break;
	case 's':
	    opt.sync = 1;
	    break;
	case 'h':
	    if(argc != 2) 
	    {
		fprintf(stderr, "%s: no argument allowed if -h is specified\n", argv[0]);
		showhelp(stderr);
		exit(ERR_OPT);
	    } 
	    else
		showhelp(stdout);
	    exit(0);
	case '?':
	    exit(ERR_OPT);
	    break;
	}
    }
}


int main(int argc, char **argv) {
    fd_set rset, wset;
    BUFFER **buf;
    int i, rptr, wptr, eof_i = 0;

    /* check the command line */
    check_args(argc, argv);

    /* allocate the buffer array */
    buf = malloc(sizeof(BUFFER *) * opt.buf_count);
    if(buf == NULL) 
    {
	fprintf(stderr, "%s: cannot allocate buffer array\n", argv[0]);
	return ERR_MEM;
    }
  
    /* allocate the buffers */
    for(i = 0; i < opt.buf_count; i++) 
    {
	buf[i] = alloc_buffer(opt.buf_size);
	if(buf[i] == NULL) 
	{
	    fprintf(stderr, "%s: cannot allocate buffers\n", argv[0]);
	    return ERR_MEM;
	}
    }

    for(rptr = 0, wptr = 0;;) 
    {
	/* can I read in new buffers? */
	FD_ZERO(&rset);
	if( !eof_i && (rptr - wptr < opt.buf_count))
	{
	    FD_SET(0, &rset);
	}

	/* can I write out buffers? */
	FD_ZERO(&wset);
	if(rptr != wptr) 
	{
	    FD_SET(1, &wset);
	}

	if(select(2, &rset, &wset, (fd_set *)NULL, (struct timeval *)NULL) != -1)

	{

	    /* read in a buffer */
	    if(FD_ISSET(0, &rset)) 
	    {
		int rbuf = rptr % opt.buf_count;
		if(opt.verbose) 
		    fprintf(stderr, "\treading buffer %d\n", rptr);
		buf[rbuf]->nbytes = fread(buf[rbuf]->data, 1, opt.buf_size, stdin);
		if(buf[rbuf]->nbytes != opt.buf_size)
		{
		    memset(buf[rbuf]->data+buf[rbuf]->nbytes,
			   0, opt.buf_size-buf[rbuf]->nbytes);
		    buf[rbuf]->buflen = opt.buf_size;
		    eof_i = 1;
		}
		rptr++;
	    }

	    /* write a buffer */
	    if(FD_ISSET(1, &wset)) 
	    {
		int ret;
		int wbuf = wptr % opt.buf_count;
		if(opt.verbose) 
		    fprintf(stderr, "\twriting buffer %d\n", wptr);

		ret = fwrite(buf[wbuf]->data, 1, buf[wbuf]->nbytes, stdout);
		if(buf[wbuf]->nbytes != opt.buf_size) 
		{
		    fflush(stdout);
		    return 0;
		}
		wptr++;
	    }	  
	}
    }
}
